﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Config.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/svc/svc_Types.common.h>

#include "osdbg_Types.h"

namespace nn { namespace osdbg {
namespace detail {

//--------------------------------------------------------------------------
//  ILP32 環境下と同じ内部イメージを持つ os::ThreadType 相当の構造体。
//  サイズやアライメントが、この構造体利用者のビルド環境によらず一定となる。
//
struct  ThreadTypeIlp32
{
    // メンバ変数
    AlignedStorageIlp32<0,2,NN_ALIGNOF(ptr32_t)>    _allThreadsListNode;
    AlignedStorageIlp32<0,2,NN_ALIGNOF(ptr32_t)>    _multiWaitObjectList;

    NN_PADDING4;        // NN_OS_DETAIL_PADDING_POINTER on NX32
    NN_PADDING4;        // NN_OS_DETAIL_PADDING_POINTER on NX32
    NN_PADDING4;        // NN_OS_DETAIL_PADDING_POINTER on NX32
    NN_PADDING4;        // NN_OS_DETAIL_PADDING_POINTER on NX32

    uint8_t     _state;
    bool        _stackIsAliased;
    bool        _autoRegistered;
    uint8_t     _padding0;

    int32_t     _basePriority;
    ptr32_t     _originalStack;
    ptr32_t     _stack;
    size32_t    _stackSize;
    ptr32_t     _argument;
    ptr32_t     _threadFunction;
    ptr32_t     _currentFiber;
    ptr32_t     _initialFiber;
    uint32_t    _lockHistory;
    uintptr32_t _tlsValueArray[nn::os::TlsSlotCountMax + nn::os::SdkTlsSlotCountMax];

    char        _threadNameBuffer[nn::os::ThreadNameLengthMax];
    ptr32_t     _namePointer;

    AlignedStorageIlp32<4,0,NN_ALIGNOF(Bit32)>      _csThread;
    AlignedStorageIlp32<4,0,NN_ALIGNOF(uint32_t)>   _cvThread;
    AlignedStorageIlp32<4,0,NN_ALIGNOF(uint32_t)>   _handle;

    // 構造体のサイズ変化防止のため
    char padding[8];
};

#if defined(NN_BUILD_CONFIG_ABI_ILP32)
NN_OSDBG_STATIC_ASSERT_SIZEOF(os::ThreadType, ThreadTypeIlp32);
NN_OSDBG_STATIC_ASSERT_ALIGNOF(os::ThreadType, ThreadTypeIlp32);

NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _allThreadsListNode);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _multiWaitObjectList);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _state);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _stackIsAliased);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _autoRegistered);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _basePriority);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _originalStack);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _stack);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _stackSize);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _argument);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _threadFunction);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _initialFiber);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _currentFiber);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _lockHistory);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _tlsValueArray);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _threadNameBuffer);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _namePointer);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _csThread);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _cvThread);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeIlp32, _handle);
#endif

//--------------------------------------------------------------------------
//  LP64 環境下と同じ内部イメージを持つ os::ThreadType 相当の構造体。
//  サイズやアライメントが、この構造体利用者のビルド環境によらず一定となる。
//
struct  ThreadTypeLp64
{
    // メンバ変数
    AlignedStorageLp64<0,2,NN_ALIGNOF(ptr64_t)>     _allThreadsListNode;
    AlignedStorageLp64<0,2,NN_ALIGNOF(ptr64_t)>     _multiWaitObjectList;

    NN_PADDING8;        // NN_OS_DETAIL_PADDING_POINTER on NX64
    NN_PADDING8;        // NN_OS_DETAIL_PADDING_POINTER on NX64
    NN_PADDING8;        // NN_OS_DETAIL_PADDING_POINTER on NX64
    NN_PADDING8;        // NN_OS_DETAIL_PADDING_POINTER on NX64

    uint8_t     _state;
    bool        _stackIsAliased;
    bool        _autoRegistered;
    uint8_t     _padding0;
    int32_t     _basePriority;

    ptr64_t     _originalStack;
    ptr64_t     _stack;
    size64_t    _stackSize;
    ptr64_t     _argument;
    ptr64_t     _threadFunction;
    ptr64_t     _currentFiber;
    ptr64_t     _initialFiber;

    uint32_t    _lockHistory;
    uint32_t    _padding1;

    uintptr64_t _tlsValueArray[nn::os::TlsSlotCountMax + nn::os::SdkTlsSlotCountMax];

    char        _threadNameBuffer[nn::os::ThreadNameLengthMax];

    ptr64_t     _namePointer;

    AlignedStorageLp64<4,0,NN_ALIGNOF(Bit32)>      _csThread;
    AlignedStorageLp64<4,0,NN_ALIGNOF(uint32_t)>   _cvThread;

    AlignedStorageLp64<4,0,NN_ALIGNOF(uint32_t)>   _handle;
    uint32_t    _padding2;
};

#if defined(NN_BUILD_CONFIG_ABI_LP64)
NN_OSDBG_STATIC_ASSERT_SIZEOF(os::ThreadType, ThreadTypeLp64);
NN_OSDBG_STATIC_ASSERT_ALIGNOF(os::ThreadType, ThreadTypeLp64);

NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _allThreadsListNode);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _multiWaitObjectList);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _state);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _stackIsAliased);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _autoRegistered);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _basePriority);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _originalStack);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _stack);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _stackSize);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _argument);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _threadFunction);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _initialFiber);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _currentFiber);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _lockHistory);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _tlsValueArray);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _threadNameBuffer);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _namePointer);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _csThread);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _cvThread);
NN_OSDBG_STATIC_ASSERT_OFFSETOF(os::ThreadType, ThreadTypeLp64, _handle);
#endif

//--------------------------------------------------------------------------
//  ILP32/LP64 ABI 共用の os::ThreadType 相当の構造体。
//
union   ThreadTypeCommon
{
    ThreadTypeIlp32 ilp32;
    ThreadTypeLp64  lp64;
};


//--------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::osdbg

