﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_SdkMutex.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"

#include <cstring>

namespace nn { namespace os {

//-----------------------------------------------------------------------------
//  os::SdkMutex の実装
//
//  Win 用の detail::InternalCriticalSection ではロックオーナーを
//  識別する手段がないため、SdkMutex 側で保持しておく。
//  NX 実機用の detail::InternalCriticalSection では内部で管理している。
//-----------------------------------------------------------------------------

bool SdkMutexType::IsLockedByCurrentThread() const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return Get(this->_mutex).IsLockedByCurrentThread();
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    return this->_ownerThread == detail::GetCurrentThread();
#endif
}

void SdkMutexType::Initialize() NN_NOEXCEPT
{
    Get(this->_mutex).Initialize();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    this->_ownerThread = nullptr;
#endif
}

void SdkMutexType::Lock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS( !this->IsLockedByCurrentThread() );
    Get(this->_mutex).Enter();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    this->_ownerThread = detail::GetCurrentThread();
#endif
}

bool SdkMutexType::TryLock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS( !this->IsLockedByCurrentThread() );
    auto ret = Get(this->_mutex).TryEnter();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    if ( ret )
    {
        this->_ownerThread = detail::GetCurrentThread();
    }
#endif
    return ret;
}

void SdkMutexType::Unlock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS( this->IsLockedByCurrentThread() );
#if defined(NN_BUILD_CONFIG_OS_WIN)
    this->_ownerThread = nullptr;
#endif
    Get(this->_mutex).Leave();
}

//-----------------------------------------------------------------------------
//  os::SdkRecursiveMutex の実装
//-----------------------------------------------------------------------------

bool SdkRecursiveMutexType::IsLockedByCurrentThread() const NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return Get(this->_mutex).IsLockedByCurrentThread();
#elif defined(NN_BUILD_CONFIG_OS_WIN)
    return this->_ownerThread == detail::GetCurrentThread();
#endif
}

void SdkRecursiveMutexType::Initialize() NN_NOEXCEPT
{
    Get(this->_mutex).Initialize();
    this->_recursiveCount = 0;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    this->_ownerThread = nullptr;
#endif
}

void SdkRecursiveMutexType::Lock() NN_NOEXCEPT
{
    if ( !this->IsLockedByCurrentThread() )
    {
        Get(this->_mutex).Enter();
#if defined(NN_BUILD_CONFIG_OS_WIN)
        this->_ownerThread = detail::GetCurrentThread();
#endif
    }
    NN_ABORT_UNLESS( ++this->_recursiveCount );
}

bool SdkRecursiveMutexType::TryLock() NN_NOEXCEPT
{
    if ( !this->IsLockedByCurrentThread() )
    {
        if ( !Get(this->_mutex).TryEnter() )
        {
            return false;
        }
#if defined(NN_BUILD_CONFIG_OS_WIN)
        this->_ownerThread = detail::GetCurrentThread();
#endif
    }
    NN_ABORT_UNLESS( ++this->_recursiveCount );
    return true;
}

void SdkRecursiveMutexType::Unlock() NN_NOEXCEPT
{
    NN_ABORT_UNLESS( this->IsLockedByCurrentThread() );
    if ( --this->_recursiveCount == 0 )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        this->_ownerThread = nullptr;
#endif
        Get(this->_mutex).Leave();
    }
}

}}  // namespace nn::os

