﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_MessageQueueTypes.h>
#include <nn/os/os_MessageQueueApi.h>
#include <nn/os/os_MultipleWaitApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfMessageQueue.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//--------------------------------------------------------------------------
//  C++ 関数の定義
//--------------------------------------------------------------------------

namespace nn { namespace os {

//--------------------------------------------------------------------------
//
//  buffer -> +-----+-----+-----+-----+-----+-----+-----+-----+
//            |     |data1|data2|data3|     |     |     |     |
//            +-----+-----+-----+-----+-----+-----+-----+-----+
//                   ^offset           ^(offset + count)
//
//  - offset は次に取出すべき先頭データを指すインデックス
//  - count  はキューに入っているデータの個数
//
//--------------------------------------------------------------------------

namespace {

    inline bool IsMessageQueueFull(const MessageQueueType* messageQueue) NN_NOEXCEPT
    {
        return messageQueue->_count >= messageQueue->_maxCount;
    }

    inline bool IsMessageQueueEmpty(const MessageQueueType* messageQueue) NN_NOEXCEPT
    {
        return messageQueue->_count == 0;
    }

    void EnqueueUnsafe(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
    {
        auto maxCount   = messageQueue->_maxCount;
        auto count      = messageQueue->_count;

        NN_SDK_ASSERT( count < maxCount, NN_TEXT_OS("内部エラー：メッセージキューのデータ数が最大データ数を超えました（count=%d, max=%d）。"), count, maxCount );

        // キューの末尾位置を計算
        auto tail = messageQueue->_offset + count;
        if (tail >= maxCount)
        {
            tail -= maxCount;
        }
        NN_SDK_ASSERT( (tail >= 0) && (tail < maxCount), NN_TEXT_OS("内部エラー：メッセージキューの末尾ポインタが異常です（tail=%d, max=%d）。"), tail, maxCount );

        // データをキューの末尾に挿入
        messageQueue->_buffer[ tail ] = data;
        ++count;

        messageQueue->_count  = count;
    }

    void JamUnsafe(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
    {
        auto maxCount    = messageQueue->_maxCount;
        auto count       = messageQueue->_count;

        NN_SDK_ASSERT( count < maxCount, NN_TEXT_OS("内部エラー：メッセージキューのデータ数が最大データ数を超えました（count=%d, max=%d）。"), count, maxCount );

        // キューの先頭位置を計算
        auto offset = messageQueue->_offset - 1;
        if (offset < 0)
        {
            offset += maxCount;
        }
        NN_SDK_ASSERT( (offset >= 0) && (offset < maxCount), NN_TEXT_OS("内部エラー：メッセージキューの先頭ポインタが異常です（offset=%d, max=%d）。"), offset, maxCount );

        // データをキューの末尾に挿入
        messageQueue->_buffer[ offset ] = data;
        ++count;

        messageQueue->_offset = offset;
        messageQueue->_count  = count;
    }

    uintptr_t DequeueUnsafe(MessageQueueType* messageQueue) NN_NOEXCEPT
    {
        auto maxCount   = messageQueue->_maxCount;
        auto count      = messageQueue->_count;
        auto offset     = messageQueue->_offset;

        NN_SDK_ASSERT( count > 0, NN_TEXT_OS("内部エラー：空のメッセージキューに対して %s() が呼ばれました。"), NN_CURRENT_FUNCTION_NAME );
        NN_SDK_ASSERT( (offset >= 0) && (offset <  maxCount), NN_TEXT_OS("内部エラー：メッセージキューの先頭ポインタが異常です（offset=%d, max=%d）。"), offset, maxCount );

        // キューの先頭からデータを取得
        auto    data = messageQueue->_buffer[ offset ];

        // キューの先頭位置を計算
        ++offset;
        if (offset >= maxCount)
        {
            offset -= maxCount;
        }
        --count;

        messageQueue->_offset = offset;
        messageQueue->_count  = count;
        return data;
    }

    inline uintptr_t PeekUnsafe(const MessageQueueType* messageQueue) NN_NOEXCEPT
    {
        auto count  = messageQueue->_count;
        auto offset = messageQueue->_offset;

        NN_SDK_ASSERT( count > 0, NN_TEXT_OS("内部エラー：空のメッセージキューに対して %s() が呼ばれました。"), NN_CURRENT_FUNCTION_NAME );
        NN_UNUSED(count);   // Warning C4189 対策

        // キューの先頭にあるデータを参照
        auto   data = messageQueue->_buffer[ offset ];
        return data;
    }
}


//---------------------------------------------------------------------------
//  MessageQueueType オブジェクトのイニシャライズ
void InitializeMessageQueue(MessageQueueType* messageQueue, uintptr_t* buffer, size_t count) NN_NOEXCEPT
{
    // 引数エラーチェック
    NN_SDK_REQUIRES( buffer != NULL, NN_TEXT_OS("nn::os::InitializeMessageQueue(): buffer に NULL が指定されました。") );
    NN_SDK_REQUIRES( count >= 1, NN_TEXT_OS("nn::os::InitializeMessageQueue(): count が不正です（count = %d）。"), count );

    // 排他リソースを初期化
    new( &messageQueue->_csQueue )       detail::InternalCriticalSection;
    new( &messageQueue->_cvNotFull )     detail::InternalConditionVariable;
    new( &messageQueue->_cvNotEmpty )    detail::InternalConditionVariable;

    // IntrusiveList のコンストラクタを呼ぶ
    new( &messageQueue->_multiWaitObjectListNotFull  ) detail::MultiWaitObjectList;
    new( &messageQueue->_multiWaitObjectListNotEmpty ) detail::MultiWaitObjectList;

    // メンバの初期化
    messageQueue->_buffer     = buffer;
    messageQueue->_maxCount   = static_cast<int>(count);
    messageQueue->_count      = 0;
    messageQueue->_offset     = 0;

    // MessageQueueType を Initialized 状態へ（最後に行なう）
    messageQueue->_state      = MessageQueueType::State_Initialized;
}


//---------------------------------------------------------------------------
//  MessageQueueType オブジェクトのファイナライズ
void FinalizeMessageQueue(MessageQueueType* messageQueue) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    // MessageQueueType オブジェクトに待ちスレッドがあれば異常停止
    NN_SDK_REQUIRES( (Get(messageQueue->_multiWaitObjectListNotFull).IsEmpty()) &&
                     (Get(messageQueue->_multiWaitObjectListNotEmpty).IsEmpty()),
        NN_TEXT_OS("nn::os::FinalizeMessageQueue(): 指定されたメッセージキューに待機スレッドが残っています。") );

    // MessageQueueType を NotInitialized 状態へ変更（最初に行なう）
    messageQueue->_state  = MessageQueueType::State_NotInitialized;

    // IntrusiveList のデストラクタを呼ぶ
    Get(messageQueue->_multiWaitObjectListNotFull).~MultiWaitObjectList();
    Get(messageQueue->_multiWaitObjectListNotEmpty).~MultiWaitObjectList();

    // 排他リソースを破棄
    Get(messageQueue->_csQueue).~InternalCriticalSection();
    Get(messageQueue->_cvNotFull).~InternalConditionVariable();
    Get(messageQueue->_cvNotEmpty).~InternalConditionVariable();
}


//--------------------------------------------------------------------------
//  MessageQueueType の最後尾にデータを１つ送信
void SendMessageQueue(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::SendMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら、NotFUll 条件変数待ちへ
        while ( IsMessageQueueFull( messageQueue ) )
        {
            Get(messageQueue->_cvNotFull).Wait( &cs );
        }

        // キューの末尾にデータを挿入
        EnqueueUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//--------------------------------------------------------------------------
//  MessageQueueType の最後尾にデータを１つ送信（ポーリング）
bool TrySendMessageQueue(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TrySendMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら false を返す
        if ( IsMessageQueueFull( messageQueue ) )
        {
            return false;
        }

        // キューの末尾にデータを挿入
        EnqueueUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の最後尾にデータを１つ送信（タイムアウト付き）
bool TimedSendMessageQueue(MessageQueueType* messageQueue, uintptr_t data, TimeSpan timeout) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TimedSendMessageQueue(): 指定されたメッセージキューが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedSendMessageQueue(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら NotFUll 条件変数待ちへ
        while ( IsMessageQueueFull( messageQueue ) )
        {
            // タイムアウト確認
            if ( tmout.IsTimedout() )
            {
                return false;
            }

            Get(messageQueue->_cvNotFull).TimedWait( &cs, tmout );
        }

        // キューの末尾にデータを挿入
        EnqueueUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭にデータを１つ送信
void JamMessageQueue(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::JamMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら NotFUll 条件変数待ちへ
        while ( IsMessageQueueFull( messageQueue ) )
        {
            Get(messageQueue->_cvNotFull).Wait( &cs );
        }

        // キューの先頭にデータを挿入
        JamUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭にデータを１つ送信（ポーリング）
bool TryJamMessageQueue(MessageQueueType* messageQueue, uintptr_t data) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TryJamMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら false を返す
        if ( IsMessageQueueFull( messageQueue ) )
        {
            return false;
        }

        // キューの先頭にデータを挿入
        JamUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭にデータを１つ送信（タイムアウト付き）
bool TimedJamMessageQueue(MessageQueueType* messageQueue, uintptr_t data, TimeSpan timeout) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TimedJamMessageQueue(): 指定されたメッセージキューが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedJamMessageQueue(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが満杯なら NotFUll 条件変数待ちへ
        while ( IsMessageQueueFull( messageQueue ) )
        {
            // タイムアウト確認
            if ( tmout.IsTimedout() )
            {
                return false;
            }

            Get(messageQueue->_cvNotFull).TimedWait( &cs, tmout );
        }

        // キューの先頭にデータを挿入
        JamUnsafe( messageQueue, data );

        // 受信待ちスレッドに NotEmpty 条件変数を通知
        Get(messageQueue->_cvNotEmpty).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotEmpty).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭からデータを１つ受信
void ReceiveMessageQueue(uintptr_t* outData, MessageQueueType* messageQueue) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::ReceiveMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら NotEmpty 条件変数待ちへ
        while ( IsMessageQueueEmpty( messageQueue ) )
        {
            Get(messageQueue->_cvNotEmpty).Wait( &cs );
        }

        // キューの先頭にあるデータを取得
        *outData = DequeueUnsafe( messageQueue );

        // 送信待ちスレッドに NotFull 条件変数を通知
        Get(messageQueue->_cvNotFull).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotFull).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭からデータを１つ受信（ポーリング）
bool TryReceiveMessageQueue(uintptr_t* outData, MessageQueueType* messageQueue) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TryReceiveMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら false を返す
        if ( IsMessageQueueEmpty( messageQueue ) )
        {
            return false;
        }

        // キューの先頭にあるデータを取得
        *outData = DequeueUnsafe( messageQueue );

        // 送信待ちスレッドに NotFull 条件変数を通知
        Get(messageQueue->_cvNotFull).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotFull).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭からデータを１つ受信（タイムアウト付き）
bool TimedReceiveMessageQueue(uintptr_t* outData, MessageQueueType* messageQueue, TimeSpan timeout) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TimedReceiveMessageQueue(): 指定されたメッセージキューが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedReceiveMessageQueue(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら NotEmpty 条件変数待ちへ
        while ( IsMessageQueueEmpty( messageQueue ) )
        {
            // タイムアウト確認
            if ( tmout.IsTimedout() )
            {
                return false;
            }

            Get(messageQueue->_cvNotEmpty).TimedWait( &cs, tmout );
        }

        // キューの先頭にあるデータを取得
        *outData = DequeueUnsafe( messageQueue );

        // 送信待ちスレッドに NotFull 条件変数を通知
        Get(messageQueue->_cvNotFull).Broadcast();

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(messageQueue->_multiWaitObjectListNotFull).WakeupAllMultiWaitThreadsUnsafe();
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭データを参照
void PeekMessageQueue(uintptr_t* outData, const MessageQueueType* messageQueue) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::PeekMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら NotEmpty 条件変数待ちへ
        while ( IsMessageQueueEmpty( messageQueue ) )
        {
            Get(messageQueue->_cvNotEmpty).Wait( &cs );
        }

        // キューの先頭にあるデータを取得
        *outData = PeekUnsafe( messageQueue );
    }
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭データを参照（ポーリング）
bool TryPeekMessageQueue(uintptr_t* outData, const MessageQueueType* messageQueue) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TryPeekMessageQueue(): 指定されたメッセージキューが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら false を返す
        if ( IsMessageQueueEmpty( messageQueue ) )
        {
            return false;
        }

        // キューの先頭にあるデータを取得
        *outData = PeekUnsafe( messageQueue );
    }

    return true;
}


//--------------------------------------------------------------------------
//  MessageQueueType の先頭データを参照（タイムアウト付き）
bool TimedPeekMessageQueue(uintptr_t* outData, const MessageQueueType* messageQueue, TimeSpan timeout) NN_NOEXCEPT
{
    // MessageQueueType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::TimedPeekMessageQueue(): 指定されたメッセージキューが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedPeekMessageQueue(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(messageQueue->_csQueue);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // キューが空なら NotEmpty 条件変数待ちへ
        while ( IsMessageQueueEmpty( messageQueue ) )
        {
            // タイムアウト確認
            if ( tmout.IsTimedout() )
            {
                return false;
            }

            Get(messageQueue->_cvNotEmpty).TimedWait( &cs, tmout );
        }

        // キューの先頭にあるデータを取得
        *outData = PeekUnsafe( messageQueue );
    }

    // データ数は減らないので通知せずにリターン
    return true;
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（MessageQueueType を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType*  pHolder, MessageQueueType*     messageQueue, MessageQueueWaitType  waitType) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( messageQueue->_state == MessageQueueType::State_Initialized, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): 指定されたメッセージキューが初期化されていません。") );

    // 初期化処理（コンストラクタを呼び出す）
    switch ( waitType )
    {
    case MessageQueueWaitType_WaitForNotFull:
            // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
            // アライメントチェックのため、Get() を使って実装しておく。
            new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfMessageQueueNotFull( messageQueue );
            break;

    case MessageQueueWaitType_WaitForNotEmpty:
            // Get() を付けている理由は上記同様
            new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfMessageQueueNotEmpty( messageQueue );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): waitType=%d が不正です。"), waitType);
            break;
    }

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

