﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/os/os_EventTypes.h>
#include <nn/os/os_EventApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_TimeoutHelper.h"
#include "detail/os_MultipleWaitHelper.h"
#include "detail/os_MultipleWaitHolderOfEvent.h"
#include "detail/os_MultipleWaitHolderImpl.h"


//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

namespace {

inline uint64_t GetBroadcastCounterUnsafe(EventType* p) NN_NOEXCEPT
{
    uint64_t upper = p->_broadcastCounterUpper;
    return (upper << (sizeof(p->_broadcastCounterLower) * 8)) | p->_broadcastCounterLower;
}

inline void IncrementBroadcastCounterUnsafe(EventType* p) NN_NOEXCEPT
{
    if (0 == (++p->_broadcastCounterLower))
    {
        ++p->_broadcastCounterUpper;
    }
}

}   // namespace

//---------------------------------------------------------------------------
//  EventType オブジェクトのイニシャライズ
void InitializeEvent(EventType* event, bool initiallySignaled, EventClearMode clearMode) NN_NOEXCEPT
{
    // 排他リソースを初期化
    new( &event->_csEvent )      detail::InternalCriticalSection;
    new( &event->_cvSignaled )   detail::InternalConditionVariable;

    // IntrusiveList のコンストラクタを呼ぶ
    new( &event->_multiWaitObjectList ) detail::MultiWaitObjectList;

    // メンバの初期化
    event->_signalState          = initiallySignaled;
    event->_initiallySignaled    = initiallySignaled;
    event->_clearMode            = static_cast<uint8_t>( clearMode );
    event->_broadcastCounterLower = 0;
    event->_broadcastCounterUpper = 0;

    // イベントを Initialized 状態へ（最後に行なう）
    event->_state                = EventType::State_Initialized;
}


//---------------------------------------------------------------------------
//  EventType オブジェクトのファイナライズ
void FinalizeEvent(EventType* event) NN_NOEXCEPT
{
    // EventType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeEvent(): 指定されたイベントが初期化されていません。") );

    // イベントを NotInitialized 状態へ（最初に行なう）
    event->_state  = EventType::State_NotInitialized;

    // IntrusiveList のデストラクタを呼ぶ
    Get(event->_multiWaitObjectList).~MultiWaitObjectList();

    // 排他リソースを破棄
    Get(event->_csEvent).~InternalCriticalSection();
    Get(event->_cvSignaled).~InternalConditionVariable();
}


//---------------------------------------------------------------------------
//  EventType をシグナル状態へ
void SignalEvent(EventType* event) NN_NOEXCEPT
{
    // EventType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::SignalEvent(): 指定されたイベントが初期化されていません。") );


    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // 既にシグナル状態であれば何もしない
        if ( event->_signalState == true )
        {
            return;
        }

        // シグナル状態にセット
        event->_signalState = true;

        if (event->_clearMode == EventClearMode_ManualClear)
        {
            IncrementBroadcastCounterUnsafe(event);
            Get(event->_cvSignaled).Broadcast();
        }
        else /* AutoClear */
        {
            Get(event->_cvSignaled).Signal();
        }

        // 多重待ちスレッドを起床（ロック取得状態で呼出す）
        Get(event->_multiWaitObjectList).WakeupAllMultiWaitThreadsUnsafe();
    }
}


//---------------------------------------------------------------------------
//  EventType をクリア状態へ
void ClearEvent(EventType* event) NN_NOEXCEPT
{
    // EventType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::ClearEvent(): 指定されたイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        // クリアモードに関係なく非シグナル化
        event->_signalState = false;
    }
}


//---------------------------------------------------------------------------
//  EventType がシグナル状態になるまで待機
void WaitEvent(EventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::WaitEvent(): 指定されたイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        auto currentCounter = GetBroadcastCounterUnsafe(event);
        while (event->_signalState == false)
        {
            // ManualClear の場合、カウンタが進んでいたら起床する
            // AutoClear   の場合、カウンタが進むことはない
            if (currentCounter != GetBroadcastCounterUnsafe(event))
            {
                break;
            }
            Get(event->_cvSignaled).Wait( &cs );
        }

        // AutoClear の場合だけ自動クリアする
        if (event->_clearMode == EventClearMode_AutoClear)
        {
            event->_signalState = false;
        }
    }
}


//---------------------------------------------------------------------------
//  EventType がシグナル状態になるまで待機（ポーリング）
bool TryWaitEvent(EventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::TryWaitEvent(): 指定されたイベントが初期化されていません。") );

    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(event->_csEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        auto prevState = event->_signalState;

        // 自動クリアモードなら非シグナル状態へ
        if (event->_clearMode == EventClearMode_AutoClear)
        {
            event->_signalState = false;
        }

        return prevState;
    }
}


//---------------------------------------------------------------------------
//  EventType がシグナル状態になるまで待機（タイムアウト付き）
bool TimedWaitEvent(EventType* event, TimeSpan timeout) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::TimedWaitEvent(): 指定されたイベントが初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("nn::os::TimedWaitEvent(): timeout 値が不正です。") );

    {   // クリティカルセクション
        detail::TimeoutHelper               tmout( timeout );
        detail::InternalCriticalSection&    cs = Get(event->_csEvent);
        std::lock_guard<detail::InternalCriticalSection>    lock( cs );

        auto currentCounter = GetBroadcastCounterUnsafe(event);
        while (event->_signalState == false)
        {
            // ManualClear の場合、カウンタが進んでいたら起床する
            // AutoClear   の場合、カウンタが進むことはない
            if (currentCounter != GetBroadcastCounterUnsafe(event))
            {
                break;
            }
            auto ret = Get(event->_cvSignaled).TimedWait( &cs, tmout );
            if (ret == ConditionVariableStatus_Timeout)
            {
                return false;
            }
        }

        // AutoClear の場合だけ自動クリアする
        if (event->_clearMode == EventClearMode_AutoClear)
        {
            event->_signalState = false;
        }
    }

    return true;
}


//---------------------------------------------------------------------------
//  MultiWaitHolderType の初期化（Event を関連付ける）
void InitializeMultiWaitHolder(MultiWaitHolderType* pHolder, EventType* event) NN_NOEXCEPT
{
    // 事前条件
    NN_SDK_REQUIRES( event->_state == EventType::State_Initialized, NN_TEXT_OS("nn::os::InitializeMultiWaitHolder(): 指定されたイベントが初期化されていません。") );

    // 初期化処理（コンストラクタを呼び出す）
    // Get() がなくてもポインタは得られるが、TypedStorage のサイズと
    // アライメントチェックのため、Get() を使って実装しておく。
    new( &Get(pHolder->_holderImpl) ) detail::MultiWaitHolderOfEvent( event );

    //  構造体メンバの初期化
    pHolder->userData = 0;
}


}} // namespace nn::os

