﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstddef>

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Allocator.h>
#include <nn/os/os_SdkMemoryAllocatorForThreadLocal.h>
#include <nn/diag/text/diag_SdkTextOs.h>

//-----------------------------------------------------------------------------

namespace {

    nn::AlignedAllocateFunction g_AlignedAllocatorForThreadLocal = NULL;
    nn::DeallocateFunction      g_DeallocatorForThreadLocal      = NULL;

    // TORIAEZU:
    //  各プログラムがロード可能な最大モジュール数
    //  これには、静的モジュール と 動的モジュール の総数が含まれる。
    //  最終的には nn::ro ライブラリで定義されるモジュール最大数に連動すべき。
    //  もしくは、SetMemoryAllocatorForThreadLocal() の第３引数か何かで
    //  最大数を指定できるようにすべきかもしれない。
    const int   ModuleCountMax = 128;

}   // anonymous


namespace nn { namespace os {

//-----------------------------------------------------------------------------
//  SetMemoryAllocatorForThreadLocal() 関数の定義
//
void SetMemoryAllocatorForThreadLocal(nn::AlignedAllocateFunction allocator, nn::DeallocateFunction deallocator) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    NN_UNUSED(allocator);
    NN_UNUSED(deallocator);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS(allocator   != NULL && g_AlignedAllocatorForThreadLocal == NULL, NN_TEXT_OS("nn::os::SetMemoryAllocatorForThreadLocal(): メモリアロケータに NULL は指定できません。もしくは、すでに登録済みです。"));
    NN_ABORT_UNLESS(deallocator != NULL && g_DeallocatorForThreadLocal      == NULL, NN_TEXT_OS("nn::os::SetMemoryAllocatorForThreadLocal(): メモリデアロケータに NULL は指定できません。もしくは、すでに登録済みです。"));

    g_AlignedAllocatorForThreadLocal = allocator;
    g_DeallocatorForThreadLocal      = deallocator;
#endif
}

//-----------------------------------------------------------------------------
//  MemoryAllocatorForThreadLocalInitialized()
//
bool MemoryAllocatorForThreadLocalInitialized()
{
    return (g_AlignedAllocatorForThreadLocal != NULL);
}

}}  // namespace nn::os

//-----------------------------------------------------------------------------
//  コンパイラのスレッドローカル実装用関数の定義
//
extern "C" void* nnosAllocateMemoryForThreadLocal(size_t size)
{
    NN_SDK_ASSERT( g_AlignedAllocatorForThreadLocal != NULL, NN_TEXT_OS("コンパイラ TLS 用のメモリアロケータが未登録です。nn::os::SetMemoryAllocatorForThreadLocal() で登録して下さい。") );
    return g_AlignedAllocatorForThreadLocal(size, NN_ALIGNOF(std::max_align_t));
}

extern "C" void nnosFreeMemoryForThreadLocal(void* p, size_t size)
{
    NN_SDK_ASSERT( g_DeallocatorForThreadLocal != NULL, NN_TEXT_OS("コンパイラ TLS 用のメモリデアロケータが未登録です。nn::os::SetMemoryAllocatorForThreadLocal() で登録して下さい。") );
    if (p)
    {
        g_DeallocatorForThreadLocal(p, size);
    }
}

extern "C" int nnosGetModuleCountMax()
{
    return ModuleCountMax;
}

//-----------------------------------------------------------------------------

