﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_SdkThreadInfo.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_FiberTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_ThreadLocalRegion.h>

namespace nn { namespace os {
namespace detail {

static_assert(static_cast<Bit32>(svc::LastThreadInfoFlag_ThreadInSystemCall) == static_cast<Bit32>(os::LastThreadInfoFlag_ThreadInSystemCall), "");

inline Result GetLastThreadInfoImpl(ThreadType** pOutThread, SdkLastThreadContext* pOutLastThreadContext, Bit32* pOutFlag) NN_NOEXCEPT
{
    svc::ThreadLocalRegion* threadLocal;
    auto result = svc::GetLastThreadInfo(reinterpret_cast<svc::LastThreadContext*>(pOutLastThreadContext),
                                         reinterpret_cast<uintptr_t*>(&threadLocal),
                                         pOutFlag);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultNoThread )
        {
            return os::ResultNoThread();
        }
        NN_RESULT_CATCH( svc::ResultUnknownThread )
        {
            return os::ResultUnknownThread();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    *pOutThread = reinterpret_cast<ThreadType*>(threadLocal->pThreadType);

    NN_RESULT_SUCCESS;
}

inline void GetThreadStackInfoImpl(uintptr_t* pOutStackTop, size_t* pOutStackSize, const ThreadType* thread) NN_NOEXCEPT
{
    auto* fiber = thread->_currentFiber;

    // スレッド or ファイバのスタック情報を代入
    uintptr_t stackTop  = reinterpret_cast<uintptr_t>(
                          (fiber == NULL) ? thread->_stack
                                                 : fiber->_stack );

    size_t    stackSize = (fiber == NULL) ? thread->_stackSize
                                                 : fiber->_stackSize;

    if (pOutStackTop)
    {
        *pOutStackTop  = stackTop;
    }
    if (pOutStackSize)
    {
        *pOutStackSize = stackSize;
    }
}

}   // namespace detail
}}  // namespace nn::os
