﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <stdint.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_SystemEventTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_InterProcessEventImpl.h"
#include "os_TimeoutHelper.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_HardwareParamsSelect.h>
#include <nn/svc/svc_Synchronization.h>
#include <nn/svc/svc_Server.h>
#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Result.h>

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
// Event ハンドル生成
Result InterProcessEventImplByHorizon::Create(NativeHandle* pOutWritableHandle, NativeHandle* pOutReadableHandle) NN_NOEXCEPT
{
    nn::svc::Handle writableHandle;
    nn::svc::Handle readableHandle;
    auto result = svc::CreateEvent( &writableHandle, &readableHandle );
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            NN_RESULT_THROW( os::ResultOutOfResource() );
        }
        NN_RESULT_CATCH( svc::ResultLimit )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( os::ResultResourceLimit() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( result );
        }
    NN_RESULT_END_TRY

    *pOutWritableHandle = static_cast<nnHandle>(writableHandle).value;
    *pOutReadableHandle = static_cast<nnHandle>(readableHandle).value;

    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
// Event ハンドルのクローズ
void InterProcessEventImplByHorizon::Close(NativeHandle handle) NN_NOEXCEPT
{
    if (handle != InvalidNativeHandle)
    {
        auto result = svc::CloseHandle( nn::svc::Handle(handle) );
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

//--------------------------------------------------------------------------
// シグナル
void InterProcessEventImplByHorizon::Signal(NativeHandle handle) NN_NOEXCEPT
{
    auto result = svc::SignalEvent( nn::svc::Handle(handle) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------
// クリア
void InterProcessEventImplByHorizon::Clear(NativeHandle handle) NN_NOEXCEPT
{
    auto result = svc::ClearEvent( nn::svc::Handle(handle) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------
// プロセス間同期イベント待機
void InterProcessEventImplByHorizon::Wait(NativeHandle handle, bool autoClear) NN_NOEXCEPT
{
    for(;;)
    {
        int32_t index;
        Result ret = svc::WaitSynchronization(&index, reinterpret_cast<nn::svc::Handle*>(&handle), 1, svc::WAIT_INFINITE );
        if ( ret.IsSuccess() )
        {
            if (autoClear)
            {
                auto result = svc::ResetSignal( svc::Handle(handle) );
                if (result <= svc::ResultInvalidState())
                {
                    // アトミックなクリアに失敗したら再度 Wait する
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            return;
        }

        // 単独の InterProcessEvent では CancelSynchronization() されることは
        // ないが、多重待ちのスレッド起床で CancelSynchronization() されると、
        // その起床用内部フラグが残っている場合に ResultCancelled が返る
        // 可能性があるため、これが返ってきた場合はループして再度待ちに入る。
        NN_SDK_ASSERT( ret <= svc::ResultCancelled(), NN_TEXT_OS("nn::os::WaitSystemEvent(): SystemEvent の待機中に異常を検出しました。"));
    }
}

//--------------------------------------------------------------------------
// プロセス間同期イベントのポーリング
bool InterProcessEventImplByHorizon::TryWait(NativeHandle handle, bool autoClear) NN_NOEXCEPT
{
    if (autoClear)
    {
        Result ret = nn::svc::ResetSignal( nn::svc::Handle(handle) );
        return ret.IsSuccess();
    }

    for (;;)
    {
        int32_t index;
        Result ret = svc::WaitSynchronization(&index, reinterpret_cast<nn::svc::Handle*>(&handle), 1, 0 );
        if ( ret.IsSuccess() )
        {
            return true;
        }
        if ( ret <= svc::ResultTimeout() )
        {
            return false;
        }

        // ResultCancelled ならばループして再度ポーリングする
        NN_SDK_ASSERT( ret <= svc::ResultCancelled(), NN_TEXT_OS("nn::os::TryWaitSystemEvent(): SystemEvent のポーリング中に異常を検出しました。"));
    }
}

//--------------------------------------------------------------------------
// プロセス間同期イベント待機（タイムアウト付き）
bool InterProcessEventImplByHorizon::TimedWait(NativeHandle handle, bool autoClear, TimeSpan timeSpan) NN_NOEXCEPT
{
    TimeoutHelper   tmout( timeSpan );

    for (;;)
    {
        int32_t index;
        Result ret = svc::WaitSynchronization(&index, reinterpret_cast<nn::svc::Handle*>(&handle), 1,
                                tmout.GetLeftTimeOnTarget().GetNanoSeconds() );
        if ( ret.IsSuccess() )
        {
            if (autoClear)
            {
                auto result = svc::ResetSignal( svc::Handle(handle) );
                if (result <= svc::ResultInvalidState())
                {
                    // アトミックなクリアに失敗したら再度 Wait する
                    continue;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            return true;
        }
        if ( ret <= svc::ResultTimeout() )
        {
            return false;
        }

        // ResultCancelled ならばループして再度待ちに入る
        NN_SDK_ASSERT( ret <= svc::ResultCancelled(), NN_TEXT_OS("nn::os::TimedWaitSystemEvent(): SystemEvent のタイムアウト付き待機中に異常を検出しました。"));
    }
}

//--------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

