﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <cctype>
#include <type_traits>

#include <nn/nn_Macro.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Argument.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nn/os/os_DefaultUserExceptionHandler.h>
#include <nn/svc/svc_Base.h>
#include <nn/util/util_BitUtil.h>
#include <nn/ldr/ldr_Types.h>
#include "os_Common.h"
#include "os_OsResourceManager.h"
#include "os_DefaultUserExceptionHandlerImpl.h"
#include "os_ThreadManager.h"

namespace
{
    void CheckProcessAffinity() NN_NOEXCEPT
    {
#if defined(NN_OS_FOR_SYSTEMCORE)
        // libnn_os_systemcore は affinity が単一コアの場合だけ利用可能
        auto coreMask = nn::os::detail::GetThreadManagerInstance()->GetThreadAvailableCoreMask();
        NN_ABORT_UNLESS(1 == nn::util::popcount(coreMask));
#endif
    }

    class MemoryArranger
    {
    public:
        explicit MemoryArranger(uintptr_t address) NN_NOEXCEPT : m_Address(address)
        {
        }

        template <typename T>
        T* Arrange() NN_NOEXCEPT
        {
            Align(std::alignment_of<T>::value);
            return reinterpret_cast<T*>(Arrange(sizeof(T)));
        }

        char* ArrangeCharArray(size_t size) NN_NOEXCEPT
        {
            return reinterpret_cast<char*>(Arrange(size));
        }

        void* Arrange(size_t size) NN_NOEXCEPT
        {
            auto a = m_Address;
            m_Address += size;
            return reinterpret_cast<void*>(a);
        }

        void Align(int n) NN_NOEXCEPT
        {
            m_Address = nn::util::align_up(m_Address, n);
        }

    private:
        uintptr_t   m_Address;
    };


    // 引数領域の構造
    // 起動時
    //  0x00    0x20    ヘッダ
    //  0x20    size    引数データ
    // 以降の領域はここで作る
    //  ----    size+2  *argv
    //  ----    ----    argv

    bool HasArguments(uintptr_t argumentRegionAddress) NN_NOEXCEPT
    {
        nn::svc::MemoryInfo mi;
        nn::svc::PageInfo pi;

        nn::Result result;

        result = nn::svc::QueryMemory(&mi, &pi, argumentRegionAddress);
        if( result.IsFailure() )
        {
            return false;
        }

        return mi.permission == nn::svc::MemoryPermission_ReadWrite;
    }

    const char* SkipSpace(const char* p, const char* pEnd) NN_NOEXCEPT
    {
        for( ; (p < pEnd) && std::isspace(*p); p++ ) {}
        return p;
    }
    const char* GetTokenEnd(const char* p, const char* pEnd) NN_NOEXCEPT
    {
        for( ; (p < pEnd) && !std::isspace(*p); p++ ) {}
        return p;
    }
    const char* GetQuotedTokenEnd(const char* p, const char* pEnd) NN_NOEXCEPT
    {
        for( ; (p < pEnd) && (*p != '"'); p++ ) {}
        return p;
    }

    int MakeArgv(char** pArgvBuffer, char* pArgumentBuffer,
            const char* pCommandLine, size_t commandLineSize, int argumentCountMax) NN_NOEXCEPT
    {
        int argumentIndex = 0;
        auto pSrc = pCommandLine;
        auto pDst = pArgumentBuffer;
        auto pSrcEnd = pSrc + commandLineSize;

        for(;;)
        {
            pSrc = SkipSpace(pSrc, pSrcEnd);

            if( pSrc >= pSrcEnd )
            {
                break;
            }

            if( argumentIndex >= argumentCountMax )
            {
                NN_SDK_LOG(NN_TEXT_OS("引数解析に失敗しました"));
                break;
            }

            const char* pSrcArgumentEnd;
            const char* pSrcNext;

            if( *pSrc == '"' )
            {
                pSrc++;
                pSrcArgumentEnd = GetQuotedTokenEnd(pSrc, pSrcEnd);
                pSrcNext = pSrcArgumentEnd + 1;
            }
            else
            {
                pSrcArgumentEnd = GetTokenEnd(pSrc, pSrcEnd);
                pSrcNext = pSrcArgumentEnd;
            }

            auto argumentSize = pSrcArgumentEnd - pSrc;

            pArgvBuffer[argumentIndex] = pDst;
            std::memcpy(pDst, pSrc, argumentSize);
            pDst += argumentSize;
            *pDst++ = '\0';

            pSrc = pSrcNext;
            argumentIndex++;
        }

        *pDst++ = '\0';

        // argv を NULL 終端する
        pArgvBuffer[argumentIndex] = NULL;

        return argumentIndex;
    }
}

namespace nn { namespace os {

    void SetHostArgc(int    argc) NN_NOEXCEPT;
    void SetHostArgv(char** argv) NN_NOEXCEPT;

}}  // namespace nn::os


//--------------------------------------------------------------------------
//  OS-API ライブラリの初期化を行ないます。
//
extern "C" void nnosInitialize(uintptr_t handle, uintptr_t argumentRegionAddress) NN_NOEXCEPT
{
    // 起動時パラメータの保存
    nn::os::detail::g_OsBootParamter = handle;

    // OS 関連リソースの初期化（コンストラクタ呼び出し）
    new( nn::os::detail::GetOsResourceManagerStorage() ) nn::os::detail::OsResourceManager;
    CheckProcessAffinity();

    // 起動直後のプログラムサイズを保持するための呼び出し
    nn::os::MemoryInfo dummyMemoryInfo;
    nn::os::QueryMemoryInfo(&dummyMemoryInfo);

    // コマンドライン引数の処理
    if( HasArguments(argumentRegionAddress) )
    {
        MemoryArranger mem(argumentRegionAddress);

        const auto& header          = *mem.Arrange<nn::ldr::ArgumentRegionHeader>();    // すでに値が入っている
        const char* pCommandLine    =  mem.ArrangeCharArray(header.argumentSize);       // すでに値が入っている
        char*  pArgumentBuffer      =  mem.ArrangeCharArray(header.argumentSize + 2);   // これから値を入れる    // + 2 は \0\0 用
        char** pArgvBuffer          =  mem.Arrange<char*>();                            // これから値を入れる

        int argvBufferSize = reinterpret_cast<uintptr_t>(pArgvBuffer) - argumentRegionAddress;
        int argumentCountMax = (header.regionSize - argvBufferSize) / sizeof(char*);

        int argumentCount = MakeArgv(pArgvBuffer, pArgumentBuffer, pCommandLine, header.argumentSize, argumentCountMax);

        nn::os::SetHostArgc(argumentCount);
        nn::os::SetHostArgv(pArgvBuffer);
    }

    // デフォルトのユーザ例外ハンドラを登録
    size_t stackSize = 0;
    void*  stackTop  = nn::os::HandlerStackUsesThreadStack;
    if (nn::os::detail::IsApplication())
    {
        // アプリで、かつ、Debug/Develop ビルド時のみ専用スタックを使う
        stackTop = nnosGetDefaultUserExceptionStackForApplication(&stackSize);
    }
    nn::os::SetUserExceptionHandler( nn::os::DefaultUserExceptionHandler, stackTop, stackSize, nn::os::UserExceptionInfoUsesHandlerStack );
}
