﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/TargetConfigs/build_Compiler.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>

#include <nn/os/os_Config.h>
#include <nn/os/os_Fiber.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_UserExceptionHandler.h>
#include <nn/os/os_MemoryHeapCommon.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Types.common.h>
#include <nn/svc/svc_ThreadLocalRegion.h>

#include "os_StackGuardManagerTypes.h"
#include "os_DefaultUserExceptionHandlerImpl.h"

#include <array>
#include <cstdlib>
#include <algorithm>

#if !defined( NN_SDK_BUILD_RELEASE )

namespace nn { namespace diag {
    void PrintBacktraceItem(uintptr_t address, bool showDetail) NN_NOEXCEPT;
    bool IsSystemSymbol(uintptr_t address) NN_NOEXCEPT;
    void PrintModules(uintptr_t* traces, int traceCount) NN_NOEXCEPT;
}}

// 以下を有効にするとデフォルトユーザー例外ハンドラで詳細なメモリ情報（attr, state）を出力する。
// 出力しないで欲しいとのコメントにより普段は無効化しておく。
// SDK 開発者にとっては重宝するかもしれないので #ifdef で切って残しておく。
//#define NN_OS_ENABLE_VERBOSE_MEMORY_INFO

namespace nn { namespace os { namespace detail {

namespace {

inline bool IsPrintSystemBacktrace() NN_NOEXCEPT
{
#if defined(NN_DETAIL_OS_PRINT_SYSTEM_BACKTRACE)
    return true;
#else
    return !nn::os::detail::IsApplication();
#endif
}

inline bool IsPrintVerboseMemoryInfo() NN_NOEXCEPT
{
#if defined(NN_OS_ENABLE_VERBOSE_MEMORY_INFO)
    return true;
#else
    return !nn::os::detail::IsApplication();
#endif
}

const std::pair<nn::os::UserExceptionType, const char*> g_exceptionTypeString[] = {
    //std::make_pair( nn::os::UserExceptionType_None,                        "" ), // None の場合は不正な値として扱う
    std::make_pair( nn::os::UserExceptionType_InvalidInstructionAccess,    NN_TEXT_OS("不正なメモリ領域への命令アクセス") ),
    std::make_pair( nn::os::UserExceptionType_InvalidDataAccess,           NN_TEXT_OS("不正なメモリ領域へのデータアクセス") ),
    std::make_pair( nn::os::UserExceptionType_UnalignedInstructionAccess,  NN_TEXT_OS("不正なアライメントでの命令アクセス") ),
    std::make_pair( nn::os::UserExceptionType_UnalignedDataAccess,         NN_TEXT_OS("不正なアライメントでのデータアクセス") ),
    std::make_pair( nn::os::UserExceptionType_UndefinedInstruction,        NN_TEXT_OS("未定義命令の実行") ),
    std::make_pair( nn::os::UserExceptionType_ExceptionalInstruction,      NN_TEXT_OS("例外命令の実行") ),
    std::make_pair( nn::os::UserExceptionType_MemorySystemError,           NN_TEXT_OS("メモリシステムでのエラー") ),
    std::make_pair( nn::os::UserExceptionType_FloatingPointException,      NN_TEXT_OS("浮動小数点数演算例外") ),
    std::make_pair( nn::os::UserExceptionType_InvalidSystemCall,           NN_TEXT_OS("無効なシステムコールの呼び出し") ),
};

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
const char* registerName[16]={
    "    ", "    ", "    ", "    ", "    ", "    ", "    ", "    ",
    "    ", "(sb)", "(sl)", "(fp)", "(ip)", "(sp)", "(lr)", "(pc)"
};
#endif

const Bit32 NnDiagBreakInstruction_A64 = 0xE7FFDEFE;
const Bit32 NnDiagBreakInstruction_A32 = 0xE7FFFFFF;
const Bit16 NnDiagBreakInstruction_T32 = 0xDEFE;

// pos:   取り出す最下位ビットの位置
// width: 取り出すビット幅（32 は指定不可）
inline Bit32 GetBits( Bit32 bits, int pos, int width )
{
    Bit32 mask = (1u << width) - 1;
    return (bits >> pos) & mask;
}

inline Bit32 GetBit( Bit32 bits, int pos )
{
    return GetBits(bits, pos, 1);
}

}   // namespace

void PrintExceptionTypeForInvalidSystemCall(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT;
void PrintExceptionTypeForInvalidDataAccess(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT;
void PrintExceptionTypeForUndefinedInstruction(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT;

//-----------------------------------------------------------------------------
//  例外種別の表示
//
void PrintExceptionHeader(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    NN_UNUSED(info);

    os::ProcessId processId = {};
    nn::svc::GetProcessId(&processId.value, svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_SDK_LOG("UserException handler is called at pid=%lld\n", processId.value);
}

void PrintExceptionType(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    NN_UNUSED( NnDiagBreakInstruction_A64 );
    NN_UNUSED( NnDiagBreakInstruction_A32 );
    NN_UNUSED( NnDiagBreakInstruction_T32 );

    // Exception type
    const char* exceptionTypeString = NN_TEXT_OS("不正な ExceptionType です");
    for (auto&& e : g_exceptionTypeString)
    {
        if (info->exceptionType == e.first)
        {
            exceptionTypeString = e.second;
            break;
        }
    }

    NN_SDK_LOG("==============================================================================\n");
    if (info->exceptionType == nn::os::UserExceptionType_InvalidSystemCall)
    {
        PrintExceptionTypeForInvalidSystemCall(info, exceptionTypeString);
    }
    else if (info->exceptionType == nn::os::UserExceptionType_InvalidDataAccess)
    {
        PrintExceptionTypeForInvalidDataAccess(info, exceptionTypeString);
    }
    else if (info->exceptionType == nn::os::UserExceptionType_UndefinedInstruction)
    {
        PrintExceptionTypeForUndefinedInstruction(info, exceptionTypeString);
    }
    else
    {
        NN_SDK_LOG("  %s (ExceptionType=0x%04X)\n", exceptionTypeString, info->exceptionType);
    }
    NN_SDK_LOG("==============================================================================\n");
    NN_SDK_LOG("\n");
}

void PrintExceptionTypeForInvalidDataAccess(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT
{
    NN_SDK_LOG("  %s (ExceptionType=0x%04X)\n", exceptionTypeString, info->exceptionType);
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    // TORIAEZU: ARMv7 用の実装は省略
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A) || \
      defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    if (info->exceptionType == nn::os::UserExceptionType_InvalidDataAccess)
    {
        Bit32 esrIssDfsc = GetBits(info->detail.esr, 0, 6);
        NN_UNUSED(esrIssDfsc);

        const char* p;
        NN_UNUSED(p);
        switch (esrIssDfsc)
        {
        case 0x00 ... 0x07:
                p = NN_TEXT_OS("未マップ空間へのメモリアクセスが発生");
                break;

        case 0x0d ... 0x0f:
                p = NN_TEXT_OS("パーミション違反のメモリアクセスが発生");
                break;

        case 0x21:
                p = NN_TEXT_OS("アライメント違反のメモリアクセスが発生");
                break;

        default:
                p = NN_TEXT_OS("OS もしくはハードウェアの異常を検出");
                break;
        }

        NN_SDK_LOG("  %s (DFSC=0x%02X)\n", p, esrIssDfsc);
        NN_SDK_LOG("  Accessed to: 0x%P\n", info->detail.far);
    }
#endif
}

void PrintExceptionTypeForUndefinedInstruction(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT
{
#if defined(NN_OS_CPU_ARM_AARCH32)
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    bool isThumb = GetBit(info->detail.cpsr, 5) ? true : false;
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    bool isThumb = GetBit(info->detail.pstate, 5) ? true : false;
#endif
    Bit32 instruction = isThumb ? *reinterpret_cast<Bit16*>(info->detail.pc)
                                : *reinterpret_cast<Bit32*>(info->detail.pc);
    bool isBreakCode = isThumb ? (instruction == NnDiagBreakInstruction_T32)
                               : (instruction == NnDiagBreakInstruction_A32);
#elif defined(NN_OS_CPU_ARM_AARCH64)
    Bit32 instruction = *reinterpret_cast<Bit32*>(info->detail.pc);
    bool isBreakCode = (instruction == NnDiagBreakInstruction_A64);
#endif

    if (isBreakCode)
    {
        NN_SDK_LOG(NN_TEXT_OS("  BREAK 命令を検出しました (ExceptionType=0x%04X)\n"), info->exceptionType);
    }
    else
    {
        NN_SDK_LOG("  %s (ExceptionType=0x%04X)\n", exceptionTypeString, info->exceptionType);
    }

#if defined(NN_OS_CPU_ARM_AARCH32)
    if (isThumb)
    {
        if (instruction >= 0xE800u)
        {
            NN_SDK_LOG("  ( pc: 0x%p, InstructionCode: 0x%04x, 0x%04x )\n", info->detail.pc, instruction, *reinterpret_cast<Bit16*>(info->detail.pc + 0x2));
        }
        else
        {
            NN_SDK_LOG("  ( pc: 0x%p, InstructionCode: 0x%04x )\n", info->detail.pc, instruction);
        }
    }
    else
#endif
    {
        NN_SDK_LOG("  ( pc: 0x%p, InstructionCode: 0x%08x )\n", info->detail.pc, instruction);
    }
}

void PrintExceptionTypeForInvalidSystemCall(const nn::os::UserExceptionInfo* info, const char* exceptionTypeString) NN_NOEXCEPT
{
    NN_UNUSED(exceptionTypeString);

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
        Bit32 thumbFlag  = GetBit(info->detail.cpsr,  5);
        Bit32 svcNum;
        if (thumbFlag)
        {
            // Thumb
            Bit16* pc = reinterpret_cast<Bit16*>(info->detail.r[15]);
            svcNum = GetBits(*pc, 0, 8);
        }
        else
        {
            // ARM
            Bit32* pc = reinterpret_cast<Bit32*>(info->detail.r[15]);
            svcNum = GetBits(*pc, 0, 24);
        }
        NN_UNUSED(svcNum);
        NN_SDK_LOG(NN_TEXT_OS("  許可されていないシステムコール(%u)の呼び出し  (ExceptionType=0x%04X)\n"), svcNum, info->exceptionType);
        NN_UNUSED(svcNum);
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A) || \
      defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
        Bit32 esrISS = GetBits(info->detail.esr, 0, 25);
        NN_UNUSED(esrISS);
        NN_SDK_LOG(NN_TEXT_OS("  許可されていないシステムコール(%u)の呼び出し  (ExceptionType=0x%04X)\n"), esrISS, info->exceptionType);
        NN_UNUSED(esrISS);
#endif
}

//-----------------------------------------------------------------------------
//  スレッド/ファイバー及びスタック情報の表示
//
void PrintThreadInformation(nn::os::UserExceptionInfo* info, uintptr_t* outStackTop, uintptr_t* outStackBottom) NN_NOEXCEPT
{
    uintptr_t stackTop;
    uintptr_t stackBottom;
    nn::os::FiberType* fiber = nn::os::GetCurrentFiber();
    if (fiber == NULL)
    {
        // スレッド実行中だった場合
        nn::os::ThreadType* thread = nn::os::GetCurrentThread();
        const char* threadName     = nn::os::GetThreadNamePointer(thread);
        stackTop    = reinterpret_cast<uintptr_t>(thread->_stack);
        stackBottom = reinterpret_cast<uintptr_t>(thread->_stack) + thread->_stackSize;
        NN_SDK_LOG("Thread         : 0x%P (%s)\n", thread, threadName ? threadName : "");
        NN_UNUSED(threadName);
    }
    else
    {
        // ファイバー実行中だった場合
        stackTop    = reinterpret_cast<uintptr_t>(fiber->_stack);
        stackBottom = reinterpret_cast<uintptr_t>(fiber->_stack) + fiber->_stackSize;
        NN_SDK_LOG("Fiber          : 0x%P\n", fiber);
    }

    // スタックオーバーフローの可能性が高い場合の処理
    if (info->exceptionType == nn::os::UserExceptionType_InvalidDataAccess)
    {
        auto accessedAddress = info->detail.far;
        if ((accessedAddress >= stackTop - StackGuardSize) &&
            (accessedAddress < stackTop))
        {
            NN_SDK_LOG("  Accessed to  : 0x%P (Detect stack overflow on guard pages)\n", info->detail.far);
        }
        else if ((accessedAddress >= stackTop - 1 * 1024 * 1024 /* 1MB */) &&
                 (accessedAddress < stackTop))
        {
            NN_SDK_LOG("  Accessed to  : 0x%P (Perhaps stack overflow ?)\n", info->detail.far);
        }
    }

    NN_SDK_LOG("  Stack Top    : 0x%P\n", stackTop);
    NN_SDK_LOG("  Stack Bottom : 0x%P (size=0x%zx)\n", stackBottom, stackBottom - stackTop);
    NN_SDK_LOG("\n");

    *outStackTop    = stackTop;
    *outStackBottom = stackBottom;
}

//-----------------------------------------------------------------------------
//  汎用レジスタ等の表示
//
void PrintGeneralPurposeRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    for (int i=0; i<16; i++)
    {
        NN_SDK_LOG("r%-2d%s = 0x%08X (%11d)\n", i, registerName[i], info->detail.r[i], info->detail.r[i]);
        NN_UNUSED(registerName);
    }
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    for (int i=0; i<30; ++i)
    {
        NN_SDK_LOG("r%-2d     = 0x%016llX (%21lld)\n", i, info->detail.r[i], info->detail.r[i]);
    }
    NN_SDK_LOG("r30(lr) = 0x%016llX (%21lld)\n", info->detail.lr, info->detail.lr);
    NN_SDK_LOG("sp      = 0x%016llX (%21lld)\n", info->detail.sp, info->detail.sp);
    NN_SDK_LOG("pc      = 0x%016llX (%21lld)\n", info->detail.pc, info->detail.pc);
#endif
    NN_SDK_LOG("\n");
}

//-----------------------------------------------------------------------------
//  特別なレジスタの表示
//
void PrintSpecialRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    NN_SDK_LOG("TPIDRURO= 0x%08X\n", svc::GetThreadLocalRegion());
    Bit32 tpidrrw;
    asm volatile(" mrc p15, 0, %0, c13, c0, 2" : "=r"(tpidrrw) :: "memory");
    NN_SDK_LOG("TPIDRURW= 0x%08X\n", tpidrrw);
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    NN_SDK_LOG("tpidrro = 0x%016llX\n", svc::GetThreadLocalRegion());
    Bit64 tpidrrw;
    asm volatile(" mrs %0, tpidr_el0" : "=r"(tpidrrw) :: "memory");
    NN_SDK_LOG("tpidr   = 0x%016llX\n", tpidrrw);
#endif
    NN_SDK_LOG("\n");

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    Bit32 psr = info->detail.cpsr;
    const char* psrName = "cpsr";
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A) || \
      defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    Bit32 psr = info->detail.pstate;
    const char* psrName = "pstate";
#endif
    Bit32 psrN  = GetBit(psr, 31);
    Bit32 psrZ  = GetBit(psr, 30);
    Bit32 psrC  = GetBit(psr, 29);
    Bit32 psrV  = GetBit(psr, 28);
    Bit32 psrQ  = GetBit(psr, 27);
    Bit32 psrJ  = GetBit(psr, 24);
    Bit32 psrE  = GetBit(psr,  9);
    Bit32 psrT  = GetBit(psr,  5);
    Bit32 psrIT = GetBits(psr, 25, 2) | (GetBits(psr, 10, 6) << 2);
    Bit32 psrGE = GetBits(psr, 16, 4);
    NN_SDK_LOG("%-7s = 0x%08X (N=%d Z=%d C=%d V=%d Q=%d J=%d E=%d T=%d IT=0x%02X GE=0x%X)\n",
            psrName, psr, psrN, psrZ, psrC, psrV, psrQ, psrJ, psrE, psrT, psrIT, psrGE);

    NN_UNUSED(psrN);
    NN_UNUSED(psrZ);
    NN_UNUSED(psrC);
    NN_UNUSED(psrV);
    NN_UNUSED(psrQ);
    NN_UNUSED(psrJ);
    NN_UNUSED(psrE);
    NN_UNUSED(psrT);
    NN_UNUSED(psrIT);
    NN_UNUSED(psrGE);
    NN_UNUSED(psrName);

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A)
    Bit32 fsrExT    = GetBit(info->detail.fsr, 12);
    Bit32 fsrWnR    = GetBit(info->detail.fsr, 11);
    Bit32 fsrDomain = GetBits(info->detail.fsr, 4, 4);
    Bit32 fsrStatus = (GetBits(info->detail.fsr, 10, 1) << 4) | GetBits(info->detail.fsr, 0, 4);
    NN_SDK_LOG("fsr     = 0x%08X (ExT=%d WnR=%d Domain=%d Status=0x%02X)\n",
            info->detail.fsr, fsrExT, fsrWnR, fsrDomain, fsrStatus);
    NN_SDK_LOG("far     = 0x%08X\n", info->detail.far);
    NN_SDK_LOG("fpexc   = 0x%08X\n", info->detail.fpexc);
    NN_SDK_LOG("fpinst  = 0x%08X\n", info->detail.fpinst);
    NN_SDK_LOG("fpinst2 = 0x%08X\n", info->detail.fpinst2);

    NN_UNUSED(fsrExT);
    NN_UNUSED(fsrWnR);
    NN_UNUSED(fsrDomain);
    NN_UNUSED(fsrStatus);
#elif defined(NN_OS_CPU_ARM_AARCH32_ARMV8A) || \
      defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    Bit32 esrEC  = GetBits(info->detail.esr, 26, 6);
    Bit32 esrIL  = GetBit(info->detail.esr, 25);
    Bit32 esrISS = GetBits(info->detail.esr, 0, 25);
    NN_SDK_LOG("afsr0   = 0x%08X\n", info->detail.afsr0);
    NN_SDK_LOG("afsr1   = 0x%08X\n", info->detail.afsr1);
    NN_SDK_LOG("esr     = 0x%08X (EC=0x%02X IL=%d ISS=0x%06X)\n", info->detail.esr, esrEC, esrIL, esrISS);
    NN_SDK_LOG("far     = 0x%P\n",   info->detail.far);

    NN_UNUSED(esrEC);
    NN_UNUSED(esrIL);
    NN_UNUSED(esrISS);
#endif
    NN_SDK_LOG("\n");
}

//-----------------------------------------------------------------------------
//  浮動小数点レジスタの表示
//
void PrintFloatingPointRegisters(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    for (int i=0; i<NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS; ++i)
    {
        if (IsApplication())
        {
            NN_SDK_LOG("D%-2d = 0x%016llX (%E)\n", i, info->detail.D[i], info->detail.D[i]);
        }
        else
        {
            NN_SDK_LOG("D%-2d = 0x%016llX\n", i, info->detail.D[i]);
        }
    }
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    for (int i=0; i<32; ++i)
    {
        if (IsApplication())
        {
            NN_SDK_LOG("V%-2d = 0x%016llX%016llX (%LE)\n", i, static_cast<nn::Bit64>(info->detail.V[i] >> 64), static_cast<nn::Bit64>(info->detail.V[i]), info->detail.V[i]);
        }
        else
        {
            NN_SDK_LOG("V%-2d = 0x%016llX%016llX\n", i, static_cast<nn::Bit64>(info->detail.V[i] >> 64), static_cast<nn::Bit64>(info->detail.V[i]));
        }
    }
#endif
    NN_SDK_LOG("\n");
}

//-----------------------------------------------------------------------------
//  スタックトレースの表示
//
namespace {

template <typename T>
T GetAddressValueWithQuery(uintptr_t address)
{
    svc::MemoryInfo block;
    svc::PageInfo   page;

    nn::Result result = svc::QueryMemory(&block, &page, address);
    if (result.IsFailure())
    {
        return 0;
    }
    if (!(block.permission & svc::MemoryPermission_Read))
    {
        return 0;
    }

    return *reinterpret_cast<T*>(address);
}


}  // namespace

// システムシンボルは、セキュリティのために表示しない。
// ただし、ユーザから直接コールされている場合は表示する。
// そのために、LR は取得直後に表示せずに、次の LR を取得して、
// ユーザからコールされているか調べてから表示する。
void PrintStackTrace(nn::os::UserExceptionInfo* info) NN_NOEXCEPT
{
    NN_UNUSED(info);

    NN_SDK_LOG("Stack trace:\n");
    const int traceAddressCountMax = 32;
    int traceAddressCount = 0;
    uintptr_t traceAddresses[traceAddressCountMax];
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    uintptr_t linkAddress = static_cast<uintptr_t>(info->detail.r[14]);
    uintptr_t frameAddress = static_cast<uintptr_t>(info->detail.r[11]);
    bool isUserSymbol = !diag::IsSystemSymbol(linkAddress);
    for (; traceAddressCount < traceAddressCountMax - 1; traceAddressCount++)
    {
        if (!frameAddress)
        {
            break;
        }
        if (frameAddress & 0x3)
        {
            break;
        }
        struct
        {
            uint32_t frame;
            uint32_t lr;
        } frame;
        {
            uint32_t v = *reinterpret_cast<uint32_t*>(frameAddress);
            if (info->detail.r[13] <= v && v < info->detail.r[13] + MemoryPageSize )
            {
                // clang
                frame.frame = GetAddressValueWithQuery<uint32_t>(frameAddress);
                frame.lr = GetAddressValueWithQuery<uint32_t>(frameAddress + 4);
            }
            else
            {
                // gcc
                frame.lr = GetAddressValueWithQuery<uint32_t>(frameAddress);
                frame.frame = GetAddressValueWithQuery<uint32_t>(frameAddress - 4);
            }
            if (frame.frame == 0 || frame.lr == 0)
            {
                // 不正なアドレスをたどってしまった場合は中断する
                break;
            }
        }
        bool isCalledByUser = !diag::IsSystemSymbol(frame.lr);
        traceAddresses[traceAddressCount] = linkAddress;
        diag::PrintBacktraceItem(linkAddress, IsPrintSystemBacktrace() || isUserSymbol || isCalledByUser);
        linkAddress = frame.lr;
        frameAddress = frame.frame;
        isUserSymbol = isCalledByUser; // ユーザが呼び出しているなら、次はユーザーシンボル
    }
    traceAddresses[traceAddressCount] = linkAddress;
    diag::PrintBacktraceItem(linkAddress, IsPrintSystemBacktrace() || isUserSymbol);
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    uintptr_t linkAddress = static_cast<uintptr_t>(info->detail.r[30]);
    uintptr_t frameAddress = static_cast<uintptr_t>(info->detail.r[29]);
    bool isUserSymbol = !diag::IsSystemSymbol(linkAddress);
    for (; traceAddressCount < traceAddressCountMax - 1; traceAddressCount++)
    {
        if (!frameAddress)
        {
            break;
        }
        if (frameAddress & 0xF)
        {
            break;
        }
        struct
        {
            uint64_t frame;
            uint64_t lr;
        } frame;
        {
            frame.frame = GetAddressValueWithQuery<uint64_t>(frameAddress);
            frame.lr = GetAddressValueWithQuery<uint64_t>(frameAddress + sizeof(frame.frame));

            if (frame.frame == 0 || frame.lr == 0)
            {
                // 不正なアドレスをたどってしまった場合は中断する
                break;
            }
        }
        bool isCalledByUser = !diag::IsSystemSymbol(frame.lr);
        traceAddresses[traceAddressCount] = linkAddress;
        diag::PrintBacktraceItem(linkAddress, IsPrintSystemBacktrace() || isUserSymbol || isCalledByUser);
        linkAddress = frame.lr;
        frameAddress = frame.frame;
        isUserSymbol = isCalledByUser; // ユーザが呼び出しているなら、次はユーザーシンボル
    }
    traceAddresses[traceAddressCount] = linkAddress;
    diag::PrintBacktraceItem(linkAddress, IsPrintSystemBacktrace() || isUserSymbol);
#endif
    NN_SDK_LOG("\n");
    diag::PrintModules(traceAddresses, traceAddressCount);
}   // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//  スタックダンプの表示
//
void PrintStackDump(nn::os::UserExceptionInfo* info, uintptr_t stackTop, uintptr_t stackBottom) NN_NOEXCEPT
{
    uintptr_t dumpTop    = std::max(static_cast<uintptr_t>(nn::util::align_down(info->detail.sp, 16)), stackTop);
    uintptr_t dumpBottom = std::min(dumpTop + 16 * 16, stackBottom);

    NN_SDK_LOG("Stack dump:\n");

    for (uintptr_t ptr=dumpTop; ptr < dumpBottom; ++ptr)
    {
        if (ptr % 16 == 0)
        {
            NN_SDK_LOG("0x%P: ", ptr);
        }

        if (ptr < info->detail.sp)
        {
            NN_SDK_LOG("   ");
        }
        else
        {
            NN_SDK_LOG("%02X ", *reinterpret_cast<uint8_t*>(ptr));
        }

        if (ptr % 4 == 3)
        {
            NN_SDK_LOG(" ");
        }

        if (ptr % 16 == 15)
        {
            for (int i=-15; i<=0; i++)
            {
                uint8_t c = *reinterpret_cast<uint8_t*>(ptr + i);
                NN_SDK_LOG("%c", (c >= 0x20 && c <= 0x7e)? c : '.');
                NN_UNUSED(c);
            }
            NN_SDK_LOG("\n");
        }
    }
    NN_SDK_LOG("\n");
}

// メモリ状態の定義が想定通りか確認
static_assert(  0 == svc::MemoryState_Free,             "" );
static_assert(  1 == svc::MemoryState_Io,               "" );
static_assert(  2 == svc::MemoryState_Static,           "" );
static_assert(  3 == svc::MemoryState_Code,             "" );
static_assert(  4 == svc::MemoryState_CodeData,         "" );
static_assert(  5 == svc::MemoryState_Normal,           "" );
static_assert(  6 == svc::MemoryState_Shared,           "" );
static_assert(  7 == svc::MemoryState_Alias,            "" );
static_assert(  8 == svc::MemoryState_AliasCode,        "" );
static_assert(  9 == svc::MemoryState_AliasCodeData,    "" );
static_assert( 10 == svc::MemoryState_Ipc,              "" );
static_assert( 11 == svc::MemoryState_Stack,            "" );
static_assert( 12 == svc::MemoryState_ThreadLocal,      "" );
static_assert( 13 == svc::MemoryState_Transfered,       "" );
static_assert( 14 == svc::MemoryState_SharedTransfered, "" );
static_assert( 15 == svc::MemoryState_SharedCode,       "" );
static_assert( 16 == svc::MemoryState_Inaccessible,     "" );
static_assert( 17 == svc::MemoryState_NonSecureIpc,     "" );
static_assert( 18 == svc::MemoryState_NonDeviceIpc,     "" );

//-----------------------------------------------------------------------------
//  メモリ情報の表示
//
void PrintMemoryInformation() NN_NOEXCEPT
{
    svc::MemoryInfo block;
    svc::PageInfo   page;
    uintptr_t       ptr = 0;

    const char* memoryStateString[] = {
        "Free",
        "Io",
        "Static",
        "Code",
        "CodeData",
        "Normal",
        "Shared",
        "Alias",
        "AliasCode",
        "AliasCodeData",
        "Ipc",
        "Stack",
        "ThreadLocal",
        "Transfered",
        "SharedTransfered",
        "SharedCode",
        "Inaccessible",
        "NonSecureIpc",
        "NonDeviceIpc",
    };
    NN_UNUSED(memoryStateString);

    NN_SDK_LOG("Memory information:\n");
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
    if (IsPrintVerboseMemoryInfo())
    {
        NN_SDK_LOG("Address                Perms  Attrs  State\n");
    }
    else
    {
        NN_SDK_LOG("Address                Perms  Mem\n");
    }
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
    if (IsPrintVerboseMemoryInfo())
    {
        NN_SDK_LOG("Address                                Perms  Attrs  State\n");
    }
    else
    {
        NN_SDK_LOG("Address                                Perms  Mem\n");
    }
#endif

    do {
        nn::Result result = svc::QueryMemory(&block, &page, ptr);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // マップされている領域だけ表示する
        if (block.state != svc::MemoryState_Free &&
            block.state != svc::MemoryState_Inaccessible)
        {
            // Address
            if (block.baseAddress + block.size != 0)
            {
                NN_SDK_LOG("0x%P-0x%P  ", static_cast<uintptr_t>(block.baseAddress), static_cast<uintptr_t>(block.baseAddress + block.size));
            }
            else
            {
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
                NN_SDK_LOG("0x%P-            ", static_cast<uintptr_t>(block.baseAddress));
#elif defined(NN_OS_CPU_ARM_AARCH64_ARMV8A)
                NN_SDK_LOG("0x%P-                    ", block.baseAddress);
#endif
            }

            // Permissions
            Bit32 knownPermission = svc::MemoryPermission_Read    | svc::MemoryPermission_Write |
                                    svc::MemoryPermission_Execute | svc::MemoryPermission_DontCare;
            if (block.permission & ~knownPermission)
            {
                // 未知のパーミッションが設定されている
                NN_SDK_LOG("0x%08X  ", block.permission);
            }
            else
            {
                char permR = (block.permission & svc::MemoryPermission_Read)?     'r' : '-';
                char permW = (block.permission & svc::MemoryPermission_Write)?    'w' : '-';
                char permX = (block.permission & svc::MemoryPermission_Execute)?  'x' : '-';
                char permD = (block.permission & svc::MemoryPermission_DontCare)? 'd' : '-';

                NN_SDK_LOG("%c%c%c%c   ", permR, permW, permX, permD);

                NN_UNUSED(permR);
                NN_UNUSED(permW);
                NN_UNUSED(permX);
                NN_UNUSED(permD);
            }

            if (IsPrintVerboseMemoryInfo())
            {
                // Attributes
                Bit32 knownAttribute = svc::MemoryAttribute_Locked
                                     | svc::MemoryAttribute_IpcLocked
                                     | svc::MemoryAttribute_DeviceShared
                                     | svc::MemoryAttribute_Uncached;
                if (block.attribute & ~knownAttribute)
                {
                    // 未知の属性が設定されている
                    NN_SDK_LOG("0x%08X  ", block.permission);
                }
                else
                {
                    char attrL = (block.attribute & svc::MemoryAttribute_Locked)?       'L' : '-';
                    char attrI = (block.attribute & svc::MemoryAttribute_IpcLocked)?    'I' : '-';
                    char attrD = (block.attribute & svc::MemoryAttribute_DeviceShared)? 'D' : '-';
                    char attrU = (block.attribute & svc::MemoryAttribute_Uncached)?     'U' : '-';

                    NN_SDK_LOG("%c%c%c%c   ", attrL, attrI, attrD, attrU);
                }

                // State
                if (block.state <= svc::MemoryState_NonDeviceIpc)
                {
                    NN_SDK_LOG("%s", memoryStateString[block.state]);
                }
                else
                {
                    NN_SDK_LOG("Unknown(%d)", block.state);
                }
            }
            else
            {
                // Attributes
                char memoryLocked = (block.attribute != 0) ? 'L' : '-';
                char attrU = (block.attribute & svc::MemoryAttribute_Uncached) ? 'U' : '-';
                NN_SDK_LOG("%c%c", memoryLocked, attrU);
            }

            NN_SDK_LOG("\n");
        }

        if (ptr + block.size <= ptr)
        {
            // 一周してオーバーフローしたら終わり
            break;
        }
        ptr += block.size;

    } while (ptr > 0);  // 一周してオーバーフローしたら終わり

    NN_SDK_LOG("\n");
    NN_SDK_LOG(" * Perms: r=Read w=Write x=Execute d=DontCare\n");
    if (IsPrintVerboseMemoryInfo())
    {
        NN_SDK_LOG("   Attrs: L=Locked I=IpcLocked D=DeviceShared U=Uncached\n");
    }
    else
    {
        NN_SDK_LOG("   Attrs: L=IsMemoryLocked() U=Uncached\n");
    }
    NN_SDK_LOG("\n");
} // NOLINT(readability/fn_size)

}}}  // namespace nn::os::detail

#endif // !defined( NN_SDK_BUILD_RELEASE )
