﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_FiberTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_DebugImpl.h"
#include "os_ThreadManager.h"
#include "os_MemoryHeapManager.h"
#include "os_UserExceptionHandlerImpl.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Types.common.h>

namespace nn { namespace os { namespace detail {

//-----------------------------------------------------------------------------
//  スタック情報の取得
void DebugImpl::GetCurrentStackInfo(uintptr_t* outStack, size_t* outStackSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( outStack     != NULL );
    NN_SDK_ASSERT( outStackSize != NULL );

    auto* currentThread = detail::GetCurrentThread();
    auto* currentFiber  = currentThread->_currentFiber;

    uintptr_t currentSp = GetCurrentStackPointer();

    // スレッド or ファイバのスタック情報を代入
    uintptr_t stackTop  = reinterpret_cast<uintptr_t>(
                          (currentFiber == NULL) ? currentThread->_stack
                                                 : currentFiber->_stack );

    size_t    stackSize = (currentFiber == NULL) ? currentThread->_stackSize
                                                 : currentFiber->_stackSize;

    uintptr_t stackBottom = stackTop + stackSize;

    // ユーザ例外ハンドラが設定済み ＆ ユーザ例外ハンドラ起動中か
    if (g_UserExceptionHandler)
    {
        // スレッド or ファイバのスタック領域外なら
        // ユーザ例外ハンドラ起動中であると判断する。
        if ( (currentSp < stackTop) || (currentSp >= stackBottom) )
        {
            stackTop    = g_UserExceptionStackTop;
            stackSize   = g_UserExceptionStackBottom - stackTop;
            stackBottom = g_UserExceptionStackBottom;
        }
    }

    // スタックオーバーフローか？
    if ( !((currentSp >= stackTop) && (currentSp < stackBottom)) )
    {
        NN_SDK_LOG(NN_TEXT_OS("現在のスタック先頭： 0x%p\n"), stackTop);
        NN_SDK_LOG(NN_TEXT_OS("現在のスタック末尾： 0x%p\n"), stackBottom);
        NN_ABORT(NN_TEXT_OS("スタックオーバーフローを検出： sp=0x%p"), currentSp);
    }

    *outStack     = stackTop;
    *outStackSize = stackSize;
}

//-----------------------------------------------------------------------------
//  メモリ情報の取得
void DebugImpl::QueryMemoryInfo(MemoryInfo* info) NN_NOEXCEPT
{
    static size_t s_InitialProgramSize = 0;
    auto* threadManager = GetThreadManagerInstance();
    auto* memoryHeap    = GetMemoryHeapManagerInstance();
    Bit64 value;

    // 物理メモリの使用済みサイズ
    // これには、プログラム本体、メモリヒープ、メインスレッドスタックが含まれる
    auto result = svc::GetInfo(&value,
                               svc::InfoType_UsingUserPhysicalMemorySize,
                               svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                               0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_SDK_ASSERT(value <= SIZE_MAX);
    size_t totalUsedMemory = static_cast<size_t>(value & SIZE_MAX);


    // 使用可能な物理メモリの合計容量を取得
    result = svc::GetInfo(&value,
                          svc::InfoType_AssingedUserPhysicalMemorySize,
                          svc::PSEUDO_HANDLE_CURRENT_PROCESS,
                          0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    uint64_t availableSize = value;

    // プログラムサイズを保持するため起動直後に nnosInitialize から呼ばれる
    if (s_InitialProgramSize == 0)
    {
        // プログラム本体（Code と CodeData）のサイズを取得
        s_InitialProgramSize = totalUsedMemory -
                               memoryHeap->GetHeapSize() -
                               threadManager->GetMainThread()->_stackSize;
        NN_SDK_ASSERT(s_InitialProgramSize > 0);
    }


    // 収集した情報をセット
    info->totalAvailableMemorySize  = availableSize;
    info->totalUsedMemorySize       = totalUsedMemory;
    info->programSize               = s_InitialProgramSize;

    info->totalMemoryHeapSize       = memoryHeap->GetHeapSize();
    info->allocatedMemoryHeapSize   = memoryHeap->GetUsedHeapSize();

    info->totalThreadStackSize      = threadManager->GetTotalThreadStackSize();
    info->threadCount               = threadManager->GetThreadCountForDebug();
}

//-----------------------------------------------------------------------------
// メモリ状態チェック
//  - svc::MemoryAttribute_None 以外を検出したら false を返す
//  - 全領域が svc::MemoryAttribute_None なら true を返す
//
bool ExamineWholeAddressSpaceHasNoAttribute(uintptr_t address, size_t size) NN_NOEXCEPT
{
    svc::MemoryInfo memoryInfo;
    svc::PageInfo   pageInfo;

    auto endAddress = address + size;
    for (auto p = address; p < endAddress; p = memoryInfo.baseAddress + memoryInfo.size)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( svc::QueryMemory( &memoryInfo, &pageInfo, p ) );
        if ( memoryInfo.attribute != 0 )
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------

}}} // namespace nn::os::detail

