﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_MemoryHeapCommon.h>

#include <nn/nn_SdkLog.h>

#include "os_AddressSpaceAllocatorTypes.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "os_AslrSpaceManagerImpl-os.win32.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "os_AslrSpaceManagerImpl-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif


namespace nn { namespace os {
namespace detail {

// ページテーブルの効率のため空間をできるだけ 2MB 単位で扱う
const size_t AslrSpaceLargeAlign = 2 * 1024 * 1024;

// ガード空間用のサイズ
const size_t AslrSpaceGuardSize  = MemoryPageSize * 4;

//--------------------------------------------------------------------------
// Aslr 領域の管理用クラス
//
class AslrSpaceManager
{
public:
    //-----------------------------------------------------------------
    // コンストラクタ
    AslrSpaceManager() NN_NOEXCEPT : m_Allocator(m_Impl.GetAslrSpaceBeginAddress(), m_Impl.GetAslrSpaceEndAddress(), AslrSpaceGuardSize)
    {
    }

    //-----------------------------------------------------------------
    // Aslr 領域から指定サイズのアドレス空間が収まる空き空間を獲得する。
    void* AllocateSpace(size_t size) NN_NOEXCEPT
    {
        // まずは 2MB アラインでの確保を試みる
        void* address = m_Allocator.AllocateSpace(size, AslrSpaceLargeAlign);
        if (!address)
        {
            // ダメならページサイズアラインで確保
            address = m_Allocator.AllocateSpace(size, MemoryPageSize);
        }
        return address;
    }

    bool CheckGuardSpace(uintptr_t address, size_t size)
    {
        return m_Allocator.CheckGuardSpace(address, size, AslrSpaceGuardSize);
    }

private:
    AddressSpaceAllocator   m_Allocator;
    AslrSpaceManagerImpl    m_Impl;

};

//--------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

