﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/ncm/ncm_PackageInstallTaskBase.h>
#include <nn/ncm/ncm_ContentIdUtil.h>

#include <nn/es.h>

namespace nn { namespace ncm {

Result PackageInstallTaskBase::Initialize(const char* packageRoot, void* buffer, size_t bufferSize, StorageId storage, InstallTaskDataBase* data, Bit32 config) NN_NOEXCEPT
{
    NN_RESULT_DO(InstallTaskBase::Initialize(storage, data, config));

    m_PackageRoot.Assign(packageRoot);
    m_Buffer = buffer;
    m_BufferSize = bufferSize;

    NN_RESULT_SUCCESS;
}

Result PackageInstallTaskBase::OnWritePlaceHolder(InstallContentInfo* data) NN_NOEXCEPT
{
    InternalPath path;
    if (data->GetType() == ContentType::Meta)
    {
        CreateContentMetaPath(&path, data->info.GetId());
    }
    else
    {
        CreateContentPath(&path, data->info.GetId());
    }

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

    while (NN_STATIC_CONDITION(true))
    {
        size_t readSize;
        NN_RESULT_DO(fs::ReadFile(&readSize, file, data->written, m_Buffer, m_BufferSize));
        if (readSize == 0)
        {
            break;
        }

        NN_RESULT_DO(WritePlaceHolderBuffer(data, m_Buffer, readSize));
    }

    NN_RESULT_SUCCESS;
}

Result PackageInstallTaskBase::InstallTicket(const nn::fs::RightsId& rightsId, ContentMetaType) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    NN_UNUSED(rightsId);
#else
    // チケットの取得
    int64_t ticketFileSize;
    std::unique_ptr<char[]> ticket;
    {
        fs::FileHandle ticketFile;
        {
            InternalPath ticketPath;
            CreateTicketPath(&ticketPath, rightsId);
            NN_RESULT_DO(fs::OpenFile(&ticketFile, ticketPath, fs::OpenMode_Read));
        }
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(ticketFile); };

        NN_RESULT_DO(fs::GetFileSize(&ticketFileSize, ticketFile));

        ticket.reset(new char[static_cast<size_t>(ticketFileSize)]);
        NN_RESULT_THROW_UNLESS(ticket, ResultAllocationMemoryFailed());
        NN_RESULT_DO(fs::ReadFile(ticketFile, 0, ticket.get(), static_cast<size_t>(ticketFileSize)));
    }

    // 証明書の取得
    int64_t certificateFileSize;
    std::unique_ptr<char[]> certificate;
    {
        fs::FileHandle certificateFile;
        {
            InternalPath certificatePath;
            CreateCertificatePath(&certificatePath, rightsId);
            NN_RESULT_DO(fs::OpenFile(&certificateFile, certificatePath, fs::OpenMode_Read));
        }
        NN_UTIL_SCOPE_EXIT{ fs::CloseFile(certificateFile); };

        NN_RESULT_DO(fs::GetFileSize(&certificateFileSize, certificateFile));

        certificate.reset(new char[static_cast<size_t>(certificateFileSize)]);
        NN_RESULT_THROW_UNLESS(certificate, ResultAllocationMemoryFailed());
        NN_RESULT_DO(fs::ReadFile(certificateFile, 0, certificate.get(), static_cast<size_t>(certificateFileSize)));
    }

    // チケットのインストール
    NN_RESULT_DO(nn::es::ImportTicket(ticket.get(), static_cast<size_t>(ticketFileSize), certificate.get(), static_cast<size_t>(certificateFileSize)));
#endif

    NN_RESULT_SUCCESS;
}

void PackageInstallTaskBase::CreateContentPath(InternalPath* outValue, ContentId contentId) NN_NOEXCEPT
{
    char contentIdString[ContentIdStringLength + 1] = {};
    GetStringFromContentId(contentIdString, sizeof(contentIdString), contentId);

    outValue->AssignFormat("%s%s%s", m_PackageRoot.Get(), contentIdString, ".nca");
}

void PackageInstallTaskBase::CreateContentMetaPath(InternalPath* outValue, ContentId contentId) NN_NOEXCEPT
{
    char contentIdString[ContentIdStringLength + 1] = {};
    GetStringFromContentId(contentIdString, sizeof(contentIdString), contentId);

    outValue->AssignFormat("%s%s%s", m_PackageRoot.Get(), contentIdString, ".cnmt.nca");
}

void PackageInstallTaskBase::CreateTicketPath(InternalPath* outValue, nn::fs::RightsId rightsId) NN_NOEXCEPT
{
    char rightsIdString[RightsIdStringLength + 1] = {};
    GetStringFromRightsId(rightsIdString, sizeof(rightsIdString), rightsId);

    outValue->AssignFormat("%s%s%s", m_PackageRoot.Get(), rightsIdString, ".tik");
}

void PackageInstallTaskBase::CreateCertificatePath(InternalPath* outValue, nn::fs::RightsId rightsId) NN_NOEXCEPT
{
    char rightsIdString[RightsIdStringLength + 1] = {};
    GetStringFromRightsId(rightsIdString, sizeof(rightsIdString), rightsId);

    outValue->AssignFormat("%s%s%s", m_PackageRoot.Get(), rightsIdString, ".cert");
}

}}
