﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_MakePlaceHolderPathFunction.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {

    namespace {
        static const char PLACEHOLDER_FILE_EXTENSION[] = ".nca";

        void MakePlaceHolderFilePath(PathString* outValue, PlaceHolderId id) NN_NOEXCEPT
        {
            auto& data = id.uuid.data;
            char hexString[3];
            for (size_t i = 0; i < sizeof(data); i++)
            {
                util::SNPrintf(hexString, NN_ARRAY_SIZE(hexString), "%02x", data[i]);
                outValue->Append(hexString);
            }
            outValue->Append(PLACEHOLDER_FILE_EXTENSION);
        }

        Bit8 GenerateBit8Sha256PlaceHolderIdHashHead(PlaceHolderId id) NN_NOEXCEPT
        {
            Bit8 hash[crypto::Sha256Generator::HashSize];
            crypto::GenerateSha256Hash(hash, sizeof(hash), &id, sizeof(id));
            return hash[0];
        }
    }

    void MakeFlatPlaceHolderFilePath(PathString* outValue, PlaceHolderId id, const char* rootPath)
    {
        PathString fileName;
        MakePlaceHolderFilePath(&fileName, id);
        outValue->AssignFormat("%s/%s", rootPath, fileName.Get());
    }

    void MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster(PathString* outValue, PlaceHolderId id, const char* rootPath)
    {
        auto hashHead = GenerateBit8Sha256PlaceHolderIdHashHead(id);

        PathString fileName;
        MakePlaceHolderFilePath(&fileName, id);

        outValue->AssignFormat("%s/%08X/%s", rootPath, hashHead, fileName.Get());
    }

    int GetHierarchicalPlaceHolderDirectoryDepth(MakePlaceHolderPathFunction func) NN_NOEXCEPT
    {
        if (func == MakeFlatPlaceHolderFilePath)
        {
            return 1;
        }
        else if (func == MakeSha256HierarchicalPlaceHolderFilePath_ForFat16KCluster)
        {
            return 2;
        }
        else
        {
            NN_ABORT();
        }
    }
}}
