﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>

#include <nn/es.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Endian.h>

#include <nn/fs/fs_Context.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_StorageUtil.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/detail/ncm_Log.h>

namespace nn { namespace ncm {

namespace
{
bool Contains(const StorageContentMetaKey* commitKeys, int count, const ContentMetaKey& key, StorageId storageId) NN_NOEXCEPT
{
    return std::any_of(commitKeys, commitKeys + count,
                       [&key, &storageId](const StorageContentMetaKey& skey) NN_NOEXCEPT -> bool
                       {
                           return skey.key == key && skey.storageId == storageId;
                       });
}

bool IsExpectedKey(const ncm::ContentMetaKey& expectedKey, const ncm::ContentMetaKey& actualKey) NN_NOEXCEPT
{
    return  expectedKey.id == actualKey.id &&
            expectedKey.version == actualKey.version &&
            expectedKey.type == actualKey.type;
}
}

void InstallTaskBase::Cancel() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CancelMutex);
    m_CancelRequested = true;
}
void InstallTaskBase::ResetCancel() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CancelMutex);
    m_CancelRequested = false;
}
bool InstallTaskBase::IsCancelRequested() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> guard(m_CancelMutex);
    return m_CancelRequested;
}



Result InstallTaskBase::Initialize(StorageId storage, InstallTaskDataBase* data, Bit32 config) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsInstallableStorage(storage), ResultUnknownStorage());

    m_InstallStorage = storage;
    m_Data = data;
    m_Config = config;

    NN_RESULT_DO(m_Data->GetProgress(&m_Progress));
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::Prepare() NN_NOEXCEPT
{
    auto result = PrepareImpl();
    SetLastResult(result);
    return result;
}

Result InstallTaskBase::GetPreparedPlaceHolderPath(ncm::Path* outValue, Bit64 id, ContentMetaType metaType, ContentType contentType) NN_NOEXCEPT
{
    int count;
    NN_RESULT_DO(CountInstallContentMetaData(&count));

    util::optional<PlaceHolderId> placeHolderId;
    util::optional<StorageId> storageId;
    for (int i = 0; !placeHolderId && i < count; i++)
    {
        ncm::InstallContentMeta data;
        NN_RESULT_DO(GetInstallContentMetaData(&data, i));
        auto reader = data.GetReader();
        auto key = reader.GetKey();
        if (key.type == metaType && key.id == id)
        {
            auto contentCount = reader.CountContent();
            for (int j = 0; j < contentCount; j++)
            {
                auto contentInfo = reader.GetContentInfo(j);
                if (contentInfo->GetType() == contentType)
                {
                    placeHolderId = contentInfo->placeHolderId;
                    storageId = contentInfo->storageId;
                    break;
                }
            }
        }
    }

    NN_RESULT_THROW_UNLESS(placeHolderId, ResultPlaceHolderNotFound());
    NN_RESULT_THROW_UNLESS(storageId, ResultPlaceHolderNotFound());

    ncm::ContentStorage storage;
    NN_RESULT_DO(ncm::OpenContentStorage(&storage, *storageId));
    storage.GetPlaceHolderPath(outValue, *placeHolderId);

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::CalculateRequiredSize(int64_t* outValue) NN_NOEXCEPT
{
    int64_t requiredSize = 0;

    int count;
    NN_RESULT_DO(m_Data->Count(&count));
    for (int i = 0; i < count; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        auto contentCount = reader.CountContent();
        for (int j = 0; j < contentCount; j++)
        {
            auto info = reader.GetContentInfo(j);
            if (info->state == InstallState::NotPrepared)
            {
                requiredSize += ncm::CalculateRequiredSize(info->info.GetSize());
            }
        }
    }

    *outValue = requiredSize;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::PrepareImpl() NN_NOEXCEPT
{
    ResetThroughputMeasurement();

    // TODO: FATAL 時の処理
    if (GetProgress().state == InstallProgressState::NotPrepared)
    {
        NN_RESULT_DO(PrepareInstallContentMetaData());
        NN_RESULT_DO(PrepareDependency());
        SetProgressState(InstallProgressState::DataPrepared);
    }
    if (GetProgress().state == InstallProgressState::DataPrepared)
    {
        NN_RESULT_DO(PreparePlaceHolder());
        SetProgressState(InstallProgressState::Prepared);
    }

    NN_RESULT_DO(OnPrepareComplete());

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::Cleanup() NN_NOEXCEPT
{
    int count;
    NN_RESULT_DO(m_Data->Count(&count));
    for(int i = 0; i < count; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));
        auto result = CleanupOne(data);
        if (result.IsFailure())
        {
            NN_DETAIL_NCM_TRACE("[InstallTaskBase] Failed to cleanup one content meta as 0x%08x\n", result.GetInnerValueForDebug());
        }
    }

    NN_RESULT_DO(m_Data->Cleanup());
    CleanupProgress();

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::CleanupOne(const InstallContentMeta& data) NN_NOEXCEPT
{
    auto reader = data.GetReader();
    auto storageId = reader.GetStorageId();
    if (storageId != StorageId::None)
    {
        ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, storageId));

        for (int i = 0; i < reader.CountContent(); i++)
        {
            auto contentInfo = reader.GetContentInfo(i);
            if (contentInfo->state == InstallState::Prepared || contentInfo->state == InstallState::Installed)
            {
                auto result = storage.DeletePlaceHolder(contentInfo->placeHolderId);
                if (result.IsFailure() && !ResultPlaceHolderNotFound().Includes(result))
                {
                    NN_DETAIL_NCM_TRACE("[InstallTaskBase] Failed to delete place holder as 0x%08x\n", result.GetInnerValueForDebug());
                }
            }
        }
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::ListContentMetaKey(int* outCount, StorageContentMetaKey outList[], int count, int offset, ListContentMetaKeyFilter filter) NN_NOEXCEPT
{
    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    if (dataCount <= offset)
    {
        *outCount = 0;
        NN_RESULT_SUCCESS;
    }
    if (filter == ListContentMetaKeyFilter::All)
    {
        // all 時の最適化
        auto end = std::min(dataCount, offset + count);
        for (int i = offset; i < end; i++)
        {
            InstallContentMeta data;
            NN_RESULT_DO(m_Data->Get(&data, i));
            auto reader = data.GetReader();
            StorageContentMetaKey key = { reader.GetKey(), reader.GetStorageId() };
            outList[i - offset] = key;
        }
        *outCount = end - offset;
    }
    else
    {
        int outIndex = 0;  // 出力データの index
        int dataIndex = 0; // filter 適用後のデータの index
        for (int i = 0; i < dataCount; ++i)
        {
            InstallContentMeta data;
            NN_RESULT_DO(m_Data->Get(&data, i));
            auto reader = data.GetReader();
            bool committed = reader.GetHeader()->committedReserve;

            if ((filter == ListContentMetaKeyFilter::Committed && committed) ||
                (filter == ListContentMetaKeyFilter::NotCommitted && !committed))
            {
                if (dataIndex >= offset)
                {
                    StorageContentMetaKey key = { reader.GetKey(), reader.GetStorageId() };
                    outList[outIndex] = key;
                    outIndex++;
                }
                dataIndex++;
                if (outIndex >= count)
                {
                    break;
                }
            }
        }
        *outCount = outIndex;
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::ListApplicationContentMetaKey(int* outCount, ApplicationContentMetaKey outList[], int count, int offset) NN_NOEXCEPT
{
    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    if (dataCount <= offset)
    {
        *outCount = 0;
        NN_RESULT_SUCCESS;
    }

    int readCount = 0;
    auto end = std::min(dataCount, offset + count);
    for (int i = offset; i < end; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto appId = data.GetReader().GetApplicationId();
        if (appId)
        {
            ncm::ApplicationContentMetaKey key = { data.GetReader().GetKey(), *appId };
            outList[readCount] = key;
            readCount++;
        }
    }

    *outCount = readCount;
    NN_RESULT_SUCCESS;
}
Result InstallTaskBase::Execute() NN_NOEXCEPT
{
    auto result = ExecuteImpl();
    SetLastResult(result);
    return result;
}

Result InstallTaskBase::ExecuteImpl() NN_NOEXCEPT
{
    StartThroughputMeasurement();

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));
    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        Result updateResult;
        {
            NN_UTIL_SCOPE_EXIT{ updateResult = m_Data->Update(data, i); };

            auto writer = data.GetWriter();
            for (int j = 0; j < writer.CountContent(); j++)
            {
                auto contentInfo = writer.GetWritableContentInfo(j);
                if (contentInfo->state == InstallState::Prepared)
                {
                    NN_RESULT_DO(WritePlaceHolder(contentInfo));
                    contentInfo->state = InstallState::Installed;
                }
            }
        }
        NN_RESULT_DO(updateResult);
    }

    NN_RESULT_DO(OnExecuteComplete());

    SetProgressState(InstallProgressState::Downloaded);

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::PrepareAndExecute() NN_NOEXCEPT
{
    auto result = PrepareImpl();
    if (result.IsFailure())
    {
        SetLastResult(result);
        NN_RESULT_THROW(result);
    }

    result = ExecuteImpl();
    if (result.IsFailure())
    {
        SetLastResult(result);
        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::VerifyAllNotCommitted(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT
{
    if (commitKeys != nullptr)
    {
        int foundCount = 0;
        int dataCount;
        NN_RESULT_DO(m_Data->Count(&dataCount));
        for (int i = 0; i < dataCount; i++)
        {
            InstallContentMeta data;
            NN_RESULT_DO(m_Data->Get(&data, i));

            auto reader = data.GetReader();

            // コミット対象に含まれている場合、未コミットであることを確認
            if (Contains(commitKeys, count, reader.GetKey(), reader.GetStorageId()))
            {
                NN_RESULT_THROW_UNLESS(!(reader.GetHeader()->committedReserve), ResultListPartiallyNotCommitted());
                foundCount++;
            }
        }
        // 数が一致している = commitKeys がすべて見つかった、とする
        // 重複がないことを前提としている
        NN_RESULT_THROW_UNLESS(foundCount == count, ResultListPartiallyNotCommitted());
    }
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::CommitImpl(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT
{
    // commitKeys == nullptr .. full commit
    // commitKeys != nullptr .. partial commit
    if (GetProgress().state != InstallProgressState::Downloaded)
    {
        NN_RESULT_THROW(ResultInvalidInstallTaskState());
    }
    NN_RESULT_DO(VerifyAllNotCommitted(commitKeys, count));

    StorageList commitList;

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));
    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        auto metaSize = reader.CalculateConvertSize();
        auto contentMetaKey = reader.GetKey();

        // リストが非 null かつ、リストに含まれていないのであればスキップ
        if (commitKeys != nullptr && !Contains(commitKeys, count, contentMetaKey, reader.GetStorageId()))
        {
            continue;
        }

        // コミット済みならスキップ
        if (reader.GetHeader()->committedReserve)
        {
            continue;
        }
        Result updateResult;
        {
            auto writer = data.GetWriter();
            auto writableHeader = writer.GetWritableHeader();
            NN_UTIL_SCOPE_EXIT {updateResult = m_Data->Update(data, i); };

            std::unique_ptr<char[]> convertedMeta(new char[metaSize]);
            NN_RESULT_THROW_UNLESS(convertedMeta, ResultAllocationMemoryFailed());
            reader.ConvertToContentMeta(convertedMeta.get(), metaSize);

            // NN_DETAIL_NCM_TRACE("[InstallTaskBase] Commit 0x%016llx version %u\n", reader.GetKey().id, reader.GetKey().version);

            auto storageId = reader.GetStorageId();
            ContentStorage storage;
            NN_RESULT_DO(OpenContentStorage(&storage, storageId));
            ContentMetaDatabase db;
            NN_RESULT_DO(OpenContentMetaDatabase(&db, storageId));

            for (int j = 0; j < reader.CountContent(); j++)
            {
                auto contentInfo = reader.GetContentInfo(j);
                if (contentInfo->state == InstallState::AlreadyExists)
                {
                    // NN_DETAIL_NCM_TRACE("[InstallTaskBase] Skip register %s since already exists\n", GetContentIdString(contentInfo->info.id).data);
                    continue;
                }

                // NN_DETAIL_NCM_TRACE("[InstallTaskBase] Register %s\n", GetContentIdString(contentInfo->info.id).data);
                NN_RESULT_DO(storage.Register(contentInfo->placeHolderId, contentInfo->info.id));
            }
            ContentMetaReader convertedReader(convertedMeta.get(), metaSize);
            NN_RESULT_DO(db.Set(contentMetaKey, convertedReader.GetData(), convertedReader.GetSize()));
            writableHeader->committedReserve = true;

            commitList.Push(storageId);
        }
        NN_RESULT_DO(updateResult);
    }

    for (int i = 0; i < commitList.Count(); i++)
    {
        ContentMetaDatabase db;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, commitList[i]));
        NN_RESULT_DO(db.Commit());
    }

    // FullCommit (= commitKeys == nullptr) を呼ばないと、Committed 状態にはならないとする
    // PartialCommit だけでは Commit は完遂できないというモデル
    if (commitKeys == nullptr)
    {
        SetProgressState(InstallProgressState::Commited);
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::Commit(const StorageContentMetaKey* commitKeys, int count) NN_NOEXCEPT
{
    auto result = CommitImpl(commitKeys, count);
    if (result.IsFailure())
    {
        SetLastResult(result);
        SetProgressState(InstallProgressState::Fatal);
        NN_RESULT_THROW(result);
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::IncludesExFatDriver(bool* outValue) NN_NOEXCEPT
{
    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));
    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));
        if (data.GetReader().GetHeader()->attributes & ContentMetaAttribute::ContentMetaAttribute_IncludesExFatDriver)
        {
            *outValue = true;
            NN_RESULT_SUCCESS;
        }
    }

    *outValue = false;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::WritePlaceHolderBuffer(InstallContentInfo* data, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    // NN_DETAIL_NCM_TRACE("[InstallTaskBase] Write placeholder contentId %s, offset %lld, buffer size %zu, content size %lld\n", GetContentIdString(data->GetId()).data, data->written, bufferSize, data->info.GetSize());

    NN_RESULT_THROW_UNLESS(!IsCancelRequested(), ResultWritePlaceHolderCancelled());

    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, data->storageId));
    NN_RESULT_DO(storage.WritePlaceHolder(data->placeHolderId, data->written, buffer, bufferSize));
    data->written += bufferSize;

    if (!data->isTemporary)
    {
        IncrementProgress(bufferSize);
        UpdateThroughputMeasurement(bufferSize);
    }
    m_CurrentSha256.Update(buffer, bufferSize);

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::WritePlaceHolder(InstallContentInfo* data) NN_NOEXCEPT
{
    if (data->isShaContextAvailable)
    {
        m_CurrentSha256.InitializeWithContext(&data->context);
        m_CurrentSha256.Update(data->bufferedData, static_cast<size_t>(data->bufferedDataSize));
    }
    else
    {
        m_CurrentSha256.Initialize();
    }

    {
        NN_UTIL_SCOPE_EXIT
        {
            m_CurrentSha256.GetContext(&data->context);
            data->bufferedDataSize = m_CurrentSha256.GetBufferedDataSize();
            m_CurrentSha256.GetBufferedData(data->bufferedData, sizeof(data->bufferedData));
            data->isShaContextAvailable = true;

            // NN_DETAIL_NCM_TRACE("[InstallTaskBase] id %s suspended %lld / %lld buffered %llu\n", GetContentIdString(data->GetId()).data, data->written, data->info.GetSize(), data->bufferedDataSize);
        };

        NN_RESULT_DO(OnWritePlaceHolder(data));
    }

    // NN_DETAIL_NCM_TRACE("[InstallTaskBase] id %s written %lld / %lld\n", GetContentIdString(data->GetId()).data, data->written, data->info.GetSize());

    if (data->verifyHash)
    {
        Bit8 hash[crypto::Sha256Generator::HashSize];
        m_CurrentSha256.GetHash(hash, sizeof(hash));
        if (std::memcmp(hash, data->hash.data, sizeof(hash)) != 0)
        {
            NN_RESULT_THROW(ncm::ResultContentHashNotMatched());
        }
    }

    // チケットを無視するフラグが立っていなければチケットをインストールする
    if (!(GetConfig() & InstallConfig_IgnoreTicket))
    {
        RightsId rightsId;
        {
            nn::ncm::ContentStorage storage;
            NN_RESULT_DO(OpenContentStorage(&storage, data->storageId));

            NN_RESULT_DO(storage.GetRightsId(&rightsId, data->placeHolderId));
        }

        if (IsNecessaryInstallTicket(rightsId.id))
        {
            // チケットをインストール
            NN_RESULT_TRY(InstallTicket(rightsId.id, data->metaType))
                NN_RESULT_CATCH(ResultIgnorableInstallTicketFailure)
                {
                    NN_DETAIL_NCM_TRACE("[InstallTaskBase] Ignore failure of install ticket");
                }
            NN_RESULT_END_TRY
        }
    }

    NN_RESULT_SUCCESS;
}

bool InstallTaskBase::IsNecessaryInstallTicket(const nn::fs::RightsId& rightsId) NN_NOEXCEPT
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(rightsId);
    return false;

#else
    // 外部鍵のコンテンツかどうかを判定
    nn::fs::RightsId zeroRightsId = {};
    if (memcmp(&rightsId, &zeroRightsId, sizeof(nn::fs::RightsId)) == 0)
    {
        return false;
    }

    // チケットがチケット DB に存在するかを確認
    nn::es::RightsIdIncludingKeyId esRightsId = nn::es::RightsIdIncludingKeyId::Construct(rightsId);
    bool hasTicket;
    nn::es::OwnTicket(&hasTicket, &esRightsId, 1);

    return !hasTicket;

#endif
}

Result InstallTaskBase::PreparePlaceHolder() NN_NOEXCEPT
{
    int64_t totalSize = 0;

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));
    for (int i = 0; i < dataCount; i++)
    {
        NN_RESULT_THROW_UNLESS(!IsCancelRequested(), ResultCreatePlaceholderCancelled());

        static os::Mutex s_Mutex(false);
        std::lock_guard<os::Mutex> guard(s_Mutex);

        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        Result updateResult;
        {
            NN_UTIL_SCOPE_EXIT{ updateResult = m_Data->Update(data, i); };

            StorageId storageId;
            auto reader = data.GetReader();
            // すでにストレージ ID が決まっていた場合は、それに従う
            if (reader.GetStorageId() != StorageId::None)
            {
                storageId = reader.GetStorageId();
            }
            else
            {
                StorageId installStorage = GetInstallStorage();
                auto requiredSize = reader.CalculateContentRequiredSize();
                NN_RESULT_DO(SelectDownloadableStorage(&storageId, installStorage, requiredSize));
            }

            ContentStorage storage;
            NN_RESULT_DO(OpenContentStorage(&storage, storageId));

            auto writer = data.GetWriter();
            writer.SetStorageId(storageId);
            for (int j = 0; j < writer.CountContent(); j++)
            {
                NN_RESULT_THROW_UNLESS(!IsCancelRequested(), ResultCreatePlaceholderCancelled());

                auto contentInfo = writer.GetWritableContentInfo(j);

                bool hasAlready;
                NN_RESULT_DO(storage.Has(&hasAlready, contentInfo->info.id));
                if (hasAlready)
                {
                    // インストール済み && AlreadyExists なプレースホルダが見つかったら
                    // 書き込み自体はされているので、totalSize を増やしておく
                    // システムアップデートメタの再ダウンロードなどで発生する（SIGLO-47170）
                    if (contentInfo->state == InstallState::Installed)
                    {
                        totalSize += contentInfo->info.GetSize();
                    }
                    contentInfo->state = InstallState::AlreadyExists;
                    continue;
                }

                if (contentInfo->state == InstallState::NotPrepared)
                {
                    contentInfo->placeHolderId = storage.GeneratePlaceHolderId();
                    NN_RESULT_DO(storage.CreatePlaceHolder(contentInfo->placeHolderId, contentInfo->info.id, contentInfo->info.GetSize()));
                    contentInfo->state = InstallState::Prepared;
                }

                contentInfo->storageId = storageId;
                totalSize += contentInfo->info.GetSize();
            }
        }
        NN_RESULT_DO(updateResult);
    }

    SetTotalSize(totalSize);

    NN_RESULT_SUCCESS;
}
Result InstallTaskBase::WriteContentMetaToPlaceHolder(InstallContentInfo* installInfo, ContentStorage* storage, const InstallContentMetaInfo& info) NN_NOEXCEPT
{
    auto placeHolderId = storage->GeneratePlaceHolderId();
    NN_RESULT_DO(storage->CreatePlaceHolder(placeHolderId, info.contentId, info.contentLength));

    *installInfo = MakeInstallContentInfoFrom(info, placeHolderId);

    NN_RESULT_TRY(WritePlaceHolder(installInfo))
        NN_RESULT_CATCH_ALL
        {
            auto result = storage->DeletePlaceHolder(placeHolderId);
            if (result.IsFailure() && !ResultPlaceHolderNotFound().Includes(result))
            {
                NN_DETAIL_NCM_TRACE("[InstallTaskBase] Failed to delete place holder as 0x%08x\n", result.GetInnerValueForDebug());
            }
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    installInfo->state = InstallState::Installed;

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::PrepareContentMeta(const InstallContentMetaInfo& info, util::optional<ContentMetaKey> expectedKey, util::optional<uint32_t> deltaSourceVersion) NN_NOEXCEPT
{
    // TODO: テンポラリにシステムのストレージ
    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, StorageId::BuildInSystem));

    InstallContentInfo installInfo;
    NN_RESULT_DO(WriteContentMetaToPlaceHolder(&installInfo, &storage, info));

    auto isTemporary = installInfo.isTemporary;
    bool isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (isTemporary || !isSuccess)
        {
            auto result = storage.DeletePlaceHolder(installInfo.placeHolderId);
            if (result.IsFailure() && !ResultPlaceHolderNotFound().Includes(result))
            {
                NN_DETAIL_NCM_TRACE("[InstallTaskBase] Failed to delete place holder as 0x%08x\n", result.GetInnerValueForDebug());
            }
        }
    };

    Path path;
    storage.GetPlaceHolderPath(&path, installInfo.placeHolderId);

    if (isTemporary)
    {
        InstallContentInfo newInfo = {};
        newInfo.info = installInfo.info;
        newInfo.metaType = installInfo.metaType;
        newInfo.hash = installInfo.hash;
        newInfo.verifyHash = installInfo.verifyHash;
        newInfo.placeHolderId = installInfo.placeHolderId;

        installInfo = newInfo;
    }

    AutoBuffer installContentMeta;
    NN_RESULT_DO(GetInstallContentMetaDataFromPath(&installContentMeta, path, installInfo, deltaSourceVersion));
    // TODO: InstallContentMeta の StorageId を汎用的に決定する
    if (m_InstallStorage == StorageId::BuildInSystem)
    {
        InstallContentMetaWriter writer(installContentMeta.Get(), installContentMeta.GetSize());
        writer.SetStorageId(StorageId::BuildInSystem);
    }

    if (expectedKey)
    {
        InstallContentMetaReader reader(installContentMeta.Get(), installContentMeta.GetSize());
        NN_RESULT_THROW_UNLESS(IsExpectedKey(*expectedKey, reader.GetKey()), ResultUnexpectedContentMetaPrepared());
    }
    NN_RESULT_DO(m_Data->Push(installContentMeta.Get(), installContentMeta.GetSize()));

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::PrepareContentMeta(ContentId id, int64_t size, ContentMetaType type, AutoBuffer* meta) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(meta);

    PackagedContentMetaReader reader(meta->Get(), meta->GetSize());

    InstallContentInfo data = {};
    data.info = ContentInfo::Make(id, size, ContentType::Meta, 0);
    data.isTemporary = false;
    data.metaType = type;

    AutoBuffer installContentMeta;
    NN_RESULT_DO(installContentMeta.Initialize(reader.CalculateConvertInstallContentMetaSize()));
    reader.ConvertToInstallContentMeta(installContentMeta.Get(), installContentMeta.GetSize(), data);
    m_Data->Push(installContentMeta.Get(), installContentMeta.GetSize());

    NN_RESULT_SUCCESS;
}

void InstallTaskBase::PrepareAgain() NN_NOEXCEPT
{
    SetProgressState(InstallProgressState::NotPrepared);
}

Result InstallTaskBase::PrepareDependency() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}
Result InstallTaskBase::PrepareSystemUpdateDependency() NN_NOEXCEPT
{
    // SystemUpdate は、中断時は安全側に倒して Cleanup してしまう
    // PrepareAgain することは無いため、ダウンロードしたものが破棄されるということもない
    // (Prepare 中のキャンセルは、容量確保した Placeholder の削除以上のロスは無い)
    auto success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            auto result = Cleanup();
            if (result.IsFailure())
            {
                NN_DETAIL_NCM_TRACE("[InstallTaskBase] Failed to cleanup 0x%08x\n", result.GetInnerValueForDebug());
            }
        }
    };

    int count;
    NN_RESULT_DO(CountInstallContentMetaData(&count));

    for (int i = 0; i < count; i++)
    {
        InstallContentMeta installContentMeta;
        NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
        InstallContentMetaReader reader(installContentMeta.data.get(), installContentMeta.size);
        if (reader.GetHeader()->type == ContentMetaType::SystemUpdate)
        {
            for (int j = 0; j < reader.CountContentMeta(); j++)
            {
                auto contentMetaInfo = reader.GetContentMetaInfo(j);
                auto key = contentMetaInfo->ToKey();
                NN_DETAIL_NCM_TRACE("[InstallTaskBase] System update meta includes 0x%016llx version %u\n", key.id, key.version);

                if ((contentMetaInfo->attributes & ContentMetaAttribute_IncludesExFatDriver) &&
                    !(GetConfig() & InstallConfig_RequiresExFatDriver))
                {
                    NN_DETAIL_NCM_TRACE("[InstallTaskBase] Ignore 0x%016llx version %u since it includes exFAT driver\n", key.id, key.version);
                    continue;
                }

                NN_RESULT_DO(PrepareContentMetaIfLatest(key));
            }
            continue;
        }
    }
    success = true;

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::GetSystemUpdateTaskApplyInfo(SystemUpdateTaskApplyInfo* outValue) NN_NOEXCEPT
{
    ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, ncm::StorageId::BuiltInSystem));

    int count;
    NN_RESULT_DO(CountInstallContentMetaData(&count));

    for (int i = 0; i < count; i++)
    {
        InstallContentMeta installContentMeta;
        NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
        InstallContentMetaReader reader(installContentMeta.data.get(), installContentMeta.size);
        if (reader.GetHeader()->type == ContentMetaType::SystemUpdate)
        {
            for (int j = 0; j < reader.CountContentMeta(); j++)
            {
                auto contentMetaInfo = reader.GetContentMetaInfo(j);
                auto key = contentMetaInfo->ToKey();
                auto attributes = contentMetaInfo->attributes;

                bool isFound = true;
                ncm::ContentMetaKey latestKey;
                NN_RESULT_TRY(db.GetLatest(&latestKey, key.id))
                    NN_RESULT_CATCH(ResultContentMetaNotFound)
                    {
                        isFound = false;
                    }
                NN_RESULT_END_TRY

                bool isExFatRequired = (GetConfig() & InstallConfig_RequiresExFatDriver) != 0;
                bool isExFatContent = (attributes & ContentMetaAttribute::ContentMetaAttribute_IncludesExFatDriver) != 0;

                // ExFatDriver が必要でない、かつ IncludesExFatDriver 属性を持っている場合は判定をスキップする
                if (!isExFatRequired && isExFatContent)
                {
                    continue;
                }

                // バージョンが同じか新しいものがインストール済みの場合は判定をスキップする
                if (isFound && key.version <= latestKey.version)
                {
                    continue;
                }

                if (!(attributes & ContentMetaAttribute::ContentMetaAttribute_Rebootless))
                {
                    *outValue = SystemUpdateTaskApplyInfo::RequireReboot;
                    NN_RESULT_SUCCESS;
                }
            }
        }
    }

    *outValue = SystemUpdateTaskApplyInfo::RequireNoReboot;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::PrepareContentMetaIfLatest(const ContentMetaKey& key) NN_NOEXCEPT
{
    bool isNewer;
    NN_RESULT_DO(IsNewerThanInstalled(&isNewer, key));

    if (isNewer)
    {
        InstallContentMetaInfo info;
        NN_RESULT_DO(GetInstallContentMetaInfo(&info, key));
        NN_RESULT_DO(PrepareContentMeta(info, key));
    }
    else
    {
        NN_DETAIL_NCM_TRACE("[InstallTaskBase] Ignore 0x%016llx version %u since already latest version is installed\n", key.id, key.version);
    }

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::IsNewerThanInstalled(bool* outValue, const ContentMetaKey& key) NN_NOEXCEPT
{
    auto list = GetStorageList(m_InstallStorage);
    for (int i = 0; i < list.Count(); i++)
    {
        ContentMetaDatabase db;
        if (OpenContentMetaDatabase(&db, list[i]).IsFailure())
        {
            continue;
        }

        ContentMetaKey latest;
        NN_RESULT_TRY(db.GetLatest(&latest, key.id))
            NN_RESULT_CATCH(ResultContentMetaNotFound)
            {
                continue;
            }
        NN_RESULT_END_TRY

        if(latest.version >= key.version)
        {
            *outValue = false;
            NN_RESULT_SUCCESS;
        }
    }

    *outValue = true;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::CountInstallContentMetaData(int* outValue) NN_NOEXCEPT
{
    return m_Data->Count(outValue);
}

Result InstallTaskBase::GetInstallContentMetaData(InstallContentMeta* outValue, int index) NN_NOEXCEPT
{
    return m_Data->Get(outValue, index);
}
Result InstallTaskBase::DeleteInstallContentMetaData(const ContentMetaKey* list, int count) NN_NOEXCEPT
{
    int installContentMetaCount;
    NN_RESULT_DO(CountInstallContentMetaData(&installContentMetaCount));
    for(int i = 0; i < installContentMetaCount; ++i)
    {
        InstallContentMeta installContentMeta;
        NN_RESULT_DO(GetInstallContentMetaData(&installContentMeta, i));
        auto key = installContentMeta.GetReader().GetKey();

        for (int j = 0; j < count; ++j)
        {
            if(key == list[j])
            {
                NN_RESULT_DO(CleanupOne(installContentMeta));
            }
        }
    }
    return m_Data->Delete(list, count);
}

Result InstallTaskBase::GetInstallContentMetaDataFromPath(AutoBuffer* outValue, const Path& path, const InstallContentInfo& data, util::optional<uint32_t> deltaSourceVersion) NN_NOEXCEPT
{
    AutoBuffer meta;
    {
        fs::ScopedAutoAbortDisabler disableAbort;
        NN_RESULT_DO(ReadContentMetaPath(&meta, path.string));
    }
    PackagedContentMetaReader reader(meta.Get(), meta.GetSize());

    AutoBuffer installMetaData;
    if (deltaSourceVersion)
    {
        size_t requiredSize;
        NN_RESULT_DO(reader.CalculateConvertFragmentOnlyInstallContentMetaSize(&requiredSize, *deltaSourceVersion));
        NN_RESULT_DO(installMetaData.Initialize(requiredSize));
        reader.ConvertToFragmentOnlyInstallContentMeta(installMetaData.Get(), installMetaData.GetSize(), data, *deltaSourceVersion);
    }
    else
    {
        NN_RESULT_DO(installMetaData.Initialize(reader.CalculateConvertInstallContentMetaSize()));
        reader.ConvertToInstallContentMeta(installMetaData.Get(), installMetaData.GetSize(), data);
    }

    *outValue = std::move(installMetaData);

    NN_RESULT_SUCCESS;
}

InstallContentInfo InstallTaskBase::MakeInstallContentInfoFrom(const InstallContentMetaInfo& info, const PlaceHolderId& placeHolderId) NN_NOEXCEPT
{
    InstallContentInfo made = {};
    made.info = ContentInfo::Make(info.contentId, info.contentLength, ContentType::Meta, 0);
    made.metaType = info.type;
    made.state = InstallState::Prepared;
    made.placeHolderId = placeHolderId;
    made.storageId = StorageId::BuildInSystem;
    made.verifyHash = info.verifyHash;
    made.hash = info.hash;
    made.isTemporary = m_InstallStorage != StorageId::BuildInSystem;

    return made;
}


InstallProgress InstallTaskBase::GetProgress() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    return m_Progress;
}

void InstallTaskBase::ResetLastResult() NN_NOEXCEPT
{
    SetLastResult(ResultSuccess());
}

void InstallTaskBase::SetTotalSize(int64_t totalSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    m_Progress.totalSize = totalSize;
}

void InstallTaskBase::IncrementProgress(int64_t installedSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    m_Progress.installedSize += installedSize;
}

void InstallTaskBase::SetProgressState(InstallProgressState state) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    m_Data->SetState(state);
    m_Progress.state = state;
}

void InstallTaskBase::SetLastResult(Result result) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    m_Data->SetLastResult(result);
    std::memcpy(&m_Progress.lastResult, &result, sizeof(result));
}

void InstallTaskBase::CleanupProgress() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ProgressMutex);

    m_Progress = InstallProgress();
}

InstallThroughput InstallTaskBase::GetThroughput() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ThroughputMutex);

    return m_Throughput;
}

void InstallTaskBase::ResetThroughputMeasurement() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ThroughputMutex);

    m_Throughput = {};
    m_ThroughputMeasurementStartTime = TimeSpan(0);
}

void InstallTaskBase::StartThroughputMeasurement() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ThroughputMutex);

    m_Throughput = {};
    m_ThroughputMeasurementStartTime = os::GetSystemTick().ToTimeSpan();
}

void InstallTaskBase::UpdateThroughputMeasurement(size_t installed) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lockar(m_ThroughputMutex);

    if (m_ThroughputMeasurementStartTime != 0)
    {
        m_Throughput.installed += installed;
        m_Throughput.elapsedTime = os::GetSystemTick().ToTimeSpan() - m_ThroughputMeasurementStartTime;
    }
}

Result InstallTaskBase::CalculateContentsSize(int64_t* outValue, const ContentMetaKey& key, StorageId storageId) NN_NOEXCEPT
{
    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    ncm::ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, storageId));

    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        if (reader.GetKey() == key &&
            (!IsUniqueStorage(storageId) || reader.GetStorageId() == storageId))
        {
            *outValue = 0;
            // TotalSize を計算するときは、既にダウンロードされているかを調べていたが、
            // こちらは単純に ContentMetaKey に紐づくコンテンツが消費するサイズを調べたいので、
            // 特に現状をケアすることはしない
            for (int j = 0; j < reader.CountContent(); j++)
            {
                auto contentInfo = reader.GetContentInfo(j);

                if (contentInfo->state != InstallState::NotPrepared)
                {
                    // ダウンロード途中に SD カードを抜いて、 他の SD カードに差し替えた場合に、プレースホルダが存在しない場合があるので、その場合はサイズに計上しない
                    bool hasPlaceHolder;
                    NN_RESULT_DO(storage.HasPlaceHolder(&hasPlaceHolder, contentInfo->placeHolderId));
                    if (hasPlaceHolder)
                    {
                        *outValue += contentInfo->info.GetSize();
                    }
                }
            }
            NN_RESULT_SUCCESS;
        }
    }

    *outValue = 0;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::FindMaxRequiredApplicationVersion(uint32_t* outValue) NN_NOEXCEPT
{
    uint32_t version = 0;

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        if (reader.GetKey().type == ContentMetaType::AddOnContent)
        {
            auto header = reader.GetExtendedHeader<AddOnContentMetaExtendedHeader>();
            version = std::max(header->requiredApplicationVersion, version);
        }
    }
    *outValue = version;

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::FindMaxRequiredSystemVersion(uint32_t* outValue) NN_NOEXCEPT
{
    uint32_t version = 0;

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    for (int i = 0; i < dataCount; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        auto key = reader.GetKey();

        if (key.type == ContentMetaType::Application)
        {
            auto header = reader.GetExtendedHeader<ApplicationMetaExtendedHeader>();
            version = std::max(header->requiredSystemVersion, version);
        }
        else if (key.type == ContentMetaType::Patch)
        {
            auto header = reader.GetExtendedHeader<PatchMetaExtendedHeader>();
            version = std::max(header->requiredSystemVersion, version);
        }
    }
    *outValue = version;

    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::ListOccupiedSize(int* outCount, InstallTaskOccupiedSize* outList, int numList, int offset) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(offset >= 0);

    int dataCount;
    NN_RESULT_DO(m_Data->Count(&dataCount));

    int count = 0;
    for (int i = offset; i < dataCount && count < numList; i++)
    {
        InstallContentMeta data;
        NN_RESULT_DO(m_Data->Get(&data, i));

        auto reader = data.GetReader();
        auto storageId = reader.GetStorageId();

        int64_t size = 0;
        for (int j = 0; j < reader.CountContent(); j++)
        {
            auto contentInfo = reader.GetContentInfo(j);

            if (contentInfo->state != InstallState::NotPrepared)
            {
                ContentStorage storage;
                NN_RESULT_TRY(OpenContentStorage(&storage, storageId))
                    NN_RESULT_CATCH(ResultContentStorageNotActive)
                    {
                        break;
                    }
                NN_RESULT_END_TRY

                // ダウンロード途中に SD カードを抜いて、 他の SD カードに差し替えた場合に、プレースホルダが存在しない場合があるので、その場合はサイズに計上しない
                bool hasPlaceHolder;
                NN_RESULT_DO(storage.HasPlaceHolder(&hasPlaceHolder, contentInfo->placeHolderId));
                if (hasPlaceHolder)
                {
                    size += contentInfo->info.GetSize();
                }
            }
        }

        auto key = reader.GetKey();
        InstallTaskOccupiedSize occupiedSize = { key, size, storageId };
        outList[count++] = occupiedSize;
    }

    *outCount = count;
    NN_RESULT_SUCCESS;
}

Result InstallTaskBase::CanContinue() NN_NOEXCEPT
{
    auto progress = GetProgress();
    switch(progress.state)
    {
    case InstallProgressState::NotPrepared:
    case InstallProgressState::DataPrepared:
        {
            NN_RESULT_THROW_UNLESS(!IsCancelRequested(), ResultCreatePlaceholderCancelled());
            break;
        }
    case InstallProgressState::Prepared:
        {
            NN_RESULT_THROW_UNLESS(!IsCancelRequested(), ResultWritePlaceHolderCancelled());
            break;
        }
    // 他の状態の時はキャンセルさせない
    default: break;
    }
    NN_RESULT_SUCCESS;
}

}}
