﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <cstring>

// ビルドスイッチのため、例外的に先頭でインクルードする
#include "ncm_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h"

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_MountPrivate.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/ncm/detail/ncm_Log.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>
#include <nn/util/util_UuidApi.h>
#include "ncm_FileSystemUtility.h"

namespace nn { namespace ncm {

    namespace {
        static const char BASE_CONTENT_DIRECTORY[] = "/registered";
        static const size_t CONTENT_FILE_EXTENSION_LENGTH = 4;
        static const size_t CONTENT_FILE_ID_LENGTH = 32;
        static const size_t CONTENT_FILE_LENGTH = CONTENT_FILE_ID_LENGTH + CONTENT_FILE_EXTENSION_LENGTH;

        void MakeBaseContentDirectoryPath(PathString* outValue, const char* rootPath) NN_NOEXCEPT
        {
            outValue->AssignFormat("%s%s", rootPath, BASE_CONTENT_DIRECTORY);
        }

        void MakeContentPath(PathString* outValue, ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
        {
            PathString filePath;
            MakeBaseContentDirectoryPath(&filePath, rootPath);
            func(outValue, id, filePath);
        }

        Result EnsureContentDirectory(ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
        {
            PathString filePath;
            MakeContentPath(&filePath, id, func, rootPath);
            return detail::EnsureParentDirectoryRecursively(filePath);
        }

        Result DeleteContentFile(ContentId id, MakeContentPathFunction func, const char* rootPath) NN_NOEXCEPT
        {
            PathString filePath;
            MakeContentPath(&filePath, id, func, rootPath);

            NN_RESULT_TRY(fs::DeleteFile(filePath))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_THROW(ResultContentNotFound());
                }
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        template<typename FuncT>
        Result TraverseDirectory(bool *pOutContinue, const char* rootPath, int maxLevel, FuncT func)
        {
            if (maxLevel <= 0)
            {
                NN_RESULT_SUCCESS;
            }

            bool retryReadDirectory = true; // もう一度そのディレクトリを走査するか
            while (retryReadDirectory)
            {
                retryReadDirectory = false;

                fs::DirectoryHandle dir;
                NN_RESULT_DO(fs::OpenDirectory(&dir, rootPath, fs::OpenDirectoryMode_All | fs::OpenDirectoryModePrivate_NotRequireFileSize));
                NN_UTIL_SCOPE_EXIT{ fs::CloseDirectory(dir); };

                while(NN_STATIC_CONDITION(true))
                {
                    fs::DirectoryEntry entry;
                    int64_t entryCount;
                    NN_RESULT_DO(fs::ReadDirectory(&entryCount, &entry, dir, 1));
                    if (entryCount == 0)
                    {
                        break;
                    }

                    PathString path;
                    path.AssignFormat("%s/%s", rootPath, entry.name);

                    if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
                    {
                        bool needsContinue = true;
                        bool needsRetryReadDirectory = false;
                        NN_RESULT_DO(
                            func(&needsContinue, &needsRetryReadDirectory, path, entry));
                        if(!needsContinue)
                        {
                            *pOutContinue = false;
                            NN_RESULT_SUCCESS;
                        }
                        if (needsRetryReadDirectory)
                        {
                            retryReadDirectory = true;
                            break;
                        }
                    }

                    if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
                    {
                        bool needsContinue = true;
                        bool needsRetryReadDirectory = false;
                        NN_RESULT_DO(
                            func(&needsContinue, &needsRetryReadDirectory, path, entry));
                        if(!needsContinue)
                        {
                            *pOutContinue = false;
                            NN_RESULT_SUCCESS;
                        }
                        if (needsRetryReadDirectory)
                        {
                            retryReadDirectory = true;
                            break;
                        }

                        NN_RESULT_DO(
                            TraverseDirectory(
                                &needsContinue, path, maxLevel - 1, func));
                        if(!needsContinue)
                        {
                            *pOutContinue = false;
                            NN_RESULT_SUCCESS;
                        }
                    }
                }
            }

            NN_RESULT_SUCCESS;
        }

        template<typename FuncT>
        Result TraverseDirectory(const char* rootPath, int maxLevel, FuncT func)
        {
            bool needsContinue;
            return TraverseDirectory(&needsContinue, rootPath, maxLevel, func);
        }

        // content と思しきパスか
        bool SeemsContentPath(const char* path) NN_NOEXCEPT
        {
            detail::PathView view(path);
            if (!view.HasSuffix(".nca"))
            {
                return false;
            }
            auto fileName = view.GetFileName();

            const auto ContentNameLength = sizeof(ContentId) * 2;
            if (fileName.length() != ContentNameLength + CONTENT_FILE_EXTENSION_LENGTH)
            {
                return false;
            }
            for (int i = 0; i < ContentNameLength; ++i)
            {
                if (std::isxdigit(fileName[i]) == 0)
                {
                    return false;
                }
            }
            return true;
        }

        // placeholder と思しきパスか。現時点では content と同じとしておく
        bool SeemsPlaceHolderPath(const char* fileName) NN_NOEXCEPT
        {
            return SeemsContentPath(fileName);
        }
    }

    Result ContentStorageImpl::VerifyBase(const char* rootPath) NN_NOEXCEPT
    {
        PathString path;

        // CreateContentStorage したときに、必ず成功する場合を判別するため、 基礎ディレクトリが全て存在しない場合は ResultContentStorageBaseNotFound を返す

        bool hasDirectory;
        NN_RESULT_DO(detail::HasDirectory(&hasDirectory, rootPath));
        if (!hasDirectory)
        {
            NN_RESULT_THROW(ResultContentStorageBaseNotFound());
        }

        bool hasRegistered;
        MakeBaseContentDirectoryPath(&path, rootPath);
        NN_RESULT_DO(detail::HasDirectory(&hasRegistered, path));

        bool hasPlaceHolder;
        PlaceHolderAccessor::MakeBaseDirectoryPath(&path, rootPath);
        NN_RESULT_DO(detail::HasDirectory(&hasPlaceHolder, path));

        NN_RESULT_THROW_UNLESS(hasRegistered || hasPlaceHolder, ResultContentStorageBaseNotFound());

        // どちらかが存在しない場合には ResultInvalidContentStorageBase
        NN_RESULT_THROW_UNLESS(hasRegistered, ResultInvalidContentStorageBase());
        NN_RESULT_THROW_UNLESS(hasPlaceHolder, ResultInvalidContentStorageBase());

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::InitializeBase(const char* rootPath) NN_NOEXCEPT
    {
        PathString path;

        MakeBaseContentDirectoryPath(&path, rootPath);
        NN_RESULT_DO(detail::EnsureDirectoryRecursively(path));
        PlaceHolderAccessor::MakeBaseDirectoryPath(&path, rootPath);
        NN_RESULT_DO(detail::EnsureDirectoryRecursively(path));

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::CleanupBase(const char* rootPath) NN_NOEXCEPT
    {
        PathString path;

        MakeBaseContentDirectoryPath(&path, rootPath);
        NN_RESULT_DO(fs::CleanDirectoryRecursively(path));
        PlaceHolderAccessor::MakeBaseDirectoryPath(&path, rootPath);
        NN_RESULT_DO(fs::CleanDirectoryRecursively(path));

        NN_RESULT_SUCCESS;
    }

    ContentStorageImpl::ContentStorageImpl() NN_NOEXCEPT : m_MakeContentPathFunction(), m_IsDisabled(false) {}

    ContentStorageImpl::~ContentStorageImpl() NN_NOEXCEPT
    {
        InvalidateFileCache();
    }

    Result ContentStorageImpl::Initialize(const char* rootPath, MakeContentPathFunction func, MakePlaceHolderPathFunction placeHolderFunc, bool delayFlush) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        NN_RESULT_DO(VerifyBase(rootPath));

        m_RootPath.Assign(rootPath);
        m_MakeContentPathFunction = func;

        m_PlaceHolderAccessor.Initialize(&m_RootPath, placeHolderFunc, delayFlush);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GeneratePlaceHolderId(nn::sf::Out<PlaceHolderId> outValue) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PlaceHolderId placeHolderId = { util::GenerateUuid() };
        *outValue = placeHolderId;

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::CreatePlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        NN_RESULT_DO(EnsureContentDirectory(contentId, m_MakeContentPathFunction, m_RootPath));
        NN_RESULT_DO(m_PlaceHolderAccessor.CreatePlaceHolderFile(placeHolderId, size));

        NN_RESULT_SUCCESS;
    }
    Result ContentStorageImpl::SetPlaceHolderSize(PlaceHolderId placeHolderId, int64_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        NN_RESULT_DO(m_PlaceHolderAccessor.SetPlaceHolderFileSize(placeHolderId, size));

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::DeletePlaceHolder(PlaceHolderId id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        return m_PlaceHolderAccessor.DeletePlaceHolderFile(id);
    }

    Result ContentStorageImpl::HasPlaceHolder(nn::sf::Out<bool> outValue, PlaceHolderId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PathString path;
        m_PlaceHolderAccessor.MakePath(&path, id);
        bool hasFile;
        NN_RESULT_DO(detail::HasFile(&hasFile, path));

        outValue.Set(hasFile);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::WritePlaceHolder(PlaceHolderId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        return m_PlaceHolderAccessor.WritePlaceHolderFile(id, offset, buffer.GetPointerUnsafe(), buffer.GetSize());
    }

    Result ContentStorageImpl::Register(PlaceHolderId placeHolderId, ContentId contentId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);

        InvalidateFileCache();

        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        {
            PathString placeHolderFilePath;
            PathString contentPath;

            m_PlaceHolderAccessor.GetPath(&placeHolderFilePath, placeHolderId);
            MakeContentPath(&contentPath, contentId, m_MakeContentPathFunction, m_RootPath);

            NN_RESULT_TRY(fs::RenameFile(placeHolderFilePath, contentPath))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_THROW(ResultPlaceHolderNotFound());
                }
                NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
                {
                    NN_RESULT_THROW(ResultContentAlreadyExists());
                }
            NN_RESULT_END_TRY
        }
        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::RevertToPlaceHolder(PlaceHolderId placeHolderId, ContentId contentId, ContentId postContentId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        InvalidateFileCache();

        NN_RESULT_DO(EnsureContentDirectory(postContentId, m_MakeContentPathFunction, m_RootPath));
        NN_RESULT_DO(m_PlaceHolderAccessor.EnsurePlaceHolderDirectory(placeHolderId));
        {
            PathString placeHolderFilePath;
            PathString contentPath;

            m_PlaceHolderAccessor.GetPath(&placeHolderFilePath, placeHolderId);
            MakeContentPath(&contentPath, contentId, m_MakeContentPathFunction, m_RootPath);

            NN_RESULT_TRY(fs::RenameFile(contentPath, placeHolderFilePath))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
                {
                    NN_RESULT_THROW(ResultContentNotFound());
                }
                NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
                {
                    NN_RESULT_THROW(ResultPlaceHolderAlreadyExists());
                }
            NN_RESULT_END_TRY
        }
        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::Delete(ContentId id) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        InvalidateFileCache();

        return DeleteContentFile(id, m_MakeContentPathFunction, m_RootPath);
    }

    Result ContentStorageImpl::Has(sf::Out<bool> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_MakeContentPathFunction);
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PathString path;
        MakeContentPath(&path, id, m_MakeContentPathFunction, m_RootPath);

        bool hasFile;
        NN_RESULT_DO(detail::HasFile(&hasFile, path));

        outValue.Set(hasFile);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetPath(sf::Out<Path> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        NN_SDK_ASSERT(m_MakeContentPathFunction);

        PathString path;
        MakeContentPath(&path, id, m_MakeContentPathFunction, m_RootPath);

        Path commonPath;
        NN_RESULT_DO(nn::fs::ConvertToFsCommonPath(commonPath.string, sizeof(commonPath.string), path));

        outValue.Set(commonPath);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetFreeSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(fs::GetFreeSpaceSize(&size, m_RootPath.Get()));
        outValue.Set(size);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetTotalSpaceSize(sf::Out<std::int64_t> outValue) const NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(fs::GetTotalSpaceSize(&size, m_RootPath.Get()));
        outValue.Set(size);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetPlaceHolderPath(sf::Out<Path> outValue, PlaceHolderId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        NN_SDK_ASSERT(m_MakeContentPathFunction);

        PathString path;
        m_PlaceHolderAccessor.GetPath(&path, id);

        Path commonPath;
        NN_RESULT_DO(nn::fs::ConvertToFsCommonPath(commonPath.string, sizeof(commonPath.string), path));

        outValue.Set(commonPath);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::CleanupAllPlaceHolder() NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        m_PlaceHolderAccessor.InvalidateAll();

        PathString directoryPath;
        PlaceHolderAccessor::MakeBaseDirectoryPath(&directoryPath, m_RootPath);

        NN_RESULT_DO(fs::CleanDirectoryRecursively(directoryPath));

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::ListPlaceHolder(sf::Out<std::int32_t> outCount, const sf::OutArray<ncm::PlaceHolderId>& outList) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PathString path;
        PlaceHolderAccessor::MakeBaseDirectoryPath(&path, m_RootPath);
        const size_t listLength = static_cast<size_t>(outList.GetLength());
        size_t readCount = 0;

        NN_RESULT_DO(
            TraverseDirectory(
                path,
                m_PlaceHolderAccessor.GetHierarchicalDirectoryDepth(),
                [&](bool* pOutContinue, bool* pOutRetryReadDirectory, const char* path, const fs::DirectoryEntry &entry) NN_NOEXCEPT -> Result
                {
                    *pOutContinue = true;
                    *pOutRetryReadDirectory = false;
                    NN_UNUSED(path);

                    if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
                    {
                        NN_RESULT_THROW_UNLESS(readCount <= listLength, ResultBufferNotEnough());

                        PlaceHolderId placeHolderId;
                        NN_RESULT_DO(PlaceHolderAccessor::GetPlaceHolderIdFromFileName(&placeHolderId, entry.name));

                        outList[readCount++] = placeHolderId;
                    }

                    NN_RESULT_SUCCESS;
                }));

        *outCount = static_cast<int32_t>(readCount);
        NN_RESULT_SUCCESS;
    }


    Result ContentStorageImpl::GetContentCount(nn::sf::Out<std::int32_t> outCount) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        int64_t count = 0;

        PathString path;
        MakeBaseContentDirectoryPath(&path, m_RootPath);

        NN_RESULT_DO(
            TraverseDirectory(
                path,
                GetHierarchicalContentDirectoryDepth(m_MakeContentPathFunction),
                [&](bool* pOutContinue, bool* pOutRetryReadDirectory, const char* path, const fs::DirectoryEntry &entry) NN_NOEXCEPT -> Result
                {
                    NN_UNUSED(path);
                    NN_UNUSED(entry);

                    *pOutContinue = true;
                    *pOutRetryReadDirectory = false;

                    if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
                    {
                        count += 1;
                    }

                    NN_RESULT_SUCCESS;
                }));

        *outCount = static_cast<int32_t>(count);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::ListContentId(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::ncm::ContentId>& outList, std::int32_t offset) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        int count = 0;
        int skipCount = offset;

        PathString path;
        MakeBaseContentDirectoryPath(&path, m_RootPath);

        NN_RESULT_DO(
            TraverseDirectory(
                path,
                GetHierarchicalContentDirectoryDepth(m_MakeContentPathFunction),
                [&](bool* pOutContinue, bool* pOutRetryReadDirectory, const char* path, const fs::DirectoryEntry &entry) NN_NOEXCEPT -> Result
                {
                    NN_UNUSED(path);
                    NN_UNUSED(entry);
                    *pOutRetryReadDirectory = false;

                    if (entry.directoryEntryType != nn::fs::DirectoryEntryType_File)
                    {
                        *pOutContinue = true;
                        NN_RESULT_SUCCESS;
                    }

                    if (0 < skipCount)
                    {
                        skipCount -= 1;
                        *pOutContinue = true;
                        NN_RESULT_SUCCESS;
                    }

                    if (count < static_cast<int>(outList.GetLength()))
                    {
                        auto contentId = GetContentIdFromString(entry.name, std::strlen(entry.name));

                        if ( contentId )
                        {
                            outList[count] = contentId.value();
                            count += 1;
                        }
                    }
                    else
                    {
                        *pOutContinue = false;
                        NN_RESULT_SUCCESS;
                    }

                    *pOutContinue = true;
                    NN_RESULT_SUCCESS;
                }));

        outCount.Set(count);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetSizeFromContentId(nn::sf::Out<std::int64_t> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        PathString path;
        MakeContentPath(&path, id, m_MakeContentPathFunction, m_RootPath);

        fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(handle); };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));
        outValue.Set(fileSize);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetSizeFromPlaceHolderId(nn::sf::Out<std::int64_t> outValue, PlaceHolderId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        bool isFileSizeSet = false;
        int64_t fileSize;
        NN_RESULT_DO(m_PlaceHolderAccessor.TryGetPlaceHolderFileSize(&isFileSizeSet, &fileSize, id));
        if (isFileSizeSet)
        {
            outValue.Set(fileSize);
            NN_RESULT_SUCCESS;
        }

        PathString path;
        m_PlaceHolderAccessor.GetPath(&path, id);

        fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(handle); };

        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));
        outValue.Set(fileSize);

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::DisableForcibly() NN_NOEXCEPT
    {
        m_IsDisabled = true;
        InvalidateFileCache();
        m_PlaceHolderAccessor.InvalidateAll();

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::ReadContentIdFile(sf::OutBuffer buffer, ContentId id, int64_t offset) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        PathString path;
        MakeContentPath(&path, id, m_MakeContentPathFunction, m_RootPath);

        NN_RESULT_DO(OpenContentIdFile(id));

        NN_RESULT_DO(fs::ReadFile(m_CacheFileHandle, offset, buffer.GetPointerUnsafe(), buffer.GetSize()));

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetRightsIdFromPlaceHolderId(sf::Out<RightsId> outValue, PlaceHolderId id) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        Path path;
        NN_RESULT_DO(this->GetPlaceHolderPath(&path, id));

        fs::RightsId rightsId;
        uint8_t keyGeneration;
        NN_RESULT_DO(nn::fs::GetRightsId(&rightsId, &keyGeneration, path.string));

        *outValue = { rightsId, keyGeneration };
        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::GetRightsIdFromContentId(sf::Out<RightsId> outValue, ContentId id) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());

        Path path;
        NN_RESULT_DO(this->GetPath(&path, id));

        fs::RightsId rightsId;
        uint8_t keyGeneration;
        NN_RESULT_DO(nn::fs::GetRightsId(&rightsId, &keyGeneration, path.string));

        *outValue = { rightsId, keyGeneration };
        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::WriteContentForDebug(ContentId id, int64_t offset, sf::InBuffer buffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultInvalidOffsetArgument());
        NN_RESULT_THROW_UNLESS(!m_IsDisabled, ResultInvalidContentStorage());
        NN_SDK_ASSERT(m_MakeContentPathFunction);
#if !defined ( NN_BUILD_CONFIG_OS_WIN ) && defined ( NN_BUILD_CONFIG_SPEC_NX )
        NN_ABORT_UNLESS(spl::IsDevelopment());
#endif

        InvalidateFileCache();

        PathString path;
        MakeContentPath(&path, id, m_MakeContentPathFunction, m_RootPath);

        fs::FileHandle file;
        NN_RESULT_DO(fs::OpenFile(&file, path.Get(), fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT { fs::CloseFile(file); };

        NN_RESULT_DO(fs::WriteFile(file, offset, buffer.GetPointerUnsafe(), buffer.GetSize(), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::OpenContentIdFile(ContentId contentId) NN_NOEXCEPT
    {
        if (!(contentId == m_CacheContentId))
        {
            InvalidateFileCache();
            PathString path;
            MakeContentPath(&path, contentId, m_MakeContentPathFunction, m_RootPath);
            NN_RESULT_TRY(fs::OpenFile(&m_CacheFileHandle, path.Get(), fs::OpenMode_Read))
                NN_RESULT_CATCH(fs::ResultPathNotFound) { NN_RESULT_THROW(ResultContentNotFound()); }
            NN_RESULT_END_TRY
            m_CacheContentId = contentId;
        }

        NN_RESULT_SUCCESS;
    }

    void ContentStorageImpl::InvalidateFileCache() NN_NOEXCEPT
    {
        if (m_CacheContentId.IsValid())
        {
            fs::CloseFile(m_CacheFileHandle);
            m_CacheContentId = {};
        }
    }

    Result ContentStorageImpl::FlushPlaceHolder() NN_NOEXCEPT
    {
        m_PlaceHolderAccessor.InvalidateAll();
        NN_RESULT_SUCCESS;
    }

    Result ContentStorageImpl::RepairInvalidFileAttribute() NN_NOEXCEPT
    {

        decltype(&SeemsContentPath) checker = nullptr;
        auto callback = [&checker](bool* pOutContinue, bool* pOutRetryReadDirectory, const char* path, const fs::DirectoryEntry &entry) NN_NOEXCEPT -> Result
        {
            // ユーザが適当に追加したディレクトリの可能性もあるため、修復に失敗してもエラーにはしない

            *pOutContinue = true;
            *pOutRetryReadDirectory = false;
            if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                if (checker(path))
                {
                    NN_DETAIL_NCM_TRACE("[ContentStorageImpl] Try to repair file attribute of %s\n", path);

                    auto result = fs::SetConcatenationFileAttribute(path);
                    if (result.IsFailure())
                    {
                        NN_DETAIL_NCM_TRACE("[ContentStorageImpl] Failed to repare %s as %08x\n", path, result.GetInnerValueForDebug());
                    }
                    else
                    {
                        // 属性を変えた場合は、そのディレクトリを走査し直す
                        *pOutRetryReadDirectory = true;
                    }
                }
            }
            NN_RESULT_SUCCESS;
        };
        // content
        {
            PathString path;
            MakeBaseContentDirectoryPath(&path, m_RootPath);
            checker = SeemsContentPath;

            NN_RESULT_DO(
                TraverseDirectory(
                    path,
                    GetHierarchicalContentDirectoryDepth(m_MakeContentPathFunction),
                    callback
                ));
        }

        // placeholder
        m_PlaceHolderAccessor.InvalidateAll();
        {
            PathString path;
            PlaceHolderAccessor::MakeBaseDirectoryPath(&path, m_RootPath);
            checker = SeemsPlaceHolderPath;

            NN_RESULT_DO(
                TraverseDirectory(
                    path,
                    m_PlaceHolderAccessor.GetHierarchicalDirectoryDepth(),
                    callback
                ));
        }
        NN_RESULT_SUCCESS;
    }

}}
