﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ncm/detail/ncm_Log.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include "ncm_FileSystemUtility.h"

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>


namespace nn { namespace ncm {

namespace
{

Result VerifyMetaDeltaNotExist(IDeltaReaderWrapper* reader) NN_NOEXCEPT
{
    auto header = reader->GetHeader();
    for (int i = 0; i < header->fragmentSetCount; ++i)
    {
        auto fragmentSet = reader->GetFragmentSet(i);
        NN_RESULT_THROW_UNLESS(fragmentSet->targetContentType != ContentType::Meta, ResultUnexpectedMetaFoundInDelta());
    }
    NN_RESULT_SUCCESS;
}
int64_t GetTotalSize(IDeltaReaderWrapper* reader) NN_NOEXCEPT
{
    int64_t totalDeltaSize = 0;
    auto header = reader->GetHeader();
    for(int i = 0; i < header->fragmentSetCount; ++i)
    {
        auto fragmentSet = reader->GetFragmentSet(i);
        for (int j = 0; j < fragmentSet->fragmentCount; ++j)
        {
            auto fragment = reader->GetFragmentIndicator(i, j);
            auto contentInfo = reader->GetContentInfo(fragment->contentInfoIndex);
            totalDeltaSize += contentInfo->info.GetSize();
        }
    }
    return totalDeltaSize;
}

Result DeleteContentMeta(ContentMetaDatabase* db, ContentStorage* storage, const ContentMetaKey& key) NN_NOEXCEPT
{
    ContentId id;
    NN_RESULT_DO(db->GetContentIdByType(&id, key, ContentType::Meta));
    NN_RESULT_DO(db->Remove(key));
    NN_RESULT_DO(db->Commit());
    NN_RESULT_TRY(storage->Delete(id))
        NN_RESULT_CATCH_CONVERT(ResultContentNotFound, ResultSourcePatchContentMetaNotFound())
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

}

Result ApplyPatchDeltaTask::Initialize(const StorageContentMetaKey& source, const StorageContentMetaKey& destination, TaskState* taskState) NN_NOEXCEPT
{
    NN_RESULT_DO(InitializeCommon());
    m_TaskState = taskState;
    std::memset(m_TaskState, 0, sizeof(TaskState));
    SetProgressState(ApplyDeltaProgressState_NotPrepared);

    m_TaskState->sourceKey = source;
    m_TaskState->deltaKey = destination;

    NN_RESULT_SUCCESS;
}

Result ApplyPatchDeltaTask::InitializeForResume(TaskState* taskState) NN_NOEXCEPT
{
    NN_RESULT_DO(InitializeCommon());
    m_TaskState = taskState;

    auto progress = GetProgress();

    if (progress.state != ApplyDeltaProgressState_NotPrepared)
    {
        ncm::ContentStorage storage;
        Path path;
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));
        storage.GetPath(&path, m_TaskState->metaContentInfo.id);
        NN_RESULT_DO(ReadContentMetaPath(&m_MetaBuffer, path.string));

        PackagedContentMetaReader reader(m_MetaBuffer.Get(), m_MetaBuffer.GetSize());
        PatchMetaExtendedDataReader patchReader(reader.GetExtendedData(), reader.GetExtendedDataSize());

        NN_RESULT_DO(PrepareDeltaReader(reader, patchReader));
    }

    NN_RESULT_SUCCESS;
}

Result ApplyPatchDeltaTask::PrepareMeta() NN_NOEXCEPT
{
    // patch delta は、DB から key を削除せず、存在させたまま処理する
    NN_RESULT_THROW_UNLESS(m_TaskState->deltaKey.key.installType == ContentInstallType::FragmentOnly, ResultUnexpectedPatchMeta());

    ncm::ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->deltaKey.storageId));

    bool has;
    NN_RESULT_DO(db.Has(&has, m_TaskState->deltaKey.key));
    NN_RESULT_THROW_UNLESS(has, ResultContentMetaNotFound());

    int offset = 0;
    for (;;)
    {
        int count;
        NN_RESULT_DO(db.ListContentInfo(&count, &(m_TaskState->metaContentInfo), 1, m_TaskState->deltaKey.key, offset));
        if (m_TaskState->metaContentInfo.type == ContentType::Meta)
        {
            break;
        }
        offset += count;
    }
    NN_RESULT_THROW_UNLESS(m_TaskState->metaContentInfo.type == ContentType::Meta, ResultUnexpectedPatchMeta());

    NN_RESULT_SUCCESS;
}
Result ApplyPatchDeltaTask::PrepareDeltaReader(const PackagedContentMetaReader& reader, const PatchMetaExtendedDataReader& patchReader) NN_NOEXCEPT
{
    // 該当する更新情報を見つけて、DeltaReader として登録
    auto sourceVersion = m_TaskState->sourceKey.key.version;
    auto destinationVersion = reader.GetHeader()->version;

    int bindIndex = -1;
    auto patchHeader = patchReader.GetHeader();
    for (int i = 0; i < static_cast<int>(patchHeader->deltaCount); ++i)
    {
        auto delta = patchReader.GetPatchDeltaHeader(i);
        if(delta->delta.sourceVersion == sourceVersion && delta->delta.destinationVersion == destinationVersion)
        {
            bindIndex = i;
            break;
        }
    }
    NN_RESULT_THROW_UNLESS(bindIndex != -1, ResultDeltaNotFoundInPatch());

    ContentMetaDatabase db;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->deltaKey.storageId));

    bool has;
    NN_RESULT_DO(db.Has(&has, m_TaskState->deltaKey.key));
    NN_RESULT_THROW_UNLESS(has, ResultDeltaNotFound());
    m_DeltaReader.reset(new PatchDeltaReaderWrapper(reader.GetExtendedData(), reader.GetExtendedDataSize(), bindIndex));
    NN_RESULT_SUCCESS;
}


Result ApplyPatchDeltaTask::Prepare() NN_NOEXCEPT
{
    NN_RESULT_DO(PrepareMeta());
    ContentMetaDatabase db;
    ContentStorage storage;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->sourceKey.storageId));
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    {
        ContentStorage deltaStorage;
        NN_RESULT_DO(OpenContentStorage(&deltaStorage, m_TaskState->deltaKey.storageId));

        Path path;
        deltaStorage.GetPath(&path, m_TaskState->metaContentInfo.id);
        NN_RESULT_DO(ReadContentMetaPath(&m_MetaBuffer, path.string));
    }

    PackagedContentMetaReader reader(m_MetaBuffer.Get(), m_MetaBuffer.GetSize());
    PatchMetaExtendedDataReader patchReader(reader.GetExtendedData(), reader.GetExtendedDataSize());

    // 更新対象が正しいかの確認
    bool existInHistory;
    NN_RESULT_DO(VerifyPatchHistory(&existInHistory, &db, m_TaskState->sourceKey.key, reader));
    NN_RESULT_THROW_UNLESS(existInHistory, ResultCannotApplyDeltaSinceNoHistory());

    NN_RESULT_DO(PrepareDeltaReader(reader, patchReader));

    // Patch の場合、FragmentSet に meta があるのは不正なので、確認
    NN_RESULT_DO(VerifyMetaDeltaNotExist(m_DeltaReader.get()));

    // Prepare を完了させるのに必要なサイズを確認する
    {
        int64_t sizeRequired;
        NN_RESULT_DO(CalculateRequiredSizeMaxForPrepareInternal(&sizeRequired));

        int64_t sizeFree;
        NN_RESULT_DO(storage.GetFreeSpaceSize(&sizeFree));
        NN_RESULT_THROW_UNLESS(sizeRequired <= sizeFree, ResultNotEnoughSpaceToApplyDelta());
    }

    // Source の meta を削除し、DB からも抹消
    NN_RESULT_DO(DeleteContentMeta(&db, &storage, m_TaskState->sourceKey.key));

    // placeholder の作成
    auto header = m_DeltaReader->GetHeader();
    m_TaskState->fragmentSetCount = header->fragmentSetCount;

    for(int i = 0; i < m_TaskState->fragmentSetCount; ++i)
    {
        auto placeHolder = storage.GeneratePlaceHolderId();
        m_TaskState->placeHolders[i] = placeHolder;
        NN_RESULT_DO(CreatePlaceHolder(i, placeHolder));
    }

    // コンテントメタを placeholder にコピーする
    m_TaskState->metaPlaceHolder = storage.GeneratePlaceHolderId();
    NN_RESULT_DO(CopyContentMeta(&storage, m_TaskState->metaPlaceHolder));

    // プログレス全体サイズの決定
    SetTotalDeltaSize(GetTotalSize(m_DeltaReader.get()));
    SetProgressState(ApplyDeltaProgressState_Prepared);

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTask::Initialize(const StorageContentMetaKey& source, const StorageContentMetaKey& delta, TaskState* taskState) NN_NOEXCEPT
{
    NN_RESULT_DO(InitializeCommon());
    m_TaskState = taskState;
    std::memset(m_TaskState, 0, sizeof(TaskState));
    SetProgressState(ApplyDeltaProgressState_NotPrepared);

    m_TaskState->sourceKey = source;
    m_TaskState->deltaKey = delta;

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTask::Prepare() NN_NOEXCEPT
{
    // メタの読み込み
    Path path;
    NN_RESULT_DO(GetContentMetaPath(&path, m_TaskState->deltaKey));
    NN_RESULT_DO(ReadContentMetaPath(&m_MetaBuffer, path.string));

    PackagedContentMetaReader reader(m_MetaBuffer.Get(), m_MetaBuffer.GetSize());
    m_DeltaReader.reset(new DeltaReaderWrapper(m_MetaBuffer.Get(), m_MetaBuffer.GetSize(), reader.GetExtendedData(), reader.GetExtendedDataSize()));

    // placeholder の作成
    auto header = m_DeltaReader->GetHeader();
    m_TaskState->fragmentSetCount = header->fragmentSetCount;

    ContentMetaDatabase db;
    ContentStorage storage;
    NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->sourceKey.storageId));
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    // 更新対象のパッチとして、期待するものがインストールされているか確認
    // FragmentSet の SourceContentId がそろっているかを見る
    for(int i = 0; i < m_TaskState->fragmentSetCount; ++i)
    {
        auto fragmentSet = m_DeltaReader->GetFragmentSet(i);
        if (fragmentSet->updateType != UpdateType::Create)
        {
            bool has;
            NN_RESULT_DO(storage.Has(&has, fragmentSet->sourceContentId));
            NN_RESULT_THROW_UNLESS(has, ResultCannotApplyDeltaSinceContentNotFound());
        }
    }

    // DB から source を削除
    db.Remove(m_TaskState->sourceKey.key);
    db.Commit();

    for(int i = 0; i < m_TaskState->fragmentSetCount; ++i)
    {
        auto fragmentSet = m_DeltaReader->GetFragmentSet(i);
        auto placeHolder = storage.GeneratePlaceHolderId();
        m_TaskState->placeHolders[i] = placeHolder;
        NN_RESULT_DO(CreatePlaceHolder(i, placeHolder));

        if(fragmentSet->targetContentType == ContentType::Meta)
        {
            // meta は別枠。そのため、m_PlaceHolders[MetaIndex] はとりあえず設定するけれども今後利用しない
            m_TaskState->metaPlaceHolder = placeHolder;
            m_TaskState->metaContentInfo = ContentInfo::Make(fragmentSet->destinationContentId, fragmentSet->GetDestinationSize(), ContentType::Meta, 0);
        }
    }

    // プログレス全体サイズの決定
    SetTotalDeltaSize(GetTotalSize(m_DeltaReader.get()));

    SetProgressState(ApplyDeltaProgressState_Prepared);

    NN_RESULT_SUCCESS;
}
}}
