﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/mbuf/detail/mbuf_MbufPool.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/util/util_BitUtil.h>
#include "../mbuf_MbufImpl.h"
#include <nn/nn_SdkText.h>

namespace nn { namespace mbuf { namespace detail { namespace
{
    int GetValue(uint64_t taggedValue) NN_NOEXCEPT
    {
        return static_cast<int>(taggedValue & static_cast<uint64_t>(0xffffffff));
    }

}}}} // end of namespace nn::mbuf::detail::<unnamed>

namespace nn { namespace mbuf { namespace detail
{
    MbufPool::MbufPool() NN_NOEXCEPT
        : m_pMetaInfo(nullptr),
          m_Base(0),
          m_Id(0)
    {
    }

    MbufPool::~MbufPool() NN_NOEXCEPT
    {
        if (m_Base != 0)
        {
            Finalize();
        }
    }

    void MbufPool::Initialize(
        int id, size_t unitSize, int count, int type, uintptr_t base, bool isServer) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Base == 0, NN_TEXT("mbuf プールは既に初期化されています。"));
        NN_SDK_ASSERT_NOT_EQUAL(base, 0U);
        NN_SDK_ASSERT(MbufUnitSizeMin <= unitSize && unitSize <= MbufUnitSizeMax);
        NN_SDK_ASSERT(PoolCapacityMin <= count && count <= PoolCapacityMax);

        // mbuf プールのパラメータを保存しておきます。
        m_Base = base;
        m_Id = static_cast<Bit16>(id);

        // メタ情報を取得します。
        size_t usableSize = GetRequiredMemorySize(unitSize, count) - sizeof(MbufPoolMetaInfo);
        m_pMetaInfo = reinterpret_cast<MbufPoolMetaInfo*>(m_Base + usableSize);

        // 自身が mbuf プールを管理している場合は共有領域を構築します。
        // 自身で管理していない場合、既に構築済の共有領域を受け取っていることを確認します。
        if(isServer)
        {
            m_pMetaInfo->Clear(unitSize, count, type);
            ConstructFreeChain();
        }
        else
        {
            NN_ABORT_UNLESS(m_pMetaInfo->IsValid());
            NN_ABORT_UNLESS_EQUAL(m_pMetaInfo->unitSize, unitSize);
            NN_ABORT_UNLESS_EQUAL(m_pMetaInfo->maxCount, count);
            NN_ABORT_UNLESS_EQUAL(m_pMetaInfo->type, type);
        }
    }

    void MbufPool::Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Base != 0, NN_TEXT("mbuf プールは初期化されていません。"));
        NN_SDK_ASSERT(IsClean(), NN_TEXT("解放されていない mbuf クラスタが存在します。"));
        m_pMetaInfo     = nullptr;
        m_Base          = 0;
        m_Id            = 0;
    }

    Mbuf* MbufPool::TryAllocate(int type) NN_NOEXCEPT
    {
        // 用途がそぐわない場合には失敗します。
        if (type != m_pMetaInfo->type)
        {
            return nullptr;
        }

        // 他のプロセスと衝突しないようにアロケートします。
        std::atomic<uint32_t>& taggedFreeIndex = m_pMetaInfo->taggedFreeIndex;
        while( NN_STATIC_CONDITION(true) )
        {
            // フリーリストにノードが存在しない場合は割り当てに失敗します。
            uint32_t oldTaggedFreeIndex = taggedFreeIndex.load();
            int freeIndex = GetValue(oldTaggedFreeIndex);
            if( freeIndex == -1 )
            {
                return nullptr;
            }

            // フリーリストからの割り当てを試行します。失敗した場合は再試行です。
            Mbuf* pFreeMbuf = GetPtrFromIndex(freeIndex);
            if( taggedFreeIndex.compare_exchange_weak(
                oldTaggedFreeIndex, GetNextFreeIndex(pFreeMbuf)) != false )
            {
                m_pMetaInfo->IncrementUsed();
                InitializeMbuf(pFreeMbuf, m_pMetaInfo->unitSize, m_Id);
                return pFreeMbuf;
            }
        }
    }

    void MbufPool::Free(Mbuf* pMbuf) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pMbuf);
        NN_SDK_ASSERT_EQUAL(GetPoolIdMbuf(pMbuf), m_Id);
        NN_SDK_ASSERT_EQUAL(GetCapacityMbuf(pMbuf), m_pMetaInfo->unitSize - MbufHeaderSize);
        NN_SDK_ASSERT(m_Base <= reinterpret_cast<uintptr_t>(pMbuf));
        NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(pMbuf) <
                      m_Base + m_pMetaInfo->maxCount * m_pMetaInfo->unitSize);
        NN_SDK_ASSERT_EQUAL(GetFlagsMbuf(pMbuf) & MbufPacketHeaderFlag_FreeList, 0);
        NN_SDK_ASSERT(!IsLinkingMbuf(pMbuf));

        // mbuf を解放します。
        SetFlagsMbuf(pMbuf, MbufPacketHeaderFlag_FreeList);
        int freeingIndex = GetIndexFromPtr(pMbuf);
        std::atomic<uint32_t>& taggedFreeIndex = m_pMetaInfo->taggedFreeIndex;

        // 開放した瞬間に確保が来ると数がおかしくなるので、先に減らしておきます。
        m_pMetaInfo->DecrementUsed();

        // フリーリストに接続します。
        uint32_t oldTaggedFreeIndex;
        do
        {
            oldTaggedFreeIndex = taggedFreeIndex.load();
            SetNextFreeIndex(pMbuf, GetValue(oldTaggedFreeIndex));
        } while (!taggedFreeIndex.compare_exchange_weak(oldTaggedFreeIndex, freeingIndex));
    }

    void MbufPool::ConstructFreeChain() NN_NOEXCEPT
    {
        const int maxCount = GetClusterCount();

        // 全ての mbuf クラスタを Free リストに追加します。
        for( int index = 0; index < maxCount; ++index )
        {
            Mbuf* pMbuf = GetPtrFromIndex(index);
            InitializeMbuf(pMbuf, 0, 0xffff);
            SetNextFreeIndex(pMbuf, index + 1);
        }

        // 末尾の mbuf クラスタは NextFreeIndex を -1 に設定します。
        Mbuf* pTail = GetPtrFromIndex(maxCount - 1);
        SetNextFreeIndex(pTail, -1);

        // Free リストの先頭をインデックス 0 に設定します。
        m_pMetaInfo->taggedFreeIndex = 0;
        NN_SDK_ASSERT(IsClean());
    }

    bool MbufPool::IsClean() const NN_NOEXCEPT
    {
        // 1 つでも使用中の mbuf クラスタが存在する場合は clean ではありません。
        if(0 < m_pMetaInfo->usedCount)
        {
            return false;
        }

        // Free リストに全てのクラスタが結合されていることを確認します。
        int count = 0;
        int index = GetValue(m_pMetaInfo->taggedFreeIndex);
        while (count <= GetClusterCount() && index != -1)
        {
            auto* pMbufFree = reinterpret_cast<const detail::MbufFree*>(GetPtrFromIndex(index));
            index = pMbufFree->_nextFree;
            ++count;
        }
        return index == -1 && count == GetClusterCount();
    }

    size_t MbufPool::GetRequiredMemorySize(size_t unitSize, int count) NN_NOEXCEPT
    {
        return nn::util::align_up(count * unitSize + sizeof(MbufPoolMetaInfo), nn::os::MemoryPageSize);
    }

    Mbuf* MbufPool::GetPtrFromIndex(int index) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, m_pMetaInfo->maxCount);
        return reinterpret_cast<Mbuf*>(m_Base + m_pMetaInfo->unitSize * index);
    }

    const Mbuf* MbufPool::GetPtrFromIndex(int index) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, m_pMetaInfo->maxCount);
        return reinterpret_cast<Mbuf*>(m_Base + m_pMetaInfo->unitSize * index);
    }

    int MbufPool::GetIndexFromPtr(const Mbuf* pMbuf) const NN_NOEXCEPT
    {
        if(pMbuf != nullptr)
        {
            uintptr_t offset = reinterpret_cast<uintptr_t>(pMbuf) - m_Base;
            NN_SDK_ASSERT((offset % GetUnitSizeMbuf(pMbuf)) == 0);
            return static_cast<int>(offset / GetUnitSizeMbuf(pMbuf));
        }
        else
        {
            return InvalidIndex;
        }
    }

}}} // end of namespace nn::mbuf::detail
