﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_LocationRedirector.h>

namespace nn { namespace lr {

    class Redirection : public util::IntrusiveListBaseNode <Redirection>
    {
        NN_DISALLOW_COPY(Redirection);
        NN_DISALLOW_MOVE(Redirection);

    public:
        Redirection(ncm::ProgramId id, const Path& path, int flag) NN_NOEXCEPT : m_Id(id), m_Path(path), m_Flag(flag) {}

        bool Matches(ncm::ProgramId id) NN_NOEXCEPT
        {
            return m_Id == id;
        }

        void GetPath(Path* outValue) const NN_NOEXCEPT
        {
            *outValue = m_Path;
        }

        int GetFlag() const NN_NOEXCEPT
        {
            return m_Flag;
        }

        void SetFlag(int flag) NN_NOEXCEPT
        {
            m_Flag = flag;
        }

    private:
        ncm::ProgramId m_Id;
        Path m_Path;
        int m_Flag;
    };

    LocationRedirector::~LocationRedirector() NN_NOEXCEPT
    {
        ClearRedirection();
    }

    void LocationRedirector::ClearRedirection(int flag) NN_NOEXCEPT
    {
        for (auto iter = m_RedirectionList.begin(); iter != m_RedirectionList.end();)
        {
            if ((iter->GetFlag() & flag) == flag)
            {
                auto tmpIter = iter;
                iter = m_RedirectionList.erase(iter);
                delete &(*tmpIter);
            }
            else
            {
                iter++;
            }
        }
    }

    bool LocationRedirector::FindRedirection(Path* outValue, ncm::ProgramId id) NN_NOEXCEPT
    {
        for (auto& redirection : m_RedirectionList)
        {
            if (redirection.Matches(id))
            {
                redirection.GetPath(outValue);
                return true;
            }
        }

        return false;
    }

    void LocationRedirector::SetRedirection(ncm::ProgramId id, const Path& path, int flag) NN_NOEXCEPT
    {
        for (auto& redirection : m_RedirectionList)
        {
            if (redirection.Matches(id))
            {
                m_RedirectionList.erase(m_RedirectionList.iterator_to(redirection));
                delete &redirection;
                break;
            }
        }

        m_RedirectionList.push_back(*(new Redirection(id, path, flag)));
    }

    void LocationRedirector::SetRedirectionFlag(ncm::ProgramId id, int flag) NN_NOEXCEPT
    {
        for (auto& redirection : m_RedirectionList)
        {
            if (redirection.Matches(id))
            {
                redirection.SetFlag(flag);
            }
        }
    }

    void LocationRedirector::EraseRedirection(ncm::ProgramId id) NN_NOEXCEPT
    {
        for (auto& redirection : m_RedirectionList)
        {
            if (redirection.Matches(id))
            {
                m_RedirectionList.erase(m_RedirectionList.iterator_to(redirection));
                delete &redirection;
                return;
            }
        }
    }
}}
