﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ヒープ共通処理の内部実装です。
 */

#include <nn/lmem/lmem_Common.h>

#include "lmem_DetailCommonHeap.h"
#include "lmem_DetailCommonImpl.h"

namespace nn { namespace lmem { namespace detail {

namespace {

uint32_t g_pFillVals[FillType_Max] =
{
    0xC3C3C3C3, // ヒープ作成時に埋める値
    0xF3F3F3F3, // メモリブロック確保時に埋める値
    0xD3D3D3D3, // メモリブロック解放時に埋める値
};

}   // unnamed namespace

/**
 * @brief ヒープヘッダの初期化を行います。
 */
void InitHeapHead(HeapHead* pOutHeader,
                  uint32_t signature,
                  void* pHeapStart,
                  void* pHeapEnd,
                  int option) NN_NOEXCEPT
{
    pOutHeader->signature = signature;

    pOutHeader->pHeapStart = pHeapStart;
    pOutHeader->pHeapEnd   = pHeapEnd;

    pOutHeader->attribute = 0;
    SetHeapOpt(pOutHeader, option);

#if defined(NN_DETAIL_HEAP_DEBUG)
    FillUnallocatedMemory(pOutHeader, pHeapStart, GetOffsetFromPtr(pHeapStart, pHeapEnd));
#endif
}

/**
 * @brief ヒープ共通の後始末を行います。
 */
void FinalizeHeap(HeapHead* pHeader) NN_NOEXCEPT
{
    NN_UNUSED(pHeader);
}


/**
 * @brief ヒープヘッダの情報を表示します。
 */
void DumpHeapHead(HeapHead const* pHeapHead) NN_NOEXCEPT
{
    NN_SDK_LOG("[Mem Library ");

    switch(pHeapHead->signature)
    {
    case MemExpHeapSignature: NN_SDK_LOG("Exp");   break;
    case MemFrameHeapSignature: NN_SDK_LOG("Frame"); break;
    case MemUnitHeapSignature: NN_SDK_LOG("Unit");  break;
    default:
        NN_SDK_ASSERT(false);
    }

    NN_SDK_LOG(" Heap]\n");

    // ヒープの管理領域（ヒープ共通ヘッダ）がヒープ内部にあるなら、それも含める
    if(reinterpret_cast<uintptr_t>(pHeapHead->pHeapStart) - sizeof(HeapHead) == reinterpret_cast<uintptr_t>(pHeapHead))
    {
        NN_SDK_LOG("    whole [%p - %p)\n", pHeapHead, pHeapHead->pHeapEnd);
    }
    else
    {
        NN_SDK_LOG("    whole [%p - %p)\n", pHeapHead->pHeapStart, pHeapHead->pHeapEnd);
    }
}

/* ========================================================================
    外部公開用関数
   ======================================================================== */

/*
 * @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリにセットする値をセットします。
 */
uint32_t SetHeapFillVal(int type, uint32_t value) NN_NOEXCEPT
{
    NN_SDK_ASSERT(type < FillType_Max);
    uint32_t oldVal = g_pFillVals[type];
    g_pFillVals[type] = value;
    return oldVal;
}

/*
 * @brief       ヒープの作成時やメモリブロックの確保・解放時にメモリにセットする値を取得します。
 */
uint32_t GetHeapFillVal(int type) NN_NOEXCEPT
{
    NN_SDK_ASSERT(type < FillType_Max);
    return g_pFillVals[type];
}

/**
    @brief      指定したアドレスがヒープに含まれているか調べます。
*/
bool HasAddress( HeapHandle heapHandle, void* address ) NN_NOEXCEPT
{
    // ヒープの管理領域（ヒープ共通ヘッダ）がヒープ内部にあるなら、それも含める
    // ヒープの管理領域がヒープ領域の先頭にある場合
    if(reinterpret_cast<uintptr_t>(heapHandle->pHeapStart) - sizeof(HeapHead) == reinterpret_cast<uintptr_t>(heapHandle))
    {
        return reinterpret_cast<uintptr_t>(heapHandle) <= reinterpret_cast<uintptr_t>(address) && reinterpret_cast<uintptr_t>(address) < reinterpret_cast<uintptr_t>(heapHandle->pHeapEnd);
    }
    // ヒープの管理領域がヒープ領域の末尾にある場合
    else if(reinterpret_cast<uintptr_t>(heapHandle) == reinterpret_cast<uintptr_t>(heapHandle->pHeapEnd))
    {
        return reinterpret_cast<uintptr_t>(heapHandle->pHeapStart) <= reinterpret_cast<uintptr_t>(address) && reinterpret_cast<uintptr_t>(address) < reinterpret_cast<uintptr_t>(heapHandle->pHeapEnd) + sizeof(HeapHead);
    }

    return reinterpret_cast<uintptr_t>(heapHandle->pHeapStart) <= reinterpret_cast<uintptr_t>(address) && reinterpret_cast<uintptr_t>(address) < reinterpret_cast<uintptr_t>(heapHandle->pHeapEnd);
}

/**
 * @brief       ヒープに割り当てられているメモリサイズ(ヘッダ部分も含む全体のメモリサイズ)を取得します。
 */
size_t GetHeapTotalSize(HeapHandle handle) NN_NOEXCEPT
{
    // ヒープの管理領域（ヒープ共通ヘッダ）がヒープ内部にあるなら、それも含めた値を返す
    if(HasAddress(handle, reinterpret_cast<void*>(handle)))
    {
        return static_cast<size_t>(reinterpret_cast<uintptr_t>(handle->pHeapEnd) - reinterpret_cast<uintptr_t>(handle->pHeapStart) + sizeof(HeapHead));
    }

    // ヒープの管理領域（ヒープ共通ヘッダ）がヒープ外部に確保されているなら、それを抜いた値を返す
    return static_cast<size_t>(reinterpret_cast<uintptr_t>(handle->pHeapEnd) - reinterpret_cast<uintptr_t>(handle->pHeapStart));
}

}}}
