﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#include <nn/htclow.h>
#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_IUserServiceObject.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/detail/htclow_UserServiceName.h>

namespace nn {
namespace htclow {

namespace
{
    const int NumberOfClients = 1;
    sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
    sf::ShimLibraryObjectHolder<detail::IUserServiceObject> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

    nn::os::SdkMutexType g_InitializeMutex = NN_OS_SDK_MUTEX_INITIALIZER();

    bool g_Initialized = false;
    nn::sf::SharedPointer<detail::IUserServiceObject> g_Service;

    // Utility Functions
    Result AttachClosedEvent(nn::os::SystemEvent* pEvent, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetClosedEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle), channel));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachEstablishedEvent(nn::os::SystemEvent* pEvent, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetEstablishedEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle), channel));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachReceiveReadyEvent(nn::os::SystemEvent* pEvent, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetReceiveReadyEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle), channel));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachSendReadyEvent(nn::os::SystemEvent* pEvent, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetSendReadyEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle), channel));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachSendCompleteEvent(nn::os::SystemEvent* pEvent, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetSendCompleteEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle), channel));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachHostConnectEvent(nn::os::SystemEvent* pEvent) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetHostConnectEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle)));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }

    Result AttachHostDisconnectEvent(nn::os::SystemEvent* pEvent) NN_NOEXCEPT
    {
        nn::sf::NativeHandle nativeHandle;
        NN_RESULT_DO(g_Service->GetHostDisconnectEvent(nn::sf::Out<nn::sf::NativeHandle>(&nativeHandle)));

        pEvent->AttachReadableHandle(nativeHandle.GetOsHandle(), nativeHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
        nativeHandle.Detach();

        NN_RESULT_SUCCESS;
    }
}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        return;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::htclow::detail::UserServiceName));
    g_Initialized = true;

    g_Service = g_Holder.GetObject();
}

namespace detail {

void InitializeForUserApiWith(nn::sf::SharedPointer<IUserServiceObject>&& manager) NN_NOEXCEPT
{
    std::lock_guard<decltype(g_InitializeMutex)> lock(g_InitializeMutex);
    if (g_Initialized)
    {
        NN_ABORT("This function cannot be called after Initialize().\n");
    }

    g_Holder.InitializeHolderDirectly(std::move(manager));
    g_Initialized = true;

    g_Service = g_Holder.GetObject();
}

}

nn::Result OpenChannel(const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);

    return g_Service->Open(channel);
}

void CloseChannel(const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);
    g_Service->Close(channel);
}

nn::Result ConnectChannel(const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);

    NN_RESULT_DO(g_Service->Connect(channel));

    nn::os::SystemEvent established;
    nn::os::SystemEvent closed;

    NN_RESULT_DO(AttachEstablishedEvent(&established, channel));
    NN_RESULT_DO(AttachClosedEvent(&closed, channel));

    // 処理の完了待ち
    if (nn::os::WaitAny(established.GetBase(), closed.GetBase()) == 1)
    {
        return ResultChannelClosed();
    }

    NN_RESULT_SUCCESS;
}

void ShutdownChannel(const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);

    if (g_Service->Shutdown(channel).IsFailure())
    {
        return;
    }

    // Closed になるのを待つ
    nn::os::SystemEvent closed;
    if (AttachClosedEvent(&closed, channel).IsFailure())
    {
        return;
    }
    closed.Wait();

    // 送信バッファが空になるのを待つ
    nn::os::SystemEvent sendComplete;
    if (AttachSendCompleteEvent(&sendComplete, channel).IsFailure())
    {
        return;
    }
    sendComplete.Wait();
}


nn::Result Send(size_t* pOutSize, const void* pBuffer, size_t bufferSize, const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pOutSize);
    NN_ABORT_UNLESS_NOT_NULL(pBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);

    nn::os::SystemEvent sendReady;
    nn::os::SystemEvent closed;

    NN_RESULT_DO(AttachSendReadyEvent(&sendReady, channel));
    NN_RESULT_DO(AttachClosedEvent(&closed, channel));

    // すべて書き込むまで Send を繰り返す
    size_t totalSendSize = 0;
    while (totalSendSize < bufferSize)
    {
        if (nn::os::WaitAny(sendReady.GetBase(), closed.GetBase()) == 1)
        {
            if (totalSendSize == 0)
            {
                // 送信サイズが 0 バイトの状態でチャネルが切断されたらエラー
                return ResultChannelClosed();
            }

            // 送信サイズが 1 バイト以上でチャネルが切断されたら、API 呼び出しは成功扱い
            break;
        }

        const auto buffer = nn::sf::InBuffer(reinterpret_cast<const char*>(pBuffer) + totalSendSize, bufferSize - totalSendSize);

        uint64_t sendSize;
        const auto result = g_Service->Send(&sendSize, buffer, channel);

        if (result <= ResultOutOfMemory())
        {
            // メモリ不足なら sendReady イベントの発火を待つ
            continue;
        }
        else if (result <= ResultChannelClosed() && totalSendSize > 0)
        {
            // 送信サイズが 1 バイト以上でチャネルが切断されたら、API 呼び出しは成功扱い
            break;
        }
        else if (result.IsFailure())
        {
            return result;
        }

        // sendSize <= bufferSize (int) なのでキャスト OK
        totalSendSize += static_cast<size_t>(sendSize);
    }

    NN_SDK_ASSERT(totalSendSize <= bufferSize);
    *pOutSize = static_cast<size_t>(totalSendSize);

    NN_RESULT_SUCCESS;
}

nn::Result Receive(size_t* pOutSize, void* pOutBuffer, size_t bufferSize, const ChannelType* pChannel) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pOutSize);
    NN_ABORT_UNLESS_NOT_NULL(pOutBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pChannel);

    const auto channel = detail::ConvertChannelType(*pChannel);

    nn::os::SystemEvent receiveReady;
    nn::os::SystemEvent closed;

    NN_RESULT_DO(AttachReceiveReadyEvent(&receiveReady, channel));
    NN_RESULT_DO(AttachClosedEvent(&closed, channel));

    // バッファがいっぱいになるまで Receive を繰り返す
    size_t totalReceiveSize = 0;
    while (totalReceiveSize < bufferSize)
    {
        if (nn::os::WaitAny(receiveReady.GetBase(), closed.GetBase()) == 1)
        {
            if (totalReceiveSize == 0)
            {
                // 受信サイズが 0 バイトの状態でチャネルが切断されたらエラー
                return ResultChannelClosed();
            }

            // 受信サイズが 1 バイト以上でチャネルが切断されたら、API 呼び出しは成功扱い
            break;
        }

        const auto outBuffer = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutBuffer) + totalReceiveSize, bufferSize - totalReceiveSize);

        uint64_t receiveSize;
        const auto result = g_Service->Receive(&receiveSize, outBuffer, channel);

        if (result <= ResultChannelReceiveBufferEmpty())
        {
            // 受信バッファが空なら、受信可能イベントを待つ
            continue;
        }
        else if (result <= ResultChannelClosed() && totalReceiveSize > 0)
        {
            // 受信サイズが 1 バイト以上でチャネルが切断されたら、API 呼び出しは成功扱い
            break;
        }
        else if (result.IsFailure())
        {
            return result;
        }

        // sendSize <= bufferSize (int) なのでキャスト OK
        totalReceiveSize += static_cast<size_t>(receiveSize);
    }

    NN_SDK_ASSERT(totalReceiveSize <= bufferSize);
    *pOutSize = static_cast<size_t>(totalReceiveSize);

    NN_RESULT_SUCCESS;
}

void BindHostConnectionEvent(nn::os::SystemEvent* pOutEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(AttachHostConnectEvent(pOutEvent));
}

void BindHostDisconnectionEvent(nn::os::SystemEvent* pOutEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_Initialized, "This function cannot be called before Initialize().\n");
    NN_ABORT_UNLESS_NOT_NULL(pOutEvent);

    NN_ABORT_UNLESS_RESULT_SUCCESS(AttachHostDisconnectEvent(pOutEvent));
}

}}
