﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nn;
using nn.sf;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn.fs
{
    [CppRefPath(@"<nn/fs/fs_File.h>")]
    [ExternalStruct(4, 4)]
    public struct ReadOption
    {
    }

    [CppRefPath(@"<nn/fs/fs_File.h>")]
    [ExternalStruct(4, 4)]
    public struct WriteOption
    {
    }

    [CppRefPath(@"<nn/fs/fs_Directory.h>")]
    [ExternalStruct(784, 8)]
    [LargeData]
    public struct DirectoryEntry
    {
    }
}

namespace nn.htcfs
{
    [CppRefPath(@"<nn/htcfs/detail/htcfs_IServiceObject.h>")]
    [LargeData]
    public struct Path
    {
        [FixedArray(769)]
        public char_t[] str;
    }

    [CppRefPath(@"<nn/htcfs/detail/htcfs_IServiceObject.h>")]
    public interface IFileSystem : IServiceObject
    {
        [MethodId(0)]
        Result GetEntryType(Out<std.int32_t> outType, Path path);

        [MethodId(1)]
        Result OpenFile(Out<IFile> outHandle, Path path, std.uint32_t mode);

        [MethodId(2)]
        Result CreateFile(Path path, std.int64_t size);

        [MethodId(3)]
        Result DeleteFile(Path path);

        [MethodId(4)]
        Result RenameFile(Path fromName, Path toName);

        [MethodId(5)]
        Result FileExists(Out<bool> outExists, Path path);

        [MethodId(6)]
        Result GetFileTimeStamp(Out<std.uint64_t> outCreateTime, Out<std.uint64_t> outAccessTime, Out<std.uint64_t> outModifyTime, Path path);

        [MethodId(7)]
        Result OpenDirectory(Out<IDirectory> outHandle, Path path, std.int32_t mode);

        [MethodId(8)]
        Result CreateDirectory(Path path);

        [MethodId(9)]
        Result DeleteDirectory(Path path, bool recursively);

        [MethodId(10)]
        Result DirectoryExists(Out<bool> outExists, Path path);

        [MethodId(11)]
        Result RenameDirectory(Path fromName, Path toName);
    }

    [CppRefPath(@"<nn/htcfs/detail/htcfs_IServiceObject.h>")]
    public interface IFile : IServiceObject
    {
        [MethodId(0)]
        Result GetPriorityForFile(Out<std.int32_t> outPriority);

        [MethodId(1)]
        Result SetPriorityForFile(std.int32_t priority);

        [MethodId(2)]
        // アプリがデバイスアドレス空間に直接ファイルをロードできるようにする (e.g. GPU メモリ空間) ために、NonSecureMapTransfer を使用
        Result ReadFile(Out<std.int64_t> outSize, [NonSecureMapTransfer] OutBuffer outBuffer, std.int64_t offset, nn.fs.ReadOption option);

        [MethodId(3)]
        // アプリがデバイスアドレス空間に直接ファイルをロードできるようにする (e.g. GPU メモリ空間) ために、NonSecureMapTransfer を使用
        Result WriteFile([NonSecureMapTransfer] nn.sf.InBuffer buffer, std.int64_t offset, nn.fs.WriteOption option);

        [MethodId(4)]
        Result FlushFile();

        [MethodId(5)]
        Result GetFileSize(Out<std.int64_t> outSize);

        [MethodId(6)]
        Result SetFileSize(std.int64_t size);
    }

    [CppRefPath(@"<nn/htcfs/detail/htcfs_IServiceObject.h>")]
    public interface IDirectory : IServiceObject
    {
        [MethodId(0)]
        Result GetPriorityForDirectory(Out<std.int32_t> outPriority);

        [MethodId(1)]
        Result SetPriorityForDirectory(std.int32_t priority);

        [MethodId(2)]
        Result GetEntryCount(Out<std.int64_t> outCount);

        [MethodId(3)]
        // SIGLO-10452 の議論より BufferTransferMode(BufferTransferMode.MapAlias) を使用
        Result ReadDirectory(Out<std.int64_t> outCount, [BufferTransferMode(BufferTransferMode.MapAlias)] OutArray<nn.fs.DirectoryEntry> entries);

    }
}
