﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>

#include <cstdlib>
#include <mutex>

#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>

#include <nn/os.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/detail/gc_AsicHandler.h>

namespace nn { namespace gc {
namespace detail {

nn::os::Tick g_Tick = nn::os::GetSystemTick();


// *** Instancelogger ***

#ifdef GC_DETAIL_LOG_ENABLE

int InstanceLogger::g_FunctionIndent[2] = {0, 0};

InstanceLogger::InstanceLogger(const char* functionName, const int lineNumber, const char* fileName, const char* targetInWord, const char* targetOutWord, const char* targetWord) NN_NOEXCEPT
{
    Reset(functionName, lineNumber, fileName);
    nn::util::Strlcpy(m_TargetInWord, targetInWord, sizeof(m_TargetInWord));
    nn::util::Strlcpy(m_TargetOutWord, targetOutWord, sizeof(m_TargetOutWord));
    nn::util::Strlcpy(m_TargetWord, targetWord, sizeof(m_TargetWord));
    PrintIn();
}

void InstanceLogger::Reset(const char* functionName, const int lineNumber, const char* fileName) NN_NOEXCEPT
{
    m_LineNumber = lineNumber;
    nn::util::Strlcpy(m_FunctionNameBuffer, functionName, sizeof(m_FunctionNameBuffer));
    nn::util::Strlcpy(m_FileNameBuffer, fileName, sizeof(m_FileNameBuffer));
    m_TargetInWord[0] = 0;
    m_TargetOutWord[0] = 0;
    m_TargetWord[0] = 0;
}

void InstanceLogger::PrintIn() NN_NOEXCEPT
{
    if(IsTargetWordSpecified())
    {
        NN_DETAIL_GC_DETAIL_LOG("@{in}: %s() - %s:%d\n", m_FunctionNameBuffer, m_FileNameBuffer, m_LineNumber);
    }
    else
    {
        NN_DETAIL_GC_DETAIL_LOG("@{%s}: <%s>: %s() - %s:%d\n", m_TargetInWord, m_TargetWord, m_FunctionNameBuffer, m_FileNameBuffer, m_LineNumber);
    }
    g_FunctionIndent[(int)(nn::gc::detail::IsGcThread())]++;
}

void InstanceLogger::PrintOut() NN_NOEXCEPT
{
    g_FunctionIndent[(int)(nn::gc::detail::IsGcThread())]--;
    if(IsTargetWordSpecified())
    {
        NN_DETAIL_GC_DETAIL_LOG("@{out}: %s() - %s:%d\n", m_FunctionNameBuffer, m_FileNameBuffer, m_LineNumber);
    }
    else
    {
        NN_DETAIL_GC_DETAIL_LOG("@{%s}: <%s>: %s() - %s:%d\n", m_TargetOutWord, m_TargetWord, m_FunctionNameBuffer, m_FileNameBuffer, m_LineNumber);
    }
}
#endif



// *** MemoryLogger ***

#ifdef GC_DETAIL_LOG_TO_MEMORY_ENABLE

namespace {
    static char LogBufferList[MemoryLogger::NumberOfRingBufferForLog][MemoryLogger::RingBufferForLogSize];
}

MemoryLogger::MemoryLogger() NN_NOEXCEPT : m_LogCounterMutex(true), m_LogMutex(true)
{
    m_LogCounter = 0;
    m_LogStartPosition = 0;

    for(int i = 0; i < NumberOfRingBufferForLog; i++)
    {
        m_RingBuffer[i].SetBuffer(LogBufferList[i], RingBufferForLogSize);
    }
}

MemoryLogger& MemoryLogger::GetInstance() NN_NOEXCEPT
{
    static MemoryLogger s_Instance;
    return s_Instance;
}

uint32_t MemoryLogger::GetCounter() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LogCounterMutex);
    return m_LogCounter++;
}

void MemoryLogger::AddLogImpl(char* buffer, size_t bufferLength) NN_NOEXCEPT
{
    // まずカウンタを取得
    uint32_t logId = GetCounter();

    // スレッドに対応するリングバッファを選択
    CharRingBuffer& ring = m_RingBuffer[this->GetBufferIndex()];

    // カウンタを挿入
    ring.Enqueue(reinterpret_cast<char*>(&logId), sizeof(uint32_t));

    // ログを挿入
    size_t length = strnlen(buffer, bufferLength);
    ring.Enqueue(buffer, length);

    // 終端文字を挿入
    ring.AddNullByteToLast();
}

void MemoryLogger::AddLog(char* buffer, size_t bufferLength) NN_NOEXCEPT
{
    // 「GC スレッドかそれ以外」でバッファをわけているので、複数のスレッドがこの関数を呼びうる場合はロックする
    bool isGcThread = nn::gc::detail::IsGcThread();
    if(isGcThread)
    {
        AddLogImpl(buffer, bufferLength);
    }
    else
    {
        std::lock_guard<nn::os::Mutex> lock(m_LogMutex);
        AddLogImpl(buffer, bufferLength);
    }
}

bool MemoryLogger::IsRingBufferForLogEmpty() NN_NOEXCEPT
{
    for (int i = 0; i < NumberOfRingBufferForLog; i++)
    {
        if(m_RingBuffer[i].GetLength() > 0)
        {
            return false;
        }
    }
    return true;
}

MemoryLogger::LogRemainFlag::LogRemainFlag() NN_NOEXCEPT
{
    nextId = 0;
    remainFlag = true;
}

void MemoryLogger::LogRemainFlag::SetNext(CharRingBuffer* ring) NN_NOEXCEPT
{
    nextId = 0;
    remainFlag = ! ring->IsEmpty();
    if(remainFlag)
    {
        nextId = GetLogId(ring);
    }
}

void MemoryLogger::OutputLog(UtilOutputCallbackFunctionPointer callback) NN_NOEXCEPT
{
    LogRemainFlag logRemain[NumberOfRingBufferForLog];

    // log data があるかないかを確認
    for (int i = 0; i < NumberOfRingBufferForLog; i++)
    {
        logRemain[i].SetNext(&(m_RingBuffer[i]));
    }


    char logBuffer[2 * 1024];
    while(IsRingBufferForLogEmpty() == false)
    {
        bool outputFlag = false;

        for (int i = 0; i < NumberOfRingBufferForLog; i++)
        {
            LogRemainFlag& lrf = logRemain[i];
            CharRingBuffer& ring = m_RingBuffer[i];

            // ログ保持状態から脱落していたら continue
            if(lrf.remainFlag == false)
            {
                continue;
            }

            // next data からリングバッファを選択
            if(lrf.nextId == m_LogStartPosition)
            {
                // ログを取得する
                size_t logBufferLength = ring.DequeueLine(logBuffer, sizeof(logBuffer));

                // TODO: DequeueLine で size_t length を取得してコールバック

                // ログを出力する
                callback(logBuffer, logBufferLength);

                // ログがリングバッファに残っているかのチェック
                lrf.SetNext(&ring);

                // ログ開始位置を進める
                m_LogStartPosition++;

                // while ループの最初から再開
                outputFlag = true;
                break;
            }
        }

        // 何も処理がなかったら抜ける
        if(outputFlag == false)
        {
            break;
        }
    }
}

void MemoryLogger::OutputToConsole(char* buffer, size_t bufferLength) NN_NOEXCEPT
{
    NN_UNUSED(bufferLength);
    NN_SDK_LOG("%s", buffer);
}

void MemoryLogger::ClearLog() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_LogCounterMutex);
    m_LogCounter = 0;
    m_LogStartPosition = 0;

    for (int i = 0; i < NumberOfRingBufferForLog; i++)
    {
        m_RingBuffer[i].Clear();
    }
}

uint32_t MemoryLogger::GetLogId(CharRingBuffer* ring) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(ring->GetLength() >= sizeof(uint32_t));
    uint32_t logId = 0;
    ring->Dequeue(reinterpret_cast<char*>(&logId), sizeof(uint32_t));
    return logId;
}

#endif


// *** その他の関数 ***

void PrintDebugArray(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            GC_NN_SDK_LOG("\n");
        }

        GC_NN_SDK_LOG("%02X ", buffer[i]);

        if(i % 8 == 7)
        {
            GC_NN_SDK_LOG(" ");
        }
    }
    GC_NN_SDK_LOG("\n");
}

bool IsGcThread() NN_NOEXCEPT
{
    const char* currentThreadNamePointer = nn::os::GetThreadNamePointer(nn::os::GetCurrentThread());
    return currentThreadNamePointer == AsicHandler::GetInstance().ThreadNamePtr;
}


} } }
