﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Thread.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_GeneralIo.h>

#include <nn/gpio/gpio_PadAccessorDev.h>

#include <nn/pcv/pcv.h>

namespace nn { namespace gc {
namespace detail {

GeneralIo::GeneralIo() NN_NOEXCEPT
{

}

GeneralIo& GeneralIo::GetInstance() NN_NOEXCEPT
{
    static GeneralIo s_Instance;
    return s_Instance;
}

void GeneralIo::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 初期化
    nn::gpio::Initialize();
    nn::pcv::Initialize();

    // ** GPIO
    // リセット端子
    nn::gpio::OpenSession(&m_GcResetPadSession, nn::gpio::GpioPadName_GameCardReset);
    nn::gpio::SetDirection(&m_GcResetPadSession, nn::gpio::Direction_Output);

    // カード挿入検知端子
    nn::gpio::OpenSession(&m_GcDetectPadSession, nn::gpio::GpioPadName_GameCardCd);

    nn::gpio::SetDirection(&m_GcDetectPadSession, nn::gpio::Direction_Input);
    nn::gpio::SetDebounceEnabled(&m_GcDetectPadSession, false);
    // 割り込み設定
    nn::gpio::SetInterruptMode(&m_GcDetectPadSession, nn::gpio::InterruptMode_AnyEdge);
    nn::gpio::ClearInterruptStatus(&m_GcDetectPadSession);
    nn::gpio::BindInterrupt(&m_DetectEvent, &m_GcDetectPadSession);
    nn::os::ClearSystemEvent(&m_DetectEvent);

    // 割り込みを有効にする
    nn::gpio::SetInterruptEnable(&m_GcDetectPadSession, true);
}


void GeneralIo::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 割り込み設定の解除
    nn::gpio::SetInterruptEnable(&m_GcDetectPadSession, false);
    nn::gpio::UnbindInterrupt(&m_GcDetectPadSession);

    nn::gpio::CloseSession(&m_GcResetPadSession);
    nn::gpio::CloseSession(&m_GcDetectPadSession);
    nn::gpio::Finalize();

    nn::pcv::Finalize();
}

nn::Result GeneralIo::SetGpioResetPin(const bool isOn) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::gpio::SetValue(&m_GcResetPadSession, static_cast<nn::gpio::GpioValue>(isOn));
    NN_RESULT_SUCCESS;
}

bool GeneralIo::GetGpioDetectPin() NN_NOEXCEPT
{
    return (nn::gpio::GetValue(&m_GcDetectPadSession) == nn::gpio::GpioValue_Low);
}

void GeneralIo::ClearGpioDetectEvent() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    // 割り込みステータスをクリアする
    nn::gpio::ClearInterruptStatus(&m_GcDetectPadSession);

    // イベントのクリア
    nn::os::ClearSystemEvent(&m_DetectEvent);

    // 割り込み許可状態に戻す
    nn::gpio::SetInterruptEnable(&m_GcDetectPadSession, true);
}

bool GeneralIo::IsGpioDetectWakeEventActive() NN_NOEXCEPT
{
    return nn::gpio::IsWakeEventActive(nn::gpio::GpioPadName_GameCardCd);
}

nn::Result GeneralIo::WaitGpioDetectPin(const bool isOn, const bool isOldPlatform) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_DETAIL_LOG("Start waiting DET signal set to %d...\n", isOn);
    if(isOldPlatform)
    {
        NN_DETAIL_GC_ERR_LOG("Warning: old platform GPIO setting is not supported\n");
    }

    // 割り込みを待つ
    while(NN_STATIC_CONDITION(true))
    {
        // 割り込みを有効にする
        ClearGpioDetectEvent();

        // 割り込み待ち
        nn::os::WaitSystemEvent(&m_DetectEvent);

        // 割り込み時の値を保持
        auto targetPinValue = GetGpioDetectPin();

        // 所定の値だったら抜ける
        if(targetPinValue == isOn)
        {
            break;
        }
    }

    NN_DETAIL_GC_DETAIL_LOG("DET signal %d\n", isOn);
    NN_RESULT_SUCCESS;
}

nn::Result GeneralIo::SetGcPower(const bool isOn) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    if(isOn == true)
    {
        NN_RESULT_DO(SetGcPowerImpl(true, BusPower_3_1V));
        NN_RESULT_DO(SetGcPowerImpl(true, BusPower_1_8V));
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(GcPowerOnWaitUsec) );
    }
    else
    {
        NN_RESULT_DO(SetGcPowerImpl(false, BusPower_1_8V));
        NN_RESULT_DO(SetGcPowerImpl(false, BusPower_3_1V));
        nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(GcPowerOffWaitUsec) );
    }
    NN_RESULT_SUCCESS;
}

nn::Result GeneralIo::SetGcPowerImpl(const bool isOn, const BusPower power) NN_NOEXCEPT
{
    NN_DETAIL_GC_DEBUG_LOG("target %s: power %s\n", power == BusPower_1_8V ? "1.8V" : "3.1V", isOn ? "on" : "off");

    nn::Result result = ResultSuccess();

    switch(power)
    {
    case BusPower_1_8V:
        {
            result = nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo5, isOn);
        }
        break;
    case BusPower_3_1V:
        {
            result = nn::pcv::SetVoltageEnabled(nn::pcv::PowerDomain_Max77620_Ldo3, isOn);
        }
        break;
    default:
       {
           NN_UNEXPECTED_DEFAULT;
       }
    }

    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardGeneralIoSetVoltageFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result GeneralIo::ReleaseAsicReset() NN_NOEXCEPT
{
    nn::Result result = SetGpioResetPin(true);

    // 10 サイクル待ち：100MHz では 1 サイクル 10ns, 適当に 1us 待つことにする
    static const int ResetWaitTimeUs = 1;
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(ResetWaitTimeUs) );

    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardGeneralIoReleaseAsicResetFailure();
    }
    NN_RESULT_SUCCESS;
}

nn::Result GeneralIo::HoldAsicReset() NN_NOEXCEPT
{
    nn::Result result = SetGpioResetPin(false);

    // 確実にリセットを行う
    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(GcPostResetWaitTimeUsec) );

    if(result.IsFailure())
    {
        return nn::fs::ResultGameCardGeneralIoHoldAsicResetFailure();
    }
    NN_RESULT_SUCCESS;
}


void GeneralIo::SetDetectPinStateForTest(bool isOn) NN_NOEXCEPT
{

}

} } }
