﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>
#include <nn/gc/detail/gc_DeviceDetector.h>

namespace nn { namespace gc {
namespace detail {

#define NN_DETAIL_GC_RESULT_THROW_IF_REMOVED if(IsRemoved()) { return nn::fs::ResultGameCardCardNotInserted(); }
#define NN_DETAIL_GC_RESULT_THROW_UNLESS_ACTIVATED  if(StateMachine::GetInstance().IsCardActivated() == false) { return nn::fs::ResultGameCardCardNotActivated(); }

// *** 外から依頼されうる仕事

nn::Result AsicHandlerProcedure::InitializeAsicImpl() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    StateMachine::StateMachineForAsicHandler& stateManager = StateMachine::GetInstanceForAsicHandler();
    AsicOperation& asicOp = AsicOperation::GetInstance();

    // （Reset 時にやっているはずだが、念のため鍵情報をクリア）
    GcCrypto::GetInstance().ClearKeys();

    // Reset を解除して sdmmc を active にする
    NN_DETAIL_GC_LOG("[GC] Released reset pin for GC ASIC\n");
    NN_RESULT_DO( ActivateAsicIo() );

    // FW を送る
    NN_DETAIL_GC_DETAIL_LOG("Sending firmware ...\n");
    NN_RESULT_DO( asicOp.SendFirmware() );

    // 証明書を交換し、署名が正しいか検証する
    NN_DETAIL_GC_DETAIL_LOG("verifying certificate ...\n");
    NN_RESULT_DO( VerifyCertificateMutually() );

    // 公開鍵を作る
    NN_DETAIL_GC_DETAIL_LOG("generating common key ...\n");
    NN_RESULT_DO( GenerateCommonKey() );

    // 相互認証を行う
    NN_DETAIL_GC_DETAIL_LOG("authenticating ...\n");
    NN_RESULT_DO( AuthenticateMutually() );

    // ブリッジASICをセキュアモードへ移行させる
    NN_DETAIL_GC_DETAIL_LOG("changing to secure mode ...\n");
    NN_RESULT_DO( asicOp.ChangeModeToSecure() );

    // 状態遷移：セキュアモード移行
    NN_RESULT_DO( stateManager.TransitStateWithSecureAsic() );
    NN_DETAIL_GC_LOG("[GC] changed to secure mode\n");

    // 鍵交換をしておく
    NN_DETAIL_GC_LOG("[GC] exchanging random values for key update ...\n");
    NN_RESULT_DO( ExchangeRandomValues() );

    NN_DETAIL_GC_LOG("[GC] asic initialized\n");
    NN_RESULT_SUCCESS;
}

// *** internal API

nn::Result AsicHandlerProcedure::ActivateAsicIo() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    DataIo& dataIo = DataIo::GetInstance();
    GeneralIo& generalIo = GeneralIo::GetInstance();
    NN_DETAIL_GC_DETAIL_LOG("Activate ASIC IO\n");
    NN_DETAIL_GC_DETAIL_LOG("Release ASIC reset\n");
    NN_RESULT_DO( generalIo.ReleaseAsicReset() );

    // DataIo 内で sdmmc チューニングが行われる
    NN_RESULT_DO( dataIo.Activate() );

    // 「Boot中」のフラグが立つまでポーリングする
    BitPack32 deviceStatus;
    GcDeviceStatusAsicError statusAsicError = GcDeviceStatusAsicError_None;
    static const int maxCount = GcPostReleaseResetWaitTimeMaxNsec / GcPostReleaseResetWaitTimeUnitNsec;
    for(int count=0; count < maxCount; count++)
    {
        NN_DETAIL_GC_RESULT_DO_LOG_RETHROW( dataIo.GetDeviceStatus(&deviceStatus), nn::fs::ResultGameCardActivateAsicFailure() );
        statusAsicError = (GcDeviceStatusAsicError)deviceStatus.Get<AsicDeviceStatus::asicError>();
        NN_DETAIL_GC_DETAIL_LOG("device status -> asic err: 0x%02X\n", statusAsicError);
        if(statusAsicError == GcDeviceStatusAsicError_Boot)
        {
            NN_RESULT_DO( StateMachine::GetInstanceForAsicHandler().TransitStateWithResetAsic() );
            NN_RESULT_SUCCESS;
        }
        else if(statusAsicError > GcDeviceStatusAsicError_Boot)
        {
            return nn::fs::ResultGameCardAsicBootFailure();
        }
        nn::os::SleepThread( nn::TimeSpan::FromNanoSeconds(GcPostReleaseResetWaitTimeUnitNsec) );
    }
    return nn::fs::ResultGameCardInitializeAsicTimeOut();
}

nn::Result AsicHandlerProcedure::DeactivateAsicIo() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    NN_DETAIL_GC_DETAIL_LOG("Deactivate ASIC IO\n");
    DataIo::GetInstance().Deactivate();
    GeneralIo& generalIo = GeneralIo::GetInstance();
    NN_DETAIL_GC_DETAIL_LOG("Hold ASIC reset\n");
    NN_RESULT_DO( generalIo.HoldAsicReset() );

    NN_RESULT_DO( generalIo.SetGcPowerOff() );
    StateMachine::GetInstanceForAsicHandler().ResetState();
    NN_RESULT_SUCCESS;
}


// 乱数交換と鍵更新
nn::Result AsicHandlerProcedure::ExchangeRandomValues() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();
    StateMachine::StateMachineForAsicHandler& stateManager = StateMachine::GetInstanceForAsicHandler();

    char rndToSendBuffer[GcRandomValueSize];
    char rndToReceiveBuffer[GcRandomValueSize];

    // 乱数RND5を生成
    NN_RESULT_DO(gcCrypto.GenerateRandomValue(rndToSendBuffer, sizeof(rndToSendBuffer)));

    // 乱数RND5は1byte少ないので、送信の際に1byte切り捨てるようにする
    size_t rndToSendBufferLength = GcRandomValueForKeyUpdateSocSize;

    // 乱数交換と乱数のセット
    NN_RESULT_DO( asicOp.ExchangeRandomValuesInSecureMode(rndToReceiveBuffer, sizeof(rndToReceiveBuffer), rndToSendBuffer, rndToSendBufferLength) );
    gcCrypto.SetRandomValuesForKeyUpdate(rndToSendBuffer, rndToSendBufferLength, rndToReceiveBuffer, sizeof(rndToReceiveBuffer));

    // 状態をセット
    stateManager.SetKeyUpdateReady(true);

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerProcedure::VerifyCertificateMutually() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();

    // ブリッジASICから証明書（公開鍵に署名がついている）を受け取る
    NN_DETAIL_GC_DETAIL_LOG("Receiving certificate ...\n");
    NN_RESULT_DO( asicOp.ReceiveCertificate(GcCrypto::g_AsicCertBuffer, sizeof(GcCrypto::g_AsicCertBuffer)) );

    // 受け取った証明書を検証し、公開鍵を得る
    NN_DETAIL_GC_DETAIL_LOG("Verifying certificate and get public key ...\n");
    NN_RESULT_DO( gcCrypto.VerifyCertificateAndGetPublicKey(GcCrypto::g_AsicCertBuffer, sizeof(GcCrypto::g_AsicCertBuffer)) );

    // SoC の証明書を送る
    NN_DETAIL_GC_DETAIL_LOG("Sending certificate ...\n");
    NN_RESULT_DO( asicOp.SendSocCertificate(EmbeddedDataHolder::g_EmmcEmbeddedSocCertificate, sizeof(EmbeddedDataHolder::g_EmmcEmbeddedSocCertificate)) );

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerProcedure::GenerateCommonKey() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();
    StateMachine::StateMachineForAsicHandler& stateManager = StateMachine::GetInstanceForAsicHandler();

    // 暗号化された乱数RND1をブリッジASICから受け取る
    char rnd1EncBuffer[GcCrypto::GcRsaBlockSize];
    NN_DETAIL_GC_DETAIL_LOG("Receiving encrypted RND1 ...\n");
    NN_RESULT_DO( asicOp.ReceiveRandomValue(rnd1EncBuffer, sizeof(rnd1EncBuffer)) );

    // SoC側で乱数RND2を作る
    char rnd2Buffer[GcRandomValueSize];
    NN_DETAIL_GC_DETAIL_LOG("Generating RND2 ...\n");
    NN_RESULT_DO( gcCrypto.GenerateRandomValue(rnd2Buffer, sizeof(rnd2Buffer)) );

    {
        // ブリッジASICの公開鍵で乱数RND2を暗号化する
        NN_DETAIL_GC_DETAIL_LOG("Encrypting RND2 ...\n");
        char rnd2EncBuffer[GcCrypto::GcRsaBlockSize];
        gcCrypto.EncryptWithRsaOaep(rnd2EncBuffer, sizeof(rnd2EncBuffer), rnd2Buffer, sizeof(rnd2Buffer));

        // 暗号化した乱数RND2をブリッジASICへ送信する
        NN_DETAIL_GC_DETAIL_LOG("Sending encrypted RND2 ...\n");
        NN_RESULT_DO( asicOp.SendRandomValue(rnd2EncBuffer, sizeof(rnd2EncBuffer)) );
    }

    // 以降で RSA OAEP の復号化処理、共通鍵の生成について、SoCとブリッジASICで並列に処理：パイプライン処理の方がよいかは要検討

    // ダミーの乱数を作っておく（正常時も作っておかないと、異常時のみ時間がかかってしまいヒントになる）
    char rndDummyBuffer[GcRandomValueSize];
    NN_RESULT_DO( gcCrypto.GenerateRandomValue(rndDummyBuffer, sizeof(rndDummyBuffer)) );

    // 受け取った乱数RND1を復号化して取り出す
    char rnd1Buffer[GcRandomValueSize];
    NN_DETAIL_GC_DETAIL_LOG("Decrypting encrypted RND1 ...\n");
    {
        size_t decRnd1Length = 0;
        nn::Result result = gcCrypto.DecryptWithRsaOaep(&decRnd1Length, rnd1Buffer, sizeof(rnd1Buffer), rnd1EncBuffer, sizeof(rnd1EncBuffer));
        if(result.IsFailure())
        {
            memcpy(rnd1Buffer, rndDummyBuffer, sizeof(rndDummyBuffer));
        }
        if(decRnd1Length != GcRandomValueSize)
        {
            NN_DETAIL_GC_ERR_LOG("Decrypted by RSA-OAEP size != 32 Byte (%zu byte)\n", decRnd1Length);
            memcpy(rnd1Buffer, rndDummyBuffer, sizeof(rndDummyBuffer));
        }
    }
    // 復号化した値を使って共通鍵を作る
    NN_DETAIL_GC_DETAIL_LOG("Generating common key ...\n");
    gcCrypto.GenerateCommonKeyFromRandomValues(rnd1Buffer, sizeof(rnd1Buffer), rnd2Buffer, sizeof(rnd2Buffer));

    // 今後は共通鍵を使って暗号化を行う
    stateManager.SetCommonKeyReady(true);

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerProcedure::AuthenticateMutually() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();

    char rndBuffer[GcRandomValueSize];
    char hashBuffer[GcCrypto::GcSha256Length];

    // ** 現時点の送受信データは共通鍵で AES-CBC 暗号化が必要だが、この暗号化・復号化は DataIo 内で処理している

    // * ブリッジASICからのチャレンジ
    // 乱数RND3をブリッジASICから受け取る
    NN_DETAIL_GC_DETAIL_LOG("Receiving a challenge from ASIC ...\n");
    NN_RESULT_DO( asicOp.ReceiveDeviceChallenge(rndBuffer, sizeof(rndBuffer)) );

    // 受け取った乱数RND3のハッシュを求め、共通鍵で暗号化の上で送り返す
    NN_DETAIL_GC_DETAIL_LOG("Calculating the response for the challenge ...\n");
    gcCrypto.CalcSha256(hashBuffer, sizeof(hashBuffer), rndBuffer, sizeof(rndBuffer));
    NN_DETAIL_GC_DETAIL_LOG("Responding the challenge from ASIC ...\n");
    NN_RESULT_DO( asicOp.RespondDeviceChallenge(hashBuffer, sizeof(hashBuffer)) );

    // * SoC からのチャレンジ
    // 乱数RND4を作る
    NN_DETAIL_GC_DETAIL_LOG("Generating a challenge ...\n");
    NN_RESULT_DO( gcCrypto.GenerateRandomValue(rndBuffer, sizeof(rndBuffer)) );
    // ブリッジASICへ送る；ブリッジASICの処理が遅いという予想のもと、ハッシュを計算する前に送る
    NN_DETAIL_GC_DETAIL_LOG("Sending a challenge to ASIC ...\n");
    NN_RESULT_DO( asicOp.SendHostChallenge(rndBuffer, sizeof(rndBuffer)) );

    // 答えのハッシュを計算する
    NN_DETAIL_GC_DETAIL_LOG("Calculating the response ...\n");
    gcCrypto.CalcSha256(hashBuffer, sizeof(hashBuffer), rndBuffer, sizeof(rndBuffer));

    // 乱数RND4のハッシュをブリッジASICから受け取る
    char receivedHashBuffer[GcCrypto::GcSha256Length];
    NN_DETAIL_GC_DETAIL_LOG("Receiving the response ...\n");
    NN_RESULT_DO( asicOp.ReceiveChallengeResponse(receivedHashBuffer, sizeof(receivedHashBuffer)) );

    // 検証する
    if ( CompareAllMemory(hashBuffer, receivedHashBuffer, GcCrypto::GcSha256Length) != 0 )
    {
        NN_DETAIL_GC_DETAIL_LOG("Challenge and response failed.\n");

        GC_DETAIL_TEST_BREAK_RESULT;
        return nn::fs::ResultGameCardChallengeAndResponseFailure();
    }

    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerProcedure::UpdateKey() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    // そもそも鍵更新できる状態でなければ Failure
    StateMachine& state = StateMachine::GetInstance();
    if( state.IsKeyUpdateReady() == false )
    {
        return nn::fs::ResultGameCardKeySourceNotFound();
    }

    // 鍵更新を行う
    DataIo& dataIo = DataIo::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();
    NN_RESULT_DO( dataIo.UpdateKey() );
    NN_RESULT_DO( dataIo.FinishOperation() );
    gcCrypto.UpdateSessionKey();

    // 乱数交換を行う
    AsicOperation& asicOp = AsicOperation::GetInstance();
    char rndToSendBuffer[GcRandomValueSize];
    char rndToReceiveBuffer[GcRandomValueSize];

    // 乱数RND5を生成
    NN_RESULT_DO( gcCrypto.GenerateRandomValue(rndToSendBuffer, sizeof(rndToSendBuffer)) );
    // 乱数RND5は1byte少ないので、送信の際に1byte切り捨てるようにする
    size_t rndToSendBufferLength = GcRandomValueForKeyUpdateSocSize;
    // 乱数交換と乱数のセット
    NN_RESULT_DO( asicOp.ExchangeRandomValuesInSecureMode(rndToReceiveBuffer, sizeof(rndToReceiveBuffer), rndToSendBuffer, rndToSendBufferLength) );
    gcCrypto.SetRandomValuesForKeyUpdate(rndToSendBuffer, rndToSendBufferLength, rndToReceiveBuffer, sizeof(rndToReceiveBuffer));

    NN_RESULT_SUCCESS;
}

} } }
