﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_GameCard.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GeneralIo.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>
#include <nn/gc/detail/gc_DeviceDetector.h>

// AsicHandlerCore の中でカードに関係する処理関数を集めたファイル

namespace nn { namespace gc {
namespace detail {

// *** 外から依頼されうる仕事


// 引数なしの ActivateCard はリトライなし
nn::Result AsicHandlerCore::ActivateCard() NN_NOEXCEPT
{
    return ActivateCard(false);
}

// 失敗時に ASIC を初期化する為、呼び出し側で InvalidateAsicIfResultFailure を呼ぶ
nn::Result AsicHandlerCore::ActivateCard(bool isRetryNeeded) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    AsicRegister& asicReg = AsicRegister::GetInstance();
    StateMachine::StateMachineForAsicHandler& stateManager = StateMachine::GetInstanceForAsicHandler();

    // 念のためステートを見る
    if(StateMachine::GetInstance().IsCardActivated())
    {
        NN_RESULT_SUCCESS;
    }

    // カード抜去に関するクリティカルセクション
    {
        NN_DETAIL_GC_CANCEL_SDMMC_LOCK_GUARD;

        // sdmmc 転送中断が確実に処理されるよう (カード状態 == 挿入 && m_IsCardBusEnabled == true) を確定してから次に進む
        //（この時点以降のカード抜去は（もしかしたら mutex の lock 獲得を待たされたあと）、m_IsCardBusEnabled が true なので必ず sdmmc 転送中断となる）
        if( StateMachine::GetInstance().IsCardInserted() == false )
        {
            NN_DETAIL_GC_WARNING_LOG("Activate card attempted when card is not inserted.\n");
            return nn::fs::ResultGameCardCardNotInserted();
        }

        // これ以降に抜去があると ResultCardNotInserted が返るようになる（m_IsRemoved はセッション単位の抜去状態）
        m_IsRemovedAfterActivate = false;

        // これ以降に抜去があると sdmmc 転送中断をするようになる
        m_SessionInfo.m_IsCardBusEnabled = true;
    }

    // これ以降で失敗すると強制初期化フラグを立てる
    bool isActivateCardCompleted = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (isActivateCardCompleted == false)
        {
            StateMachine::GetInstanceForAsicHandler().SetForceResetFlag();
        }
    };

    // カードバス有効化
    nn::Result result = ExecuteOperationWithRetry(&AsicHandlerCore::EnableCardBus);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "Enable card bus failed\n");
    NN_RESULT_DO(result);

    // カードヘッダの試し読み
    {
        char cardHeaderBuffer[GcPageSize];
        int maxReadCardHeaderCount = isRetryNeeded ? GcSendCommandMaxCount : 1;
        for (int retryCount = 0; retryCount < maxReadCardHeaderCount; retryCount++)
        {
            result = asicOp.SendCardReadPageWithNotAlignedBuffer(cardHeaderBuffer, sizeof(cardHeaderBuffer), 0, 1);
            if (result.IsSuccess())
            {
                break;
            }
            NN_DETAIL_GC_WARNING_LOG("Read CardHeader first failed (%d).\n", retryCount);
        }
        if(result.IsFailure())
        {
            if( StateMachine::GetInstance().IsCardInserted() == false )
            {
                NN_DETAIL_GC_WARNING_LOG("Activate card attempted when card is not inserted.\n");
                return nn::fs::ResultGameCardCardNotInserted();
            }
            NN_DETAIL_GC_ERR_LOG("Read CardHeader first retry out\n");
            // カードヘッダの試し読みに失敗した時はそれ用の Result にする
            return nn::fs::ResultGameCardCardHeaderReadFailure();
        }

        GcCrypto::GetInstance().CalcSha256(m_SessionInfo.m_CardImageHash, sizeof(m_SessionInfo.m_CardImageHash),
            cardHeaderBuffer + (sizeof(cardHeaderBuffer) - sizeof(CardHeader)), sizeof(CardHeader));
        NN_DETAIL_GC_DEBUG_MEMDUMP(cardHeaderBuffer, sizeof(cardHeaderBuffer), "Read Card Header Success\n");
    }

    // カードヘッダの取得
    result = ExecuteOperationWithRetry(&AsicHandlerCore::GetCardHeader);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "Failed to get card header\n");
    NN_RESULT_DO(result);

    // 状態を更新
    NN_RESULT_DO( stateManager.TransitStateWithActivateGameCard() );

    // レイテンシチェックの値をレジスタに設定
    {
        asicReg.SetLatencyCheckValue(0x64); // TODO: TEMP:
    }
    // レジスタ読み
    NN_RESULT_DO( asicOp.ReadRegister() );

    // ID読み
    NN_RESULT_DO( asicOp.SendCardReadId2(&(m_SessionInfo.m_CardId2)) );
    NN_RESULT_DO( asicOp.SendCardReadId3(&(m_SessionInfo.m_CardId3)) );

    isActivateCardCompleted = true;
    NN_RESULT_SUCCESS;
}

void AsicHandlerCore::DeactivateCard() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // カードアクセス可能状態をメモ
    StateMachine& state = StateMachine::GetInstance();
    bool IsLastCardActivated = state.IsCardActivated();

    // カードが Activated 状態だったら、リセットの必要がある：リセット要求が来ていた場合もリセット
    if(IsLastCardActivated || state.IsForceReset())
    {
        this->Reset();

        // カード電源 OFF
        GeneralIo::GetInstance().SetGcPowerOff();
    }
}

nn::Result AsicHandlerCore::SetCardToSecureMode() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 念のためステートを見る
    if(StateMachine::GetInstance().IsCardSecure())
    {
        NN_RESULT_SUCCESS;
    }

    // ゲームカード側をセキュアモードへ移行させる
    NN_RESULT_DO( ExecuteOperationWithRetry(&AsicHandlerCore::ChangeGcModeToSecure) );
    NN_RESULT_DO( StateMachine::GetInstanceForAsicHandler().TransitStateWithSecureGameCard() );

    // 一応ドライバ側でも ID をチェックしておく
    if( (CompareAllMemory(m_pCardId1, &(m_SessionInfo.m_CardSecurityInformation.id1), sizeof(CardId1)) != 0) ||
        (CompareAllMemory(&(m_SessionInfo.m_CardId2), &(m_SessionInfo.m_CardSecurityInformation.id2), sizeof(CardId2)) != 0) )
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        ;
#else
        return nn::fs::ResultGameCardCardIdMismatch();
#endif
    }

    NN_RESULT_SUCCESS;
}

// GameCard の情報を控えた後、カード電源 OFF + ASIC Reboot
void AsicHandlerCore::EvacuateCardInfoAndInitializeAsicIfNeeded() NN_NOEXCEPT
{
    StateMachine& state = StateMachine::GetInstance();
    // カードの状態・Device ID を控える（ここでしか値が入らない）
    m_CardInfoBeforeSleep.isInserted = state.IsCardInserted();
    m_CardInfoBeforeSleep.cardMode = state.GetGameCardMode();

    memset(m_CardInfoBeforeSleep.cardHeader, 0, sizeof(m_CardInfoBeforeSleep.cardHeader));
    memset(m_CardInfoBeforeSleep.deviceId, 0, sizeof(m_CardInfoBeforeSleep.deviceId));
    if(state.IsCardActivated() && IsRemoved() == false)
    {
        if(m_CardInfoBeforeSleep.cardMode >= GameCardMode_Normal)
        {
            memcpy(m_CardInfoBeforeSleep.cardHeader, m_SessionInfo.m_CardHeaderArray, sizeof(CardHeader));
        }
        if(m_CardInfoBeforeSleep.cardMode == GameCardMode_Secure)
        {
            GetCardDeviceId(m_CardInfoBeforeSleep.deviceId, sizeof(m_CardInfoBeforeSleep.deviceId));
        }
    }
    // Deactivate（＋カード電源 OFF）
    DeactivateCard();

    // セッション構築
    InitializeAsicIfNeeded();

}

// EvacuateCardInfoAndInitializeAsicIfNeeded で控えた情報と比較しつつゲームカードの状態を復元
nn::Result AsicHandlerCore::ResumeCardStateAndCheckCardInfo(bool isRetryNeeded) NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        std::memset(&m_CardInfoBeforeSleep, 0, sizeof(m_CardInfoBeforeSleep));
    };

    // ノーマルの処理
    if(m_CardInfoBeforeSleep.cardMode >= GameCardMode_Normal)
    {
        NN_RESULT_DO(ActivateCard(isRetryNeeded));

        // カードヘッダチェック
        NN_DETAIL_GC_LOG("Check if card header was changed ...\n");
        if(CompareAllMemory(m_CardInfoBeforeSleep.cardHeader, m_SessionInfo.m_CardHeaderArray, sizeof(m_CardInfoBeforeSleep.cardHeader)) != 0)
        {
            NN_DETAIL_GC_WARNING_LOG("card header changed during sleep!\n");
            NN_DETAIL_GC_LOG("expect:\n");
            NN_DETAIL_GC_DEBUG_MEMDUMP(m_CardInfoBeforeSleep.cardHeader, sizeof(m_CardInfoBeforeSleep.cardHeader));
            NN_DETAIL_GC_LOG("actual:\n");
            NN_DETAIL_GC_DEBUG_MEMDUMP(m_SessionInfo.m_CardHeaderArray, sizeof(m_SessionInfo.m_CardHeaderArray));
            DeactivateCard();
            // 一致しない場合はカードが一瞬抜けたことにする
            return nn::fs::ResultGameCardCardNotInserted();
        }
    }

    // セキュアの処理
    if(m_CardInfoBeforeSleep.cardMode == GameCardMode_Secure)
    {
        NN_RESULT_DO(SetCardToSecureMode());

        // Device ID チェック
        NN_DETAIL_GC_LOG("Check if device ID was changed ...\n");
        char deviceId[CardDeviceIdLength];
        GetCardDeviceId(deviceId, sizeof(deviceId));
        if(CompareAllMemory(m_CardInfoBeforeSleep.deviceId, deviceId, sizeof(m_CardInfoBeforeSleep.deviceId)) != 0)
        {
            NN_DETAIL_GC_WARNING_LOG("device ID changed during sleep!\n");
            DeactivateCard();
            // 一致しない場合はカードが一瞬抜けたことにする
            return nn::fs::ResultGameCardCardNotInserted();
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::PutToSleep() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // 念のためステートを見る
    if(IsAsleep())
    {
        NN_RESULT_SUCCESS;
    }

    // 現状の保存
    EvacuateCardInfoAndInitializeAsicIfNeeded();

    // ASIC のスリープ
    DataIo::GetInstance().PutGcAsicToSleep();

    // sdmmc のスリープ
    DataIo::GetInstance().PutSdmmcToSleep();

    // DeviceDetector を凍結（完了までブロックされる）：カードバスを有効化していないので、抜去があっても sdmmc 中断は起きない
    DeviceDetector::GetInstance().Pause();

    // 状態を設定
    m_SessionInfo.m_IsAsleep = true;

    // カード電源を切るタイミングでカウンタをリセット
    ClearSessionInfoForErrorReport();
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::Awaken() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    // Awaken に入った回数自体を記録
    m_TotalAsicInfo.awakenCount++;

    // 念のためステートを見る
    StateMachine& state = StateMachine::GetInstance();
    if(IsAsleep() == false)
    {
        NN_RESULT_SUCCESS;
    }

    // DeviceDetector を復帰させる（完了までブロックされる）
    DeviceDetector& detector = DeviceDetector::GetInstance();
    bool isRemovedDuringSleep = detector.Resume();

    // 以降で抜けたらスリープ復帰したことにする
    NN_UTIL_SCOPE_EXIT
    {
        m_SessionInfo.m_IsAsleep = false;
    };

    // sdmmc のスリープ復帰
    NN_RESULT_DO(DataIo::GetInstance().AwakenSdmmc());

    // ASIC のスリープ復帰
    NN_RESULT_DO(DataIo::GetInstance().AwakenGcAsic());

    // 挿抜状態のチェック
    if(m_CardInfoBeforeSleep.isInserted == false || isRemovedDuringSleep || state.IsCardInserted() == false)
    {
        // 抜去されていたか、復帰時の調査で抜去されたことがわかったか、今現在抜去されているならスリープ復帰はここで終わらせる
        // 挿入されていた状態から抜去されていれば、DeviceDetector::WakeUp 内で抜去イベントが通知されているし、
        // そもそもスリープ復帰後には抜去されていないかを上側でも確認してもらうハズ
        NN_DETAIL_GC_LOG("card not inserted before/after sleep\n");
        NN_RESULT_SUCCESS;
    }
    // （この時点でカードはスリープ前後で 挿 → 挿 のケースのはず）

    // ** カードの状態復帰
    // 特殊な状態の場合の処理
    if(m_CardInfoBeforeSleep.cardMode == GameCardMode_Initial)
    {
        NN_DETAIL_GC_LOG("card was in initial state before sleep\n");
        NN_RESULT_SUCCESS;
    }
    else if(m_CardInfoBeforeSleep.cardMode == GameCardMode_Debug)
    {
        // 開発モードのケース（開発カードへの書き込み）を必要以上にケアしたくないので、カードが一瞬抜けたことにする
        return nn::fs::ResultGameCardCardNotInserted();
    }

    // Awaken の時はカードヘッダ読みのリトライを行う
    NN_RESULT_DO(ResumeCardStateAndCheckCardInfo(true));
    NN_RESULT_SUCCESS;
}

// AsicHandler のキューに積まれた Read で呼ばれる
nn::Result AsicHandlerCore::Read(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    // 引数チェック
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo != nullptr);
    ReadInfo& rinfo = *(pReadInfo);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outCounterBuffer != nullptr);
    NN_DETAIL_GC_SDK_REQUIRES(rinfo.outDataBuffer != nullptr);

    // 前処理
    m_SessionInfo.m_DeviceStatusAfterRead = nn::ResultSuccess();

    // カウンタ値が 4Byte 上限に達していない場合はカウンタをインクリ
    if (m_CardAccessInternalInfo.readCountFromInsert < UINT32_MAX)
    {
        m_CardAccessInternalInfo.readCountFromInsert++;
    }
    if (m_CardAccessInternalInfo.readCountFromAwaken < UINT32_MAX)
    {
        m_CardAccessInternalInfo.readCountFromAwaken++;
    }
    // 純粋なリード処理
    size_t readSize = rinfo.pageCount * GcPageSize;
    auto result = (this->ReadWithErrorHandling(rinfo.outDataBuffer, rinfo.dataBufferSize, rinfo.pageAddress, rinfo.pageCount));
    if (result.IsFailure())
    {
        m_CardAccessInternalInfo.lastReadErrorPageAddress = rinfo.pageAddress;
        m_CardAccessInternalInfo.lastReadErrorPageCount   = rinfo.pageCount;
        return result;
    }

    // オペレーション 64 byte 暗号化直後、read data 復号化直前の状態のカウンタ値をメモ
    GcCrypto& gcCrypto = GcCrypto::GetInstance();
    gcCrypto.GetCounter(rinfo.outCounterBuffer, rinfo.counterBufferLength);

    // 次の read をどんどん処理させるため、カウンタの値を read で読んだ量の分ずらしておく
    gcCrypto.IncrementCounter(readSize);

    //（ここでは復号化は行わず、呼び出しスレッドでやってもらう：これによりパイプライン処理を実現する）

    // 通信直後に CMD13 を発行して届かなかったレスポンスエラーがないか調べる
    {
        BitPack32 deviceStatus;
        m_SessionInfo.m_DeviceStatusAfterRead = DataIo::GetInstance().GetDeviceStatus(&deviceStatus);
    }

    NN_RESULT_SUCCESS;
}

// Read に失敗した場合の処理
nn::Result AsicHandlerCore::ReinitializeAsicWhenReadFailed() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_WARNING_LOG("ReinitializeAsicWhenReadFailed occurred!\n\n");
    // Reinitialize が走った数をカウント
    m_CardAccessInternalInfo.asicReinitializeNum++;

    // 念のためステートを見る
    StateMachine& state = StateMachine::GetInstance();
    if(!state.IsCardInserted())
    {
        // Reinitialize の失敗回数を取得
        m_CardAccessInternalInfo.asicReinitializeFailureNum++;
        m_CardAccessInternalInfo.lastReinitializeFailureResult = nn::fs::ResultGameCardCardNotInserted();
        return nn::fs::ResultGameCardCardReinitializeFailure();
    }
    // GameCard の状態を保存後 ASIC 初期化
    EvacuateCardInfoAndInitializeAsicIfNeeded();

    // 確認しつつ復帰
    // 復帰時のカードヘッダ読み失敗は一発でアウト
    auto result = ResumeCardStateAndCheckCardInfo(false);
    if (result.IsFailure())
    {
        // Reinitialize の失敗回数を取得
        m_CardAccessInternalInfo.asicReinitializeFailureNum++;
        m_CardAccessInternalInfo.lastReinitializeFailureResult = result;
        return nn::fs::ResultGameCardCardReinitializeFailure();
    }

    NN_RESULT_SUCCESS;
}


// Read に失敗した場合の処理
nn::Result AsicHandlerCore::HandleReadError() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();
    AsicRegister& asicReg = AsicRegister::GetInstance();

    bool isNeedRetryAfterAsicReboot = false;

    // カード抜け確認、挿入状態なら UpdateKey（UpdateKey でカウンタ類がリセットされる）
    nn::Result result = CheckCardInsertionToUpdateKey();
    if(result.IsSuccess())
    {
        // カードが挿入状態なのでレジスタを読み、調べる（UpdateKey 直後の通信エラーは救わない）
        NN_RESULT_DO(asicReg.ReadAndCheckRegisterErrorStatus(&result));

        // レジスタ読みの結果、何もエラーが起きていなかった場合
        if(result.IsSuccess())
        {
            NN_DETAIL_GC_ERR_LOG("Unknown error occurred in send card read sequence\n");
            // 強制的にリトライする（読んだデータは捨てる）
            // TODO : ここで isNeedRetryAfterAsicReboot を立てるか相談
            result = nn::fs::ResultGameCardCardNeedRetry();
        }

        // カードが fatal している場合 ResultGameCardCardNeedRetryAfterAsicReinitialize が返ってくる。
        // その場合は AsicReboot 後リトライを行う
        if(nn::fs::ResultGameCardCardNeedRetryAfterAsicReinitialize::Includes(result))
        {
            isNeedRetryAfterAsicReboot = true;
            result = nn::fs::ResultGameCardCardNeedRetry();
        }

        // リフレッシュが必要な場合
        if(nn::fs::ResultGameCardNeedRefresh::Includes(result))
        {
            // リフレッシュは連続して要求が来ることもあるので、ある程度繰り返す（そのほか、カードによってはリフレッシュコマンドの応答にリフレッシュ要求が立つ）
            // リフレッシュ要求直後に別のエラーが来るのはハンドリングが複雑すぎるのであきらめる
            // 確認済み事項：リフレッシュは何もないときに打っても無視されるだけだし、SetKey を挟まずに打ってもよい
            nn::Result refreshResult = ResultSuccess();
            const int RefreshRetryMaxNum = 4;
            for(int j = 0; j < RefreshRetryMaxNum; j++)
            {
                NN_DETAIL_GC_DETAIL_LOG("Send card refresh (%d)\n", j);
                RefreshResponse refreshResponse;
                refreshResult = asicOp.SendCardRefresh(&refreshResponse);
                if(refreshResult.IsSuccess())
                {
                    m_CardAccessInternalInfo.refreshSeccessNum++;
                    break;
                }
            }
            // TODO: リフレッシュ要求時のタイムアウト対策（現状 sdmmc に Result を置き換えてもらっているので検知できない）

            // 連続リフレッシュ数を超えたリフレッシュ要求は上へ返す
            NN_RESULT_DO(refreshResult);

            // SetKey は Refresh とセットの処理（これも失敗は救わない）
            NN_RESULT_DO(asicOp.SendCardSetKey());

            // リフレッシュ時は強制的にリトライする（読んだデータは捨てる）
            result = nn::fs::ResultGameCardCardNeedRetry();
        }
    }
    else
    {
        // UpdateKey に失敗した理由が「カード抜け」でない場合 AsicReboot 後リトライを行う
        if(!nn::fs::ResultGameCardCardNotInserted::Includes(result))
        {
            isNeedRetryAfterAsicReboot = true;
            // ASIC Reboot 後に Retry を行う
            result = nn::fs::ResultGameCardCardNeedRetry();
        }
    }

    if(isNeedRetryAfterAsicReboot)
    {
        // ASIC の再初期化
        NN_RESULT_DO(ReinitializeAsicWhenReadFailed());
    }

    // その他ハンドリングされていないエラー result はここで帰る
    return result;
}

nn::Result AsicHandlerCore::ReadWithErrorHandling(char* outDataBuffer, const size_t outDataBufferLength, const u32 pageAddress, const u32 pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(pageAddress < (GetCardSize() / GcPageSize));
    NN_DETAIL_GC_SDK_REQUIRES(pageCount <= ((GetCardSize() / GcPageSize) - pageAddress));

    const size_t readSize = pageCount * GcPageSize;
    NN_DETAIL_GC_SDK_REQUIRES(readSize <= outDataBufferLength);

    AsicOperation& asicOp = AsicOperation::GetInstance();
    AsicRegister& asicReg = AsicRegister::GetInstance();

    // 読み出し領域のチェック
    if(pageCount == 0)
    {
        NN_DETAIL_GC_LOG("0 size read occurred\n");
        NN_RESULT_SUCCESS;
    }
    bool isInNormalArea = pageAddress < m_SessionInfo.m_CardHeader.limArea;
    bool isInSecureArea = (pageAddress + (pageCount - 1)) >= m_SessionInfo.m_CardHeader.limArea;
    if(isInNormalArea && isInSecureArea)
    {
        NN_DETAIL_GC_ERR_LOG("ResultInvalidAccessAcrossMode\n");
        return nn::fs::ResultGameCardInvalidAccessAcrossMode();
    }
    else if(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Normal && isInSecureArea)
    {
        NN_DETAIL_GC_ERR_LOG("ResultInvalidSecureAccess\n");
        return nn::fs::ResultGameCardInvalidSecureAccess();
    }
    else if(StateMachine::GetInstance().GetGameCardMode() == GameCardMode_Secure && isInNormalArea)
    {
        NN_DETAIL_GC_ERR_LOG("ResultInvalidNormalAccess\n");
        return nn::fs::ResultGameCardInvalidNormalAccess();
    }

    // リトライのループ
    for(int i = 0; i < GcSendCommandMaxCount; i++)
    {
        // レジスタをクリア
        nn::Result result = asicReg.ClearRegisterErrorStatus();
        if (result.IsFailure())
        {
            NN_DETAIL_GC_WARNING_LOG("ClearRegisterErrorStatus before Read Failure. (Module:%d, Description:%d)\nRetry after Asic reInitialize\n", result.GetModule(), result.GetDescription());
            // ASIC の再初期化
            NN_RESULT_DO(ReinitializeAsicWhenReadFailed());
            continue;
        }

        // カードから読む
        result = asicOp.SendCardReadPage(outDataBuffer, readSize, pageAddress, pageCount);
        if(result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        // *** （以下エラーハンドリング） ***

        // エラー発生時は変なデータが読めている可能性がある：が問題が起きないように 0 セット
        // TODO: TEMP: 今はデバイス暴走は知りたいのでコメントアウト
        // memset(outDataBuffer, 0, readSize);
        result = HandleReadError();
        // リトライ要求
        if(nn::fs::ResultGameCardCardNeedRetry::Includes(result) || nn::fs::ResultGameCardCardAccessTimeout::Includes(result))
        {
            NN_DETAIL_GC_WARNING_LOG("Retry communication (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            continue; // Retry
        }

        // その他ハンドリングされていないエラー result はここで返る
        return result;
    }
    // この時のページアドレスとか残す？
    m_CardAccessInternalInfo.retryLimitOutNum++;
    return nn::fs::ResultGameCardRetryLimitOut();
}


// *** エラーハンドリング

nn::Result AsicHandlerCore::CheckCardInsertionToUpdateKey() NN_NOEXCEPT
{
    // まずカード抜けを疑う
    const int WaitMilliSecForGpio = 10;
    NN_DETAIL_GC_DETAIL_LOG("Confirm Card Removal\n");
    for(int retryNum = 0; ; retryNum++)
    {
        if( IsRemoved() || ! StateMachine::GetInstance().IsCardInserted())
        {
            NN_DETAIL_GC_DETAIL_LOG("Failure: Card Removed\n");
            // カード抜去があると、ASIC は SoC からの命令を受け付けなくなりリセット必須なので、そのまま帰ればよい
            return nn::fs::ResultGameCardCardNotInserted();
        }

        // retry
        if(retryNum == 0)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitMilliSecForGpio));
            continue;
        }
        else if(retryNum == 1)
        {
            // カード抜去に関するクリティカルセクション
            {
                NN_DETAIL_GC_CANCEL_SDMMC_LOCK_GUARD;

                // MMC Re-Tuning 時に転送中断されてはいけないのでガードする
                NN_DETAIL_GC_DETAIL_LOG("Start MMC Re-Tuning\n");
                DataIo::GetInstance().Deactivate();
                NN_RESULT_DO( DataIo::GetInstance().Activate() );
            }
            continue;
        }
        // retryNum == 2 になるまで回る：GPIO 抜けイベントが MMC Re-Tuning よりも遅れてきたら抜去を拾えない（上でハンドリングする必要がある）
        break;
    }

    // 通信失敗後においてカード挿入状態なら UpdateKey を打つ
    NN_RESULT_DO(AsicHandlerProcedure::UpdateKey());
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::ExecuteOperationWithRetry(nn::Result (AsicHandlerCore::*operationFunc)()) NN_NOEXCEPT
{
    if((this->*operationFunc)().IsFailure())
    {
        // 失敗したら UpdateKey を打ってリトライ
        NN_RESULT_DO(CheckCardInsertionToUpdateKey());
        NN_RESULT_DO((this->*operationFunc)());
    }
    NN_RESULT_SUCCESS;
}

bool AsicHandlerCore::IsActivationValid() NN_NOEXCEPT
{
    return StateMachine::GetInstance().IsCardActivated() && (IsRemoved() == false);
}

nn::Result AsicHandlerCore::CheckCardReady() NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_THROW_IF_REMOVED;
    NN_DETAIL_GC_RESULT_THROW_UNLESS_ACTIVATED;
    NN_RESULT_SUCCESS;
}



// *** 内部で呼ばれる API

nn::Result AsicHandlerCore::EnableCardBus() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    AsicOperation& asicOp = AsicOperation::GetInstance();

    // カード電源投入
    GeneralIo::GetInstance().SetGcPowerOn();

    // カードバスの有効化
    NN_RESULT_DO(asicOp.EnableCardBus());

    // カードメモリが安定するまでウェイト
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(BusWaitTimeMsec) );
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::GetCardHeader() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    char buf[sizeof(m_SessionInfo.m_CardInitReceiveDataArray) + sizeof(m_SessionInfo.m_CardHeaderArray)];
    NN_RESULT_DO(AsicOperation::GetInstance().GetCardHeader(buf, sizeof(buf)));

    // データの振り分け
    memcpy(m_SessionInfo.m_CardInitReceiveDataArray, buf, sizeof(m_SessionInfo.m_CardInitReceiveDataArray));
    memcpy(m_SessionInfo.m_CardHeaderArray, buf + sizeof(m_SessionInfo.m_CardInitReceiveDataArray), sizeof(m_SessionInfo.m_CardHeaderArray));

    NN_DETAIL_GC_DEBUG_MEMDUMP(m_SessionInfo.m_CardHeaderArray,  sizeof(m_SessionInfo.m_CardHeaderArray), "card header:\n");
    NN_RESULT_SUCCESS;
}

nn::Result AsicHandlerCore::ChangeGcModeToSecure() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    NN_RESULT_DO(AsicOperation::GetInstance().ChangeGcModeToSecure(m_SessionInfo.m_CardSecurityInformationAray, sizeof(m_SessionInfo.m_CardSecurityInformationAray)));
    NN_DETAIL_GC_DEBUG_MEMDUMP(m_SessionInfo.m_CardSecurityInformationAray,  sizeof(m_SessionInfo.m_CardSecurityInformationAray), "card security information:\n");

    // T1 なら、TitleKey, KEK 差し替えをチェックする
    if(m_SessionInfo.m_CardHeader.selSec == SelectSecurityMode_T1)
    {
        // 受け取った Initial Data がカードヘッダに書かれているハッシュと一致するかのチェック
        char initialDataHashBuffer[GcCrypto::GcSha256Length];
        GcCrypto::GetInstance().CalcSha256(initialDataHashBuffer, sizeof(initialDataHashBuffer),
            reinterpret_cast<char*>(m_SessionInfo.m_CardSecurityInformation.initialDataArray), sizeof(m_SessionInfo.m_CardSecurityInformation.initialDataArray));
        if(CompareAllMemory(initialDataHashBuffer, m_SessionInfo.m_CardHeader.initialDataHash, GcCrypto::GcSha256Length) != 0)
        {
            NN_DETAIL_GC_ERR_LOG("initial data hash does not match\n");
            return nn::fs::ResultGameCardInitialDataMismatch();
        }
        // initial data の reserved 領域が 0 埋めであることのチェック（ここの領域で全体ハッシュを無理やり合わすことができないよう、明示的にチェックする）
        bool isNotZeroContained = false;
        for(size_t i = 0; i < sizeof(m_SessionInfo.m_CardSecurityInformation.initialData.reserved); i++)
        {
            isNotZeroContained |= (m_SessionInfo.m_CardSecurityInformation.initialData.reserved[i] != 0);
        }
        if(isNotZeroContained)
        {
            NN_DETAIL_GC_ERR_LOG("reserved area in initial data is not filled with zero\n");
            return nn::fs::ResultGameCardInitialNotFilledWithZero();
        }

        // 受け取った T1 証明書の KEK index が一致するかをチェック
        if(m_SessionInfo.m_CardSecurityInformation.t1CardCertificate.kekIndex != m_SessionInfo.m_CardHeader.keyIndex.Get<CardHeaderMap_KeyIndex::kekIndex>())
        {
            NN_DETAIL_GC_ERR_LOG("kek index in certificate does not match\n");
            return nn::fs::ResultGameCardKekIndexMismatch();
        }
    }

    NN_RESULT_SUCCESS;
}


nn::Result AsicHandlerCore::GetErrorInfo(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    NN_DETAIL_GC_SDK_REQUIRES(pReadInfo->dataBufferSize >= sizeof(nn::fs::GameCardErrorReportInfo));

    AsicRegister& asicReg = AsicRegister::GetInstance();
    AsicRegisterErrorInfo asicRegisterErrorInfo;
    asicReg.GetRegisterErrorInfo(&asicRegisterErrorInfo);

    DeviceDetectorErrorInfo deviceDetectorErrorInfo;
    DeviceDetector::GetInstance().GetDeviceDetectorErrorInfo(&deviceDetectorErrorInfo);

    nn::fs::GameCardErrorReportInfo* pGameCardErrorInfo = reinterpret_cast<nn::fs::GameCardErrorReportInfo*>(pReadInfo->outDataBuffer);
    pGameCardErrorInfo->gameCardCrcErrorNum            = asicRegisterErrorInfo.gameCardCrcErrorNum;
    pGameCardErrorInfo->asicCrcErrorNum                = asicRegisterErrorInfo.asicCrcErrorNum;
    pGameCardErrorInfo->refreshNum                     = asicRegisterErrorInfo.refreshNum;
    pGameCardErrorInfo->timeoutRetryNum                = asicRegisterErrorInfo.gameCardTimeoutErrorNum;

    pGameCardErrorInfo->retryLimitOutNum               = m_CardAccessInternalInfo.retryLimitOutNum;
    pGameCardErrorInfo->asicReinitializeNum            = m_CardAccessInternalInfo.asicReinitializeNum;
    pGameCardErrorInfo->asicReinitializeFailureNum     = m_CardAccessInternalInfo.asicReinitializeFailureNum;
    pGameCardErrorInfo->asicReinitializeFailureDetail  = m_CardAccessInternalInfo.lastReinitializeFailureResult.GetDescription();
    pGameCardErrorInfo->lastReadErrorPageAddress       = m_CardAccessInternalInfo.lastReadErrorPageAddress;
    pGameCardErrorInfo->lastReadErrorPageCount         = m_CardAccessInternalInfo.lastReadErrorPageCount;
    pGameCardErrorInfo->refreshSucceededCount          = m_CardAccessInternalInfo.refreshSeccessNum;
    pGameCardErrorInfo->readCountFromInsert            = m_CardAccessInternalInfo.readCountFromInsert;
    pGameCardErrorInfo->readCountFromAwaken            = m_CardAccessInternalInfo.readCountFromAwaken;

    pGameCardErrorInfo->insertionCount                 = deviceDetectorErrorInfo.insertionCount;
    pGameCardErrorInfo->removalCount                   = deviceDetectorErrorInfo.removalCount;

    pGameCardErrorInfo->initializeCount                = m_TotalAsicInfo.initializeCount;
    pGameCardErrorInfo->awakenCount                    = m_TotalAsicInfo.awakenCount;
    pGameCardErrorInfo->awakenFailureNum               = m_TotalAsicInfo.awakenFailureNum;

    NN_RESULT_SUCCESS;
}

// Reset 時に内部情報をクリアする
void AsicHandlerCore::ClearSessionInfo() NN_NOEXCEPT
{
    // セッション状態のリセット
    memset(m_SessionInfo.m_CardHeaderArray, 0, sizeof(m_SessionInfo.m_CardHeaderArray));
    memset(m_SessionInfo.m_CardSecurityInformationAray, 0, sizeof(CardSecurityInformation));
    memset(m_SessionInfo.m_CardImageHash, 0, sizeof(m_SessionInfo.m_CardImageHash));
    m_SessionInfo.m_IsCardBusEnabled = false;

    // m_IsAsleep が true になるのは Deactivate（内部で Reset が呼ばれる）したあとなので、ここで false にしても問題ない
    m_SessionInfo.m_IsAsleep = false;
    m_SessionInfo.m_DeviceStatusAfterRead = nn::ResultSuccess();
}

// ReInitialize 時の Reset 以外の Reset 時にエラーレポート向けの内部情報をクリアする
// Reset 時に ClearSessionInfo でクリアするとエラーレポートに必要な情報が残らないため
void AsicHandlerCore::ClearSessionInfoForErrorReport() NN_NOEXCEPT
{
    // レジスタ関連情報クリア
    AsicRegister& asicReg = AsicRegister::GetInstance();
    asicReg.ClearErrorInfo();
    // スリープ起因のクリアなら、 readCountFromInsert はクリアしない
    uint32_t readCountFromInsert = m_SessionInfo.m_IsAsleep ? m_CardAccessInternalInfo.readCountFromInsert : 0;

    // カードアクセスエラー関連情報クリア
    std::memset(&m_CardAccessInternalInfo, 0, sizeof(m_CardAccessInternalInfo));
    m_CardAccessInternalInfo.readCountFromInsert = readCountFromInsert;

    // ゲームカードの ID 情報をクリア( Reset されても残しておきたい ID のみ)
    memset(m_SessionInfo.m_CardInitReceiveDataArray, 0, sizeof(CardInitReceiveData));
    memset(&(m_SessionInfo.m_CardId2), 0, sizeof(m_SessionInfo.m_CardId2));
    memset(&(m_SessionInfo.m_CardId3), 0, sizeof(m_SessionInfo.m_CardId3));
}

// writer 向け
nn::Result AsicHandlerCore::EnableCardBusForWriter() NN_NOEXCEPT
{
    return ExecuteOperationWithRetry(&AsicHandlerCore::EnableCardBus);
}

// XCIE ライター向け
nn::Result AsicHandlerCore::GetCardHeader(ReadInfo* pReadInfo) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    AsicOperation& asicOp = AsicOperation::GetInstance();

    // カードバス有効化
    nn::Result result = ExecuteOperationWithRetry(&AsicHandlerCore::EnableCardBus);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "Enable card bus failed\n");
    NN_RESULT_DO(result);

    // カードヘッダの試し読み
    for (int retryCount = 0; retryCount < GcSendCommandMaxCount; retryCount++)
    {
        result = asicOp.SendCardReadPageWithNotAlignedBuffer(pReadInfo->outDataBuffer, GcPageSize, 0, 1);
        if (result.IsSuccess())
        {
            break;
        }
        NN_DETAIL_GC_WARNING_LOG("Read CardHeader first failed2 (%d).\n", retryCount);
    }

    if(result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("Read CardHeader first retry out2\n");
        return result;
    }

    NN_RESULT_SUCCESS;
}


} } }
