﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <nn/nn_SdkAssert.h>
#include <nn/fssystem/utilTool/fs_RelocatedBinaryMatch.h>

namespace nn { namespace fssystem { namespace utilTool {

NN_DEFINE_STATIC_CONSTANT(const int64_t RelocationShiftRangeContainer::SplitSizeMin);
NN_DEFINE_STATIC_CONSTANT(const int64_t RelocationShiftRangeContainer::LargeSplitSizeMin);

namespace {

const int AnchorableExtendedSize = 1 * 1024 * 1024;
const int SplitCountMax = 10;

typedef BinaryMatchResult MatchResult;
typedef RelocationShiftRange ShiftRange;
typedef RelocationShiftRangeContainer ShiftRangeContainer;

template< typename T >
inline size_t GetDistance(T begin, T end) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(begin, end);
    return static_cast<size_t>(std::distance(begin, end));
}

template< typename T >
inline void Sort(T* pContainer) NN_NOEXCEPT
{
    std::sort(pContainer->begin(), pContainer->end());
}

// サイズの昇順にソートする。
inline void SortOrderLessSize(std::vector<ShiftRange>* pContainer) NN_NOEXCEPT
{
    std::sort(
        pContainer->begin(),
        pContainer->end(),
        [](const ShiftRange& lhs, const ShiftRange& rhs) NN_NOEXCEPT
        {
            if( lhs.GetSize() == rhs.GetSize() )
            {
                return lhs.GetOffset() < rhs.GetOffset();
            }
            return lhs.GetSize() < rhs.GetSize();
        }
    );
}

// サイズの降順にソートする。
inline void SortOrderGreaterSize(std::vector<ShiftRange>* pContainer) NN_NOEXCEPT
{
    std::sort(
        pContainer->begin(),
        pContainer->end(),
        [](const ShiftRange& lhs, const ShiftRange& rhs) NN_NOEXCEPT
        {
            if( lhs.GetSize() == rhs.GetSize() )
            {
                return lhs.GetOffset() < rhs.GetOffset();
            }
            return lhs.GetSize() > rhs.GetSize();
        }
    );
}

// サイズ 0 の領域を削除する。
inline void Resize(std::vector<ShiftRange>* pContainer) NN_NOEXCEPT
{
    const auto point = std::partition(
        pContainer->begin(),
        pContainer->end(),
        [](const ShiftRange& range) NN_NOEXCEPT
        {
            return 0 < range.GetSize();
        }
    );

    pContainer->resize(GetDistance(pContainer->begin(), point));
}

// ShiftRange の合計を扱うクラス
class ShiftRangeCombiner
{
public:
    ShiftRangeCombiner() NN_NOEXCEPT
        : m_Offset(0)
        , m_Size(0)
    {
    }

    void Reset(const ShiftRange& range) NN_NOEXCEPT
    {
        m_Offset = range.GetOffset() + range.GetSize();
        m_Size = 0;
    }

    void Expand(const ShiftRange& range) NN_NOEXCEPT
    {
        m_Size += range.GetSize();
    }

    void Reduce(const ShiftRange& range) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL(range.GetSize(), m_Size);
        m_Offset = range.GetOffset() + range.GetSize();
        m_Size -= range.GetSize();
    }

    int64_t GetOffset() const NN_NOEXCEPT
    {
        return m_Offset;
    }

    int64_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

private:
    int64_t m_Offset;
    int64_t m_Size;
};

// std::vector<ShiftRange> の範囲を表すクラス
class ShiftRangeBounds
{
public:
    typedef std::vector<ShiftRange>::iterator iterator;
    typedef std::vector<ShiftRange>::const_iterator const_iterator;

public:
    ShiftRangeBounds() NN_NOEXCEPT
        : m_Begin()
        , m_End()
    {
    }

    ShiftRangeBounds(iterator begin, iterator end) NN_NOEXCEPT
        : m_Begin(begin)
        , m_End(end)
    {
    }

    template< typename TFunc >
    ShiftRangeBounds(TFunc&& func, iterator begin, iterator end) NN_NOEXCEPT
        : m_Begin(func(begin, end))
        , m_End(end)
    {
    }

    template< typename TFunc >
    ShiftRangeBounds(iterator begin, iterator end, TFunc&& func) NN_NOEXCEPT
        : m_Begin(begin)
        , m_End(func(begin, end))
    {
    }

    ShiftRangeBounds(const ShiftRangeBounds& bounds, iterator end) NN_NOEXCEPT
        : m_Begin(bounds.begin())
        , m_End(end)
    {
        NN_SDK_REQUIRES_MINMAX(end, bounds.begin(), bounds.end());
    }

    ShiftRangeBounds(iterator begin, const ShiftRangeBounds& bounds) NN_NOEXCEPT
        : m_Begin(begin)
        , m_End(bounds.end())
    {
        NN_SDK_REQUIRES_MINMAX(begin, bounds.begin(), bounds.end());
    }

    iterator begin() const NN_NOEXCEPT
    {
        return m_Begin;
    }

    iterator end() const NN_NOEXCEPT
    {
        return m_End;
    }

    bool empty() const NN_NOEXCEPT
    {
        return m_Begin == m_End;
    }

private:
    iterator m_Begin;
    iterator m_End;
};

}

RelocationShiftRangeContainer::RelocationShiftRangeContainer(
    std::vector<MatchResult>* pResults,
    int64_t endOffset
) NN_NOEXCEPT
    : m_EndOffset(endOffset)
    , m_Results(*pResults)
    , m_MoveRanges()
    , m_FreeRanges()
    , m_FreeBackup()
    , m_FreeRangeLastOffset(endOffset)
{
    NN_SDK_REQUIRES_GREATER(endOffset, 0);
}

RelocationShiftRangeContainer::~RelocationShiftRangeContainer() NN_NOEXCEPT
{
    // 残った領域を m_Results に返還する
    for( const auto& range : m_MoveRanges )
    {
        m_Results.push_back(
            MatchResult::MakeUnknown(range.GetOffset(), range.GetSize()));
    }

    std::sort(m_Results.begin(), m_Results.end());
}

// BinaryMatchResult の分割・移動を行うのに必要な情報を生成
bool RelocationShiftRangeContainer::Build(int64_t expandedSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(expandedSize, m_EndOffset);

    // oldOffset の降順（InvalidOffset を後方へ）、newOffset の昇順になるようにソート
    std::sort(
        m_Results.begin(),
        m_Results.end(),
        [](const MatchResult& lhs, const MatchResult& rhs) NN_NOEXCEPT
        {
            if( lhs.oldOffset == rhs.oldOffset )
            {
                NN_SDK_ASSERT_EQUAL(lhs.oldOffset, MatchResult::InvalidOffset);
                return lhs.newOffset < rhs.newOffset;
            }
            return lhs.oldOffset > rhs.oldOffset;
        }
    );

    const auto begin = m_Results.begin();
    const auto end = m_Results.end();
    auto point = begin;
    auto endOffset = m_EndOffset;

    // 空き領域を取得
    for( ; point < end; ++point )
    {
        if( point->oldOffset == MatchResult::InvalidOffset )
        {
            break;
        }

        const auto offset = point->GetOldEndOffset();
        if( offset < endOffset )
        {
            m_FreeRanges.push_back(ShiftRange(offset, endOffset - offset));
        }

        endOffset = point->oldOffset;
    }

    // 空き領域があり かつ 移動する領域がある
    if( begin < point && point < end )
    {
        if( 0 < endOffset )
        {
            m_FreeRanges.push_back(ShiftRange(0, endOffset));
        }

        if( !m_FreeRanges.empty() )
        {
            // オフセットの降順になっているので反転
            std::reverse(m_FreeRanges.begin(), m_FreeRanges.end());

            // NOTE: 拡張した（であろう）末尾の空き領域は特別扱い
            //       （空き領域で一番大きくなる可能性があり、無駄に割り当てが発生するのを防ぐため）
            {
                auto& last = m_FreeRanges.back();
                const auto lastOffset = last.GetOffset();

                if( m_EndOffset - expandedSize <= lastOffset + last.GetSize() )
                {
                    m_FreeRangeLastOffset = lastOffset;

                    m_FreeRanges.pop_back();
                }
            }

            if( !m_FreeRanges.empty() )
            {
                NN_SDK_ASSERT_LESS(
                    m_FreeRanges.back().GetOffset() + m_FreeRanges.back().GetSize(),
                    m_FreeRangeLastOffset
                );

                m_MoveRanges.reserve(GetDistance(point, end));

                // 移動する領域を取得
                for( auto iter = point; iter < end; ++iter )
                {
                    NN_SDK_ASSERT_EQUAL(iter->oldOffset, MatchResult::InvalidOffset);
                    m_MoveRanges.push_back(ShiftRange(iter->newOffset, iter->size));
                }

                m_Results.resize(GetDistance(begin, point)); // 移動する領域を削除
            }

            return true;
        }
    }
    return false;
}

// BinaryMatchResult がマッチ済みの領域の後続になるように移動する
void RelocationShiftRangeContainer::ShiftByAnchor(int64_t limitNewOffset) NN_NOEXCEPT
{
    if( m_FreeRanges.empty() || m_MoveRanges.empty() )
    {
        return;
    }

    Sort(&m_FreeRanges);
    Sort(&m_MoveRanges);

    // 新しいデータの末尾まで＋αを空き領域として追加する
    {
        const auto newEndOffset = limitNewOffset + AnchorableExtendedSize;
        if( m_FreeRangeLastOffset < newEndOffset )
        {
            ShiftRange range(m_FreeRangeLastOffset, newEndOffset - m_FreeRangeLastOffset);
            m_FreeRanges.push_back(range);
        }
    }

    // マッチ済み領域を newOffset でソート
    Sort(&m_Results);
    const auto resultCount = m_Results.size();

    for( auto& moveRange : m_MoveRanges )
    {
        const auto move = moveRange;
        const auto resultBegin = m_Results.begin();
        const auto resultEnd = resultBegin + resultCount;

        // 移動領域の直前のマッチ済み領域を検索
        const auto resultIter = std::lower_bound(
            resultBegin,
            resultEnd,
            move.GetOffset(),
            [](const MatchResult& result, int64_t offset) NN_NOEXCEPT
            {
                return result.GetNewEndOffset() < offset;
            }
        );

        if( resultIter != resultEnd && resultIter->GetNewEndOffset() == move.GetOffset() )
        {
            const auto& result = *resultIter;
            NN_SDK_ASSERT_NOT_EQUAL(result.oldOffset, MatchResult::InvalidOffset);

            const auto oldOffset = result.GetOldEndOffset();

            // マッチ済み領域の後続の空き領域を検索
            const auto rangeIter = std::lower_bound(
                m_FreeRanges.begin(),
                m_FreeRanges.end(),
                oldOffset,
                [](const ShiftRange& range, int64_t offset) NN_NOEXCEPT
                {
                    return range.GetOffset() < offset;
                }
            );

            if( rangeIter != m_FreeRanges.end() && rangeIter->GetOffset() == oldOffset )
            {
                auto& freeRange = *rangeIter;
                const auto free = freeRange;
                NN_SDK_ASSERT_GREATER(freeRange.GetSize(), 0);

                if( move.GetSize() == free.GetSize() )
                {
                    m_Results.push_back(BinaryMatchResult::MakeAnchor(
                        move.GetOffset(), oldOffset, move.GetSize()));

                    moveRange.Advance(free.GetSize());
                    freeRange.Advance(free.GetSize());
                }
                else if( move.GetSize() < free.GetSize() )
                {
                    m_Results.push_back(BinaryMatchResult::MakeAnchor(
                        move.GetOffset(), oldOffset, move.GetSize()));

                    moveRange.Advance(move.GetSize());
                    freeRange.Advance(move.GetSize());
                }
                else
                {
                    m_Results.push_back(BinaryMatchResult::MakeAnchor(
                        move.GetOffset(), oldOffset, free.GetSize()));

                    moveRange.Advance(free.GetSize());
                    freeRange.Advance(free.GetSize());
                }
            }
        }
    }

    // 追加した領域を削除する
    {
        const auto& lastOffset = m_FreeRanges.back().GetOffset();
        if( m_FreeRangeLastOffset <= lastOffset )
        {
            m_FreeRangeLastOffset = lastOffset;

            m_FreeRanges.pop_back();
        }
    }

    Resize(&m_FreeRanges);
    Resize(&m_MoveRanges);
}

// 小さな BinaryMatchResult をファーストフィットで移動する
void RelocationShiftRangeContainer::ShiftBySmallFit() NN_NOEXCEPT
{
    if( m_FreeRanges.empty() || m_MoveRanges.empty() )
    {
        return;
    }

    const auto predicate = [](const ShiftRange& range) NN_NOEXCEPT
    {
        return range.GetSize() < SplitSizeMin;
    };

    const ShiftRangeBounds freeBounds(
        m_FreeRanges.begin(),
        m_FreeRanges.end(),
        [=](Container::iterator begin, Container::iterator end) NN_NOEXCEPT
        {
            // 小さい領域を前方に移動
            return std::partition(begin, end, predicate);
        }
    );
    if( freeBounds.empty() )
    {
        return;
    }

    const ShiftRangeBounds moveBounds(
        m_MoveRanges.begin(),
        m_MoveRanges.end(),
        [=](Container::iterator begin, Container::iterator end) NN_NOEXCEPT
        {
            // 小さい領域を前方に移動
            return std::partition(begin, end, predicate);
        }
    );
    if( moveBounds.empty() )
    {
        return;
    }

    // 小さい領域のみをオフセットの昇順にソート
    std::sort(freeBounds.begin(), freeBounds.end());
    std::sort(moveBounds.begin(), moveBounds.end());

    // マッチ済み領域を newOffset でソート
    Sort(&m_Results);
    const auto resultCount = m_Results.size();

    for( auto& moveRange : moveBounds )
    {
        const auto move = moveRange;

        // 移動領域の直前のマッチ済み領域を検索
        const auto resultBegin = m_Results.begin();
        const auto resultEnd = resultBegin + resultCount;
        const auto resultIter = std::lower_bound(
            resultBegin,
            resultEnd,
            move.GetOffset(),
            [](const MatchResult& result, int64_t offset) NN_NOEXCEPT
            {
                return result.GetNewEndOffset() < offset;
            }
        );
        if( resultIter != resultEnd && resultIter->GetNewEndOffset() == move.GetOffset() )
        {
            const auto moveResult = *resultIter;

            // マッチ済み領域の後続の空き領域を検索
            const ShiftRangeBounds bounds(
                [&](Container::iterator begin, Container::iterator end) NN_NOEXCEPT
                {
                    return std::lower_bound(
                        begin,
                        end,
                        moveResult.GetOldEndOffset(),
                        [](const ShiftRange& range, int64_t offset) NN_NOEXCEPT
                        {
                            return range.GetOffset() < offset;
                        }
                    );
                },
                freeBounds.begin(),
                freeBounds.end()
            );

            // マッチ済み領域の後続に空き領域がある
            if( !bounds.empty() )
            {
                for( auto& freeRange : bounds )
                {
                    const auto free = freeRange;

                    if( move.GetSize() <= free.GetSize() )
                    {
                        m_Results.push_back(MatchResult::MakeSmallFit(
                            move.GetOffset(), free.GetOffset(), move.GetSize()));

                        freeRange.Advance(move.GetSize());
                        moveRange.Empty();

                        break;
                    }
                }
            }
        }
    }

    // 移動領域に合う空き領域がなかったものを先頭から検索し直し
    for( auto& moveRange : moveBounds )
    {
        const auto move = moveRange;

        if( 0 < move.GetSize() )
        {
            for( auto& freeRange : freeBounds )
            {
                auto free = freeRange;

                if( move.GetSize() <= free.GetSize() )
                {
                    m_Results.push_back(MatchResult::MakeSmallFit(
                        move.GetOffset(), free.GetOffset(), move.GetSize()));

                    freeRange.Advance(move.GetSize());
                    moveRange.Empty();

                    break;
                }
            }
        }
    }

    Resize(&m_FreeRanges);
    Resize(&m_MoveRanges);
} // NOLINT(impl/function_size)

// 大きな BinaryMatchResult を分割して移動する
void RelocationShiftRangeContainer::ShiftByLargeSplit() NN_NOEXCEPT
{
    if( m_FreeRanges.empty() || m_MoveRanges.empty() )
    {
        return;
    }

    // 小さい空き領域を退避
    SetupSplit();

    if( !m_FreeRanges.empty() )
    {
        SortOrderGreaterSize(&m_MoveRanges);

        // サイズの降順で処理をする
        for( auto& moveRange : m_MoveRanges )
        {
            if( LargeSplitSizeMin <= moveRange.GetSize() )
            {
                if( ShiftByLargeSplit(moveRange) )
                {
                    moveRange.Empty();
                }
            }
            else
            {
                break;
            }

            if( m_FreeRanges.empty() )
            {
                break;
            }
        }

        Resize(&m_MoveRanges);
    }

    // 小さい空き領域を復帰
    CleanupSplit();
}

bool RelocationShiftRangeContainer::ShiftByLargeSplit(const ShiftRange& moveRange) NN_NOEXCEPT
{
    // サイズの昇順に並べる
    SortOrderLessSize(&m_FreeRanges);

    ShiftRangeCombiner combiner;
    const auto moveSize = moveRange.GetSize();
    const auto begin = m_FreeRanges.begin();
    const auto end = m_FreeRanges.end();
    auto last = end;
    auto first = end;

    // ソートした空き領域の末尾（サイズの大きいもの）から割り当てていく
    while( begin < first )
    {
        --first;

        const auto& free = *first;
        NN_SDK_ASSERT_GREATER_EQUAL(free.GetSize(), SplitSizeMin);

        combiner.Expand(free);

        if( moveSize <= combiner.GetSize() )
        {
            std::sort(first, last);

            --last;

            break;
        }

        NN_SDK_ASSERT_GREATER(moveSize, combiner.GetSize());

        // 最大分割数を超えていたら先頭をずらす
        if( std::distance(first, last) == SplitCountMax )
        {
            combiner.Reduce(*first);
            ++first;

            std::sort(first, last);

            break;
        }
    }

    SplitCommon(&m_Results, moveRange, first, last, BinaryMatchDetail_LargeSplit);

    return true;
}

// 分割しないで BinaryMatchResult を前詰めしていく
void RelocationShiftRangeContainer::ShiftByLeftJustify() NN_NOEXCEPT
{
    if( m_FreeRanges.empty() || m_MoveRanges.empty() )
    {
        return;
    }

    // オフセットの昇順に並べる
    Sort(&m_FreeRanges);
    Sort(&m_MoveRanges);

    // 分割しないで空き領域に前詰めしていく
    for( auto& freeRange : m_FreeRanges )
    {
        auto free = freeRange;

        for( auto& moveRange : m_MoveRanges )
        {
            const auto move = moveRange;

            // 入れられる領域を詰めていく
            if( move.GetSize() <= free.GetSize() )
            {
                moveRange.Empty();

                m_Results.push_back(MatchResult::MakeLeftJustify(
                    move.GetOffset(), free.GetOffset(), move.GetSize()));

                free.Advance(move.GetSize());

                if( free.GetSize() == 0 )
                {
                    break;
                }
            }
        }

        if( freeRange.GetSize() != free.GetSize() )
        {
            freeRange = free;

            Resize(&m_MoveRanges);
            Sort(&m_MoveRanges);

            if( m_MoveRanges.empty() )
            {
                break;
            }
        }
    }

    Resize(&m_FreeRanges);
}

// BinaryMatchResult を分割して前詰めしていく
bool RelocationShiftRangeContainer::ShiftBySoftSplit(int64_t expandedSize) NN_NOEXCEPT
{
    SetupSplit();

    if( m_FreeRanges.empty() || m_MoveRanges.empty() )
    {
        CleanupSplit();

        return GetTotalFreeSize() <= expandedSize;
    }

    Sort(&m_FreeRanges);
    Sort(&m_MoveRanges);

    // この関数呼び出し後は m_FreeRanges を使用しないので原状復帰しなくてよい。
    // 一方、分割できなかった時のために m_Results,m_MoveRanges は原状復帰する必要がある。
    std::vector<MatchResult> results;
    {
        std::vector<ShiftRange> backup;

        // 分割して空き領域に前詰めしていく
        for( auto& moveRange : m_MoveRanges )
        {
            if( ShiftBySoftSplit(&results, moveRange) )
            {
                backup.push_back(moveRange);

                moveRange.Empty();
            }
        }
        Resize(&m_MoveRanges);

        CleanupSplit();

        // 拡張可能なサイズより大きくなる場合は ShiftBySoftSplit() をキャンセル
        if( expandedSize < GetTotalFreeSize() )
        {
            std::copy(backup.begin(), backup.end(), std::back_inserter(m_MoveRanges));

            return false;
        }
    }

    // ShiftBySoftSplit() を正式採用
    std::copy(results.begin(), results.end(), std::back_inserter(m_Results));

    return true;
}

bool RelocationShiftRangeContainer::ShiftBySoftSplit(
                                        std::vector<MatchResult>* pResults,
                                        const ShiftRange& moveRange
                                    ) NN_NOEXCEPT
{
    ShiftRangeCombiner combiner;
    const auto moveSize = moveRange.GetSize();
    const auto freeSizeMin = std::min(moveSize, SplitSizeMin);
    const auto end = m_FreeRanges.end();
    auto first = end;
    auto last = m_FreeRanges.begin();

    // 割り当て可能な領域の最初を取得
    while( last < end )
    {
        const auto& free = *last;

        if( freeSizeMin <= free.GetSize() )
        {
            // 割り当て開始位置を設定
            if( first == end )
            {
                first = last;
            }

            combiner.Expand(free);

            // 連続での割り当て可能
            if( moveSize <= combiner.GetSize() )
            {
                break;
            }

            // 余っている部分が端数なので先頭をずらす
            if( moveSize - combiner.GetSize() < SplitSizeMin )
            {
                if( first == last )
                {
                    combiner.Reset(free);
                    first = end;
                }
                else
                {
                    combiner.Reduce(*first);
                    ++first;
                }
            }
        }
        // 連続で確保できない場合はやり直し
        else
        {
            combiner.Reset(free);
            first = end;
        }

        ++last;

        // 最大分割数を超えていたら先頭をずらす
        if( std::distance(first, last) == SplitCountMax )
        {
            combiner.Reduce(*first);
            ++first;
        }
    }

    if( first != end )
    {
        SplitCommon(pResults, moveRange, first, last, BinaryMatchDetail_SoftSplit);

        return true;
    }
    return false;
}

void RelocationShiftRangeContainer::SplitCommon(
                                        std::vector<MatchResult>* pResults,
                                        const ShiftRange& moveRange,
                                        Container::iterator begin,
                                        Container::iterator end,
                                        BinaryMatchDetail detail
                                    ) NN_NOEXCEPT
{
    auto& results = *pResults;
    auto move = moveRange;
    auto last = end;
    int16_t index = 0;

    // 分割して登録していく
    for( auto iter = begin; iter < end; ++iter, ++index )
    {
        const auto free = *iter;

        if( move.GetSize() < free.GetSize() )
        {
            last = iter;
            break;
        }

        results.push_back(MatchResult::MakeSoftSplit(
            move.GetOffset(), free.GetOffset(), free.GetSize(), detail, index));

        move.Advance(free.GetSize());
    }

    // 空き領域の末尾から領域を確保
    if( last == m_FreeRanges.end() )
    {
        results.push_back(MatchResult::MakeSoftSplit(
            move.GetOffset(), m_FreeRangeLastOffset, move.GetSize(), detail, index));

        m_FreeRangeLastOffset += move.GetSize();
    }
    else
    {
        auto free = *last;
        NN_SDK_ASSERT(0 < move.GetSize() && move.GetSize() <= free.GetSize());

        // 末尾を登録
        results.push_back(MatchResult::MakeSoftSplit(
            move.GetOffset(), free.GetOffset(), move.GetSize(), detail, index));

        free.Advance(move.GetSize());

        // 空き領域が小さいので退避する
        if( free.GetSize() < SplitSizeMin )
        {
            if( 0 < free.GetSize() )
            {
                m_FreeBackup.push_back(free);
            }

            ++last; // 末尾も削除対象へ
        }
        else
        {
            *last = free; // 空き領域として修正
        }
    }

    m_FreeRanges.erase(begin, last);
}

// BinaryMatchResult を後詰めしていく
void RelocationShiftRangeContainer::ShiftByRightJustify() NN_NOEXCEPT
{
    if( m_MoveRanges.empty() )
    {
        return;
    }

    const auto last = std::max_element(
        m_Results.begin(),
        m_Results.end(),
        [](const MatchResult& lhs, const MatchResult& rhs) NN_NOEXCEPT
        {
            return lhs.oldOffset < rhs.oldOffset;
        }
    );
    NN_SDK_ASSERT_NOT_EQUAL(last, m_Results.end());

    auto offset = last->GetOldEndOffset();

    for( const auto range : m_MoveRanges )
    {
        m_Results.push_back(
            MatchResult::MakeRightJustify(range.GetOffset(), offset, range.GetSize()));

        offset += range.GetSize();
    }

    m_MoveRanges.clear();
}

void RelocationShiftRangeContainer::SetupSplit() NN_NOEXCEPT
{
    // SplitSizeMin 未満の領域を後方へ移動
    const auto begin = m_FreeRanges.begin();
    const auto end = m_FreeRanges.end();
    const auto point = std::partition(
        begin,
        end,
        [](const ShiftRange& range) NN_NOEXCEPT
        {
            return SplitSizeMin <= range.GetSize();
        }
    );

    // SplitSizeMin 未満の空き領域を退避（領域の無駄な分割を避けるため）
    if( point < end )
    {
        m_FreeBackup.reserve(GetDistance(point, end));
        std::copy(point, end, std::back_inserter(m_FreeBackup));

        m_FreeRanges.resize(GetDistance(begin, point));
    }
}

void RelocationShiftRangeContainer::CleanupSplit() NN_NOEXCEPT
{
    Resize(&m_FreeRanges);

    std::copy(m_FreeBackup.begin(), m_FreeBackup.end(), std::back_inserter(m_FreeRanges));
    m_FreeBackup = Container();
}

// BinaryMatch の結果から oldOffset について重複を取り除く
void OptimizeBinaryMatchResult(std::vector<MatchResult>* pResults) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResults);

    auto& results = *pResults;

    if( results.empty() )
    {
        return;
    }

    // NOTE: newOffset でソートされていることが前提
    const auto newEndOffset = results.back().GetNewEndOffset();

    // oldOffset が有効なリザルトのみを抽出
    {
        // oldOffset でソートしつつ InvalidOffset は後ろに移動する
        std::sort(results.begin(), results.end(), PartitionOrderFunctor());

        const auto end =
            std::partition_point(results.begin(), results.end(), PartitionPointFunctor());

        // oldOffset == InvalidOffset は切り落とす
        results.resize(GetDistance(results.begin(), end));
    }

    // 重複をなくしたリザルトに書き換え
    {
        size_t resultIndex = 0;
        int64_t oldEndOffset = 0;

        for( const auto& result : results )
        {
            // 最後の要素とつながっていなければそのまま上書き
            if( oldEndOffset <= result.oldOffset )
            {
                results[resultIndex] = result;
                results[resultIndex].detail = BinaryMatchDetail_Match;
            }
            else
            {
                // 最後の要素と全て被っていたら何もしない
                if( result.oldOffset + result.size <= oldEndOffset )
                {
                    continue;
                }

                // この時点で
                //     result.oldOffset < oldEndOffset &&
                //     oldEndOffset < result.oldOffset + result.size
                // なので
                //     0 < oldEndOffset - result.oldOffset < result.size
                // が成り立つ
                const auto overlapSize = oldEndOffset - result.oldOffset;

                // 被っている分を追加する方から削って上書きする
                results[resultIndex] = MatchResult::MakeMatch(
                                                        result.newOffset + overlapSize,
                                                        result.oldOffset + overlapSize,
                                                        result.size - overlapSize
                                                    );
            }

            oldEndOffset = results[resultIndex].GetOldEndOffset();

            ++resultIndex;
        }

        // 余分を切り落とし
        results.resize(resultIndex);
    }

    // newOffset でソートする
    std::sort(results.begin(), results.end());

    // newOffset の空いた領域を補完する
    {
        int64_t endOffset = 0;

        const auto resultCount = results.size();
        for( size_t i = 0; i < resultCount; ++i )
        {
            const auto current = results[i];
            if( endOffset < current.newOffset )
            {
                results.push_back(
                    MatchResult::MakeUnknown(endOffset, current.newOffset - endOffset));
            }

            endOffset = current.GetNewEndOffset();
        }

        if( endOffset < newEndOffset )
        {
            results.push_back(
                MatchResult::MakeUnknown(endOffset, newEndOffset - endOffset));
        }
    }
}

// BinaryMatchResult の分割・移動を行う
void ShiftBinaryMatchResult(
         std::vector<MatchResult>* pResults,
         int64_t limitOldOffset,
         int64_t limitNewOffset,
         int64_t expandedSize,
         BinaryMatchProgress* pProgress
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pResults);
    NN_SDK_REQUIRES(!pResults->empty());
    NN_SDK_REQUIRES_GREATER(limitOldOffset, 0);
    NN_SDK_REQUIRES_GREATER(limitNewOffset, 0);
    NN_SDK_REQUIRES_RANGE(expandedSize, 0, limitOldOffset);
    NN_SDK_REQUIRES_NOT_NULL(pProgress);

    ShiftRangeContainer shiftRanges(pResults, limitOldOffset);

    if( shiftRanges.Build(expandedSize) )
    {
        shiftRanges.ShiftByAnchor(limitNewOffset);

        pProgress->SetValue(int(BinaryMatchOptimizePhase::ShiftByLargeSplit));
        shiftRanges.ShiftByLargeSplit();

        pProgress->SetValue(int(BinaryMatchOptimizePhase::ShiftBySmallFit));
        shiftRanges.ShiftBySmallFit();

        pProgress->SetValue(int(BinaryMatchOptimizePhase::ShiftByLeftJustify));
        shiftRanges.ShiftByLeftJustify();

        pProgress->SetValue(int(BinaryMatchOptimizePhase::ShiftBySoftSplit));
        if( shiftRanges.ShiftBySoftSplit(expandedSize) )
        {
            shiftRanges.ShiftByRightJustify();
        }
    }
}

}}}
