﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fssystem/fs_StorageFile.h>
#include <nn/fssystem/fs_ZeroBitmapFile.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>
#include <nn/fssystem/dbm/fs_DbmPathTool.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_SaveDataInternalStorageFileSystem.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>

namespace nn { namespace fssystem { namespace save {

namespace
{

// 指定したセーブデータ内部ストレージが存在するかどうかを探す Visitor
class SaveDataInternalStorageStorageFinder : public IInternalStorageFileSystemVisitor
{
public:
    explicit SaveDataInternalStorageStorageFinder(const char* name) NN_NOEXCEPT
        : m_Name(name),
          m_IsFound(false)
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
    }

    virtual ~SaveDataInternalStorageStorageFinder() NN_NOEXCEPT NN_OVERRIDE {}

public:
    virtual Result VisitStorage(const char* name, fs::IStorage* pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Name);
        NN_SDK_REQUIRES_NOT_NULL(name);

        NN_UNUSED(pStorage);

        if( std::strcmp(name, m_Name) == 0 )
        {
            m_IsFound = true;
        }

        NN_RESULT_SUCCESS;
    }

    bool IsFound() const NN_NOEXCEPT
    {
        return m_IsFound;
    }

private:
    const char* m_Name;
    bool m_IsFound;
};

// 指定したセーブデータ内部ストレージに対して IFile インターフェースでアクセスする Visitor
template<typename Function>
class SaveDataInternalStorageStorageFileAccessor : public IInternalStorageFileSystemVisitor
{
public:
    SaveDataInternalStorageStorageFileAccessor(const char* name, fs::OpenMode mode, Function fileAccessor) NN_NOEXCEPT
        : m_Name(name),
          m_Mode(mode),
          m_pJournalStorage(nullptr),
          m_FileAccessor(fileAccessor)
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
    }

    virtual ~SaveDataInternalStorageStorageFileAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_pJournalStorage != nullptr )
        {
            m_pJournalStorage->SetMappingTableFrozon(false);
        }
    }

public:
    virtual Result VisitStorage(const char* name, fs::IStorage* pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        NN_SDK_REQUIRES_NOT_NULL(m_Name);

        if( std::strcmp(name, JournalIntegritySaveDataStorage::InternalStorageFileNameJournal) == 0 )
        {
            m_pJournalStorage = reinterpret_cast<save::JournalStorage*>(pStorage);
            m_pJournalStorage->SetMappingTableFrozon(true);
        }
        else if( std::strcmp(m_Name, name) == 0 )
        {
            if( std::strcmp(name, JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrity) == 0)
            {
                HierarchicalIntegrityVerificationStorage* pIntegrityStorage = reinterpret_cast<HierarchicalIntegrityVerificationStorage*>(pStorage);
                fs::SubStorage storage = pIntegrityStorage->GetL1HashStorage();
                fssystem::StorageFile file(&storage, m_Mode);
                NN_RESULT_DO(m_FileAccessor(&file));
            }
            else
            {
                fssystem::StorageFile file(pStorage, m_Mode);
                NN_RESULT_DO(m_FileAccessor(&file));
            }
        }
        NN_RESULT_SUCCESS;
    }

private:
    const char* m_Name;
    fs::OpenMode m_Mode;
    save::JournalStorage* m_pJournalStorage;
    Function m_FileAccessor;
};

// セーブデータ内部ストレージを列挙する Visitor
class SaveDataInternalStorageStorageEnumerator : public IInternalStorageFileSystemVisitor
{
public:
    SaveDataInternalStorageStorageEnumerator(int64_t position, fs::DirectoryEntry* pEntry, int64_t count) NN_NOEXCEPT
        : m_Position(position),
          m_pEntry(pEntry),
          m_Count(count),
          m_Index(0)
    {
        NN_SDK_REQUIRES_LESS_EQUAL(0, position);
        NN_SDK_REQUIRES_NOT_NULL(pEntry);
        NN_SDK_REQUIRES_LESS_EQUAL(0, count);
    }

    virtual ~SaveDataInternalStorageStorageEnumerator() NN_NOEXCEPT NN_OVERRIDE {}

public:
    virtual Result VisitStorage(const char* name, fs::IStorage* pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        NN_SDK_REQUIRES_LESS_EQUAL(0, m_Position);
        NN_SDK_REQUIRES_LESS_EQUAL(0, m_Index);

        if( std::strcmp(name, JournalIntegritySaveDataStorage::InternalStorageFileNameJournal) == 0 )
        {
            NN_RESULT_SUCCESS;
        }

        if( m_Position > 0 )
        {
            --m_Position;
            NN_RESULT_SUCCESS;
        }

        if( m_pEntry != nullptr )
        {
            if( m_Index >= m_Count )
            {
                NN_RESULT_SUCCESS;
            }

            std::memset(&m_pEntry[m_Index], 0, sizeof(m_pEntry[m_Index]));
            m_pEntry[m_Index].directoryEntryType = fs::DirectoryEntryType_File;
            util::Strlcpy(m_pEntry[m_Index].name, name, sizeof(m_pEntry[m_Index].name));

            NN_RESULT_DO(pStorage->GetSize(&m_pEntry[m_Index].fileSize));
        }

        ++m_Index;
        NN_SDK_REQUIRES_LESS_EQUAL(0, m_Index);

        NN_RESULT_SUCCESS;
    }

    int64_t GetReadCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS_EQUAL(0, m_Index);
        return m_Index;
    }

private:
    int64_t m_Position;
    fs::DirectoryEntry* m_pEntry;
    int64_t m_Count;
    int64_t m_Index;
};

}

namespace detail {

// 未割り当て領域ビットマップ
template<typename Function>
class SaveDataInternalStorageFreeBitmap : public IInternalStorageFileSystemVisitor
{
public:
    SaveDataInternalStorageFreeBitmap(
        SaveDataInternalStorageFileSystem* pFileSystem, Function bitmap) NN_NOEXCEPT
        : m_pFileSystem(pFileSystem),
          m_Bitmap(bitmap)
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
    }

    virtual ~SaveDataInternalStorageFreeBitmap() NN_NOEXCEPT NN_OVERRIDE {}

public:
    virtual Result VisitStorage(const char* name, fs::IStorage* pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        if( std::strcmp(name, SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta) == 0 )
        {
            int64_t size;
            pStorage->GetSize(&size);

            const size_t sizeOfBlock = 8;
            const size_t bitsPerByte = 8;
            int32_t blocks = static_cast<int32_t>(nn::util::DivideUp(size, sizeOfBlock));

            nn::fssystem::dbm::AllocationTable table;
            table.Initialize(nn::fs::SubStorage(pStorage, 0, size), blocks);

            size_t bitmapSize = nn::util::DivideUp(blocks, bitsPerByte);
            std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pBitmap;
            pBitmap = fs::detail::MakeUnique<Bit8[]>(bitmapSize);
            NN_RESULT_THROW_UNLESS(pBitmap, nn::fs::ResultAllocationMemoryFailedMakeUnique());
            memset(pBitmap.get(), 0, bitmapSize);

            uint32_t index;
            if (table.ReadFreeListHead(&index).IsSuccess())
            {
                // リストを辿っていく
                uint32_t indexNext;
                uint32_t blockCount;
                while(
                    table.IndexEnd != index &&
                    table.ReadNext(&indexNext, &blockCount, index).IsSuccess()
                )
                {
                    if(index + blockCount > bitmapSize * bitsPerByte)
                    {
                        NN_RESULT_THROW(nn::fs::ResultInvalidSaveDataAllocationTableBlock());
                    }

                    for(uint32_t i = 0;i < blockCount; i++)
                    {
                        pBitmap.get()[(index + i) / bitsPerByte] |= (1 << ((index + i) % bitsPerByte));
                    }
                    index = indexNext;
                }
            }
            m_Bitmap(std::move(pBitmap), bitmapSize);
        }

        NN_RESULT_SUCCESS;
    }

private:
    SaveDataInternalStorageFileSystem* m_pFileSystem;
    Function m_Bitmap;
};

// 階層ハッシュ情報
template<typename Function>
class IntegrityVerificationInternalStorageSizeVerificationBlock : public IInternalStorageFileSystemVisitor
{
public:
    IntegrityVerificationInternalStorageSizeVerificationBlock(
        SaveDataInternalStorageFileSystem* pFileSystem, Function block) NN_NOEXCEPT
        : m_pFileSystem(pFileSystem),
          m_SizeVerificationBlock(block)
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
    }

    virtual ~IntegrityVerificationInternalStorageSizeVerificationBlock() NN_NOEXCEPT NN_OVERRIDE {}

public:
    virtual Result VisitStorage(const char* name, fs::IStorage* pStorage) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(name);
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        if( std::strcmp(name, JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrity) == 0)
        {
            HierarchicalIntegrityVerificationStorage* pIntegrityStorage = reinterpret_cast<HierarchicalIntegrityVerificationStorage*>(pStorage);
            int64_t block = pIntegrityStorage->GetL1HashVerificationBlockSize();
            m_SizeVerificationBlock(block);
        }

        NN_RESULT_SUCCESS;
    }

private:
    SaveDataInternalStorageFileSystem* m_pFileSystem;
    Function m_SizeVerificationBlock;
};

// セーブデータのための内部ストレージアクセス用疑似ファイル
class SaveDataInternalStorageFile : public fs::fsa::IFile, public fs::detail::Newable
{
    NN_DISALLOW_COPY(SaveDataInternalStorageFile);

public:
    SaveDataInternalStorageFile(
        SaveDataInternalStorageFileSystem* pFileSystem,
        const char* name,
        fs::OpenMode mode
    ) NN_NOEXCEPT
        : m_pFileSystem(pFileSystem),
          m_Mode(mode)
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
        NN_SDK_REQUIRES_NOT_NULL(name);

        auto length = util::Strlcpy(m_Name, name, sizeof(m_Name));
        NN_SDK_ASSERT_LESS(static_cast<size_t>(length), sizeof(m_Name));
        NN_UNUSED(length);
    }

    virtual ~SaveDataInternalStorageFile() NN_NOEXCEPT NN_OVERRIDE {}

private:
    virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    SaveDataInternalStorageFileSystem* m_pFileSystem;
    fs::OpenMode m_Mode;
    char m_Name[SaveDataInternalStorageFileSystem::PathLengthMax];
};

// セーブデータのための内部ストレージアクセス用疑似ディレクトリ
class SaveDataInternalStorageDirectory : public fs::fsa::IDirectory, public fs::detail::Newable
{
    NN_DISALLOW_COPY(SaveDataInternalStorageDirectory);

public:
    SaveDataInternalStorageDirectory(
        SaveDataInternalStorageFileSystem* pFileSystem,
        fs::OpenDirectoryMode mode
    ) NN_NOEXCEPT
        : m_pFileSystem(pFileSystem),
          m_Mode(mode),
          m_ReadCount(0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
    }

    virtual ~SaveDataInternalStorageDirectory() NN_NOEXCEPT NN_OVERRIDE {}

private:
    virtual Result DoRead(int64_t* outValue, fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

private:
    SaveDataInternalStorageFileSystem* m_pFileSystem;
    fs::OpenDirectoryMode m_Mode;
    int64_t m_ReadCount;
    char m_Path[SaveDataInternalStorageFileSystem::PathLengthMax];
};

Result SaveDataInternalStorageFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto reader = [=](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->Read(outValue, offset, buffer, size, option));
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(reader)> fileAccessor(m_Name, m_Mode, reader);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFile::DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto writer = [=](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->Write(offset, buffer, size, option));
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(writer)> fileAccessor(m_Name, m_Mode, writer);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFile::DoFlush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto flusher = [](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->Flush());
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(flusher)> fileAccessor(m_Name, m_Mode, flusher);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto sizeSetter = [=](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->SetSize(size));
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(sizeSetter)> fileAccessor(m_Name, m_Mode, sizeSetter);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto sizeGetter = [=](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->GetSize(outValue));
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(sizeGetter)> fileAccessor(m_Name, m_Mode, sizeGetter);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    auto rangeOperator = [=](fs::fsa::IFile* pFile) NN_NOEXCEPT -> Result
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_RESULT_DO(pFile->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
        NN_RESULT_SUCCESS;
    };
    SaveDataInternalStorageStorageFileAccessor<decltype(rangeOperator)> fileAccessor(m_Name, m_Mode, rangeOperator);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&fileAccessor));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageDirectory::DoRead(int64_t* outValue, fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(outEntries);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);
    NN_SDK_REQUIRES_LESS_EQUAL(0, m_ReadCount);

    SaveDataInternalStorageStorageEnumerator enumerator(m_ReadCount, outEntries, count);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&enumerator));

    m_ReadCount += enumerator.GetReadCount();
    NN_SDK_REQUIRES_LESS_EQUAL(0, m_ReadCount);
    *outValue = enumerator.GetReadCount();

    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageDirectory::DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_LESS_EQUAL(0, m_ReadCount);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem);
    NN_SDK_REQUIRES_NOT_NULL(m_pFileSystem->m_pInternalStorageFileSystem);

    SaveDataInternalStorageStorageEnumerator enumerator(m_ReadCount, nullptr, 0);
    NN_RESULT_DO(m_pFileSystem->m_pInternalStorageFileSystem->AcceptVisitor(&enumerator));
    NN_SDK_REQUIRES_LESS_EQUAL(enumerator.GetReadCount(), m_ReadCount);
    *outValue = enumerator.GetReadCount() - m_ReadCount;
    NN_RESULT_SUCCESS;
}

}

Result SaveDataInternalStorageFileSystem::Initialize(
    IInternalStorageFileSystem* pInternalStorageFileSystem
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pInternalStorageFileSystem == nullptr);
    NN_SDK_REQUIRES_NOT_NULL(pInternalStorageFileSystem);
    m_pInternalStorageFileSystem = pInternalStorageFileSystem;
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFileSystem::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_UNUSED(size);
    NN_UNUSED(option);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoDeleteFile(const char* path) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoCreateDirectory(const char* path) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoDeleteDirectory(const char* path) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
{
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    NN_UNUSED(currentPath);
    NN_UNUSED(newPath);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
{
    NN_UNUSED(currentPath);
    NN_UNUSED(newPath);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoGetEntryType(fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
{
    NN_UNUSED(outValue);
    NN_UNUSED(path);
    NN_RESULT_THROW(fs::ResultNotImplemented());
}

Result SaveDataInternalStorageFileSystem::DoOpenFile(std::unique_ptr<fs::fsa::IFile>* outValue, const char* path, fs::OpenMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(m_pInternalStorageFileSystem);

    dbm::PathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(path));

    dbm::DirectoryName directoryName;
    size_t length;
    NN_RESULT_DO(parser.GetNextDirectoryName(&directoryName, &length));

    dbm::FileName fileName;
    NN_RESULT_DO(parser.GetAsFileName(&fileName, &length));

    SaveDataInternalStorageStorageFinder storageFinder(fileName.name);
    NN_RESULT_DO(m_pInternalStorageFileSystem->AcceptVisitor(&storageFinder));

    if( !storageFinder.IsFound() )
    {
        if( std::strcmp(fileName.name, SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree) == 0 )
        {
            std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pFreeBitmap;
            int64_t freeBitmapSize;
            auto bitmap = [&](std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pBitmap, int64_t bitmapSize) NN_NOEXCEPT -> Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pBitmap);
                pFreeBitmap = std::move(pBitmap);
                freeBitmapSize = bitmapSize;
                NN_RESULT_SUCCESS;
            };
            detail::SaveDataInternalStorageFreeBitmap<decltype(bitmap)> freeBitmap(this, bitmap);
            NN_RESULT_DO(m_pInternalStorageFileSystem->AcceptVisitor(&freeBitmap));
            std::unique_ptr<detail::SaveDataInternalStorageFile> pFile(new detail::SaveDataInternalStorageFile(this, SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData, mode));
            NN_RESULT_THROW_UNLESS(pFile, nn::fs::ResultAllocationMemoryFailedNew());

            const size_t BitsPerByte = 8;
            int64_t fileSize;
            NN_RESULT_DO(pFile->GetSize(&fileSize));
            NN_RESULT_THROW_UNLESS(freeBitmapSize >= nn::util::DivideUp(nn::util::DivideUp(fileSize, nn::fs::detail::DefaultSaveDataBlockSize), BitsPerByte), nn::fs::ResultInvalidAllocationTableBlock());

            std::unique_ptr<ZeroBitmapFile> pZeroFile(new ZeroBitmapFile());
            NN_RESULT_THROW_UNLESS(pZeroFile, nn::fs::ResultAllocationMemoryFailedNew());
            NN_RESULT_DO(
                pZeroFile->Initialize(
                    std::move(pFile),
                    std::move(pFreeBitmap),
                    nn::util::DivideUp(nn::util::DivideUp(fileSize, nn::fs::detail::DefaultSaveDataBlockSize), BitsPerByte),
                    nn::fs::detail::DefaultSaveDataBlockSize,
                    mode));
            *outValue = std::move(pZeroFile);

            NN_RESULT_SUCCESS;
        }
        else if( std::strcmp(fileName.name, JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrityWithZeroFree) == 0 )
        {
            NN_RESULT_THROW_UNLESS((mode == nn::fs::OpenMode_Read), nn::fs::ResultInvalidOperationForOpenMode());

            // ハッシュブロックサイズの取得
            int64_t sizeVerificationBlock;
            auto getBlock = [&](int64_t block) NN_NOEXCEPT -> Result
            {
                sizeVerificationBlock = block;
                NN_RESULT_SUCCESS;
            };
            detail::IntegrityVerificationInternalStorageSizeVerificationBlock<decltype(getBlock)> integrityVerificationInternalStorageSizeVerificationBlock(this, getBlock);
            NN_RESULT_DO(m_pInternalStorageFileSystem->AcceptVisitor(&integrityVerificationInternalStorageSizeVerificationBlock));
            NN_RESULT_THROW_UNLESS((sizeVerificationBlock <= nn::fs::detail::DefaultSaveDataBlockSize), nn::fs::ResultIntegrityVerificationStorageCorrupted());

            std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pFreeBitmap;
            int64_t freeBitmapSize;
            auto bitmap = [&](std::unique_ptr<Bit8[], nn::fs::detail::Deleter> pBitmap, int64_t bitmapSize) NN_NOEXCEPT -> Result
            {
                NN_SDK_REQUIRES_NOT_NULL(pBitmap);
                pFreeBitmap = std::move(pBitmap);
                freeBitmapSize = bitmapSize;
                NN_RESULT_SUCCESS;
            };
            detail::SaveDataInternalStorageFreeBitmap<decltype(bitmap)> freeBitmap(this, bitmap);
            NN_RESULT_DO(m_pInternalStorageFileSystem->AcceptVisitor(&freeBitmap));

            const size_t BitsPerByte = 8;
            int64_t fileSize;
            {
                std::unique_ptr<detail::SaveDataInternalStorageFile> pFile(new detail::SaveDataInternalStorageFile(this, SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData, mode));
                NN_RESULT_THROW_UNLESS(pFile, nn::fs::ResultAllocationMemoryFailedNew());

                NN_RESULT_DO(pFile->GetSize(&fileSize));
                NN_RESULT_THROW_UNLESS(freeBitmapSize >= nn::util::DivideUp(nn::util::DivideUp(fileSize, nn::fs::detail::DefaultSaveDataBlockSize), BitsPerByte), nn::fs::ResultInvalidAllocationTableBlock());
            }

            std::unique_ptr<detail::SaveDataInternalStorageFile> pFile(new detail::SaveDataInternalStorageFile(this, JournalIntegritySaveDataStorage::InternalStorageFileNameIntegrity, mode));
            NN_RESULT_THROW_UNLESS(pFile, nn::fs::ResultAllocationMemoryFailedNew());

            std::unique_ptr<ZeroBitmapFile> pZeroFile(new ZeroBitmapFile());
            NN_RESULT_THROW_UNLESS(pZeroFile, nn::fs::ResultAllocationMemoryFailedNew());
            NN_RESULT_DO(
                pZeroFile->Initialize(
                    std::move(pFile),
                    std::move(pFreeBitmap),
                    nn::util::DivideUp(nn::util::DivideUp(fileSize, nn::fs::detail::DefaultSaveDataBlockSize), BitsPerByte),
                    HashSize * (nn::fs::detail::DefaultSaveDataBlockSize / sizeVerificationBlock),
                    mode));
            *outValue = std::move(pZeroFile);

            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_RESULT_THROW(fs::ResultPathNotFound());
        }
    }
    else
    {
        if( std::strcmp(fileName.name, JournalIntegritySaveDataFileSystem::InternalStorageFileNameIntegritySeed) == 0 )
        {
            NN_RESULT_THROW_UNLESS((mode == nn::fs::OpenMode_Read), nn::fs::ResultInvalidOperationForOpenMode());
        }
    }

    std::unique_ptr<detail::SaveDataInternalStorageFile> pFile(new detail::SaveDataInternalStorageFile(this, fileName.name, mode));
    *outValue = std::move(pFile);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result SaveDataInternalStorageFileSystem::DoOpenDirectory(std::unique_ptr<fs::fsa::IDirectory>* outValue, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(path);

    dbm::PathTool::PathParser parser;
    NN_RESULT_DO(parser.Initialize(path));

    dbm::DirectoryName directoryName;
    size_t length;
    NN_RESULT_DO(parser.GetNextDirectoryName(&directoryName, &length));
    NN_RESULT_DO(parser.GetAsDirectoryName(&directoryName, &length));

    // ルート以外のディレクトリはない
    if( length != 0 )
    {
        NN_RESULT_THROW(fs::ResultPathNotFound());
    }

    std::unique_ptr<detail::SaveDataInternalStorageDirectory> pDirectory(new detail::SaveDataInternalStorageDirectory(this, mode));
    *outValue = std::move(pDirectory);

    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFileSystem::DoCommit() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInternalStorageFileSystem);

    NN_RESULT_DO(m_pInternalStorageFileSystem->Commit());
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInternalStorageFileSystem);

    NN_RESULT_DO(m_pInternalStorageFileSystem->CommitProvisionally(counter));
    NN_RESULT_SUCCESS;
}

Result SaveDataInternalStorageFileSystem::DoRollback() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInternalStorageFileSystem);

    NN_RESULT_DO(m_pInternalStorageFileSystem->Rollback());
    NN_RESULT_SUCCESS;
}

}}}
