﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_BinTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fssystem/save/fs_RemapStorage.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    //! 初期化済みコード 'RMAP'
    static const uint32_t MagicCode = NN_UTIL_CREATE_SIGNATURE_4('R','M','A','P');

    //! バージョン
    static const uint32_t VersionCode = 0x00010000;

    //! バージョンのマスク
    static const uint32_t MajorVersionMask = 0xffff0000;
    static const uint32_t MinorVersionMask NN_IS_UNUSED_MEMBER = 0x0000ffff;
}

NN_DEFINE_STATIC_CONSTANT(const int64_t RemapStorage::AlignmentSmall);
NN_DEFINE_STATIC_CONSTANT(const int64_t RemapStorage::AlignmentLarge);
NN_DEFINE_STATIC_CONSTANT(const int64_t RemapStorage::SizeBlockSmall);
NN_DEFINE_STATIC_CONSTANT(const int64_t RemapStorage::SizeBlockLarge);
NN_DEFINE_STATIC_CONSTANT(const int RemapStorage::CountMaxStorages);

/**
* @brief        メタデータサイズを取得します。
*
* @return       メタデータサイズを返します。
*
* @details      メタデータサイズを取得します。
*/
int64_t RemapStorage::QueryMetaSize() NN_NOEXCEPT
{
    return sizeof(ControlArea);
}

/**
* @brief        マッピングを指定の回数更新できるエントリテーブルを格納するために必要なサイズを取得します。
*
* @param[in]    countMapUpdate  マッピングを更新できる回数の最大
*
* @return       エントリテーブルを格納するために必要なサイズを返します。
*
* @pre
*               - 0 < countMapUpdate
*
* @details      マッピングを指定の回数更新できるエントリテーブルを格納するために必要なサイズを取得します。
*/
int64_t RemapStorage::QueryEntryTableSize(int countMapUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(countMapUpdate, 0);
    // 最大で (countMapUpdate - 1) / 2 個のエントリが余分に消費される可能性がある
    return (countMapUpdate + (countMapUpdate - 1) / 2) * sizeof(MapEntry);
}

/**
* @brief        指定のサイズのストレージにエントリを記録した場合に
*               マッピングの登録および拡張を必ず行うことができる回数を取得します。
*
* @param[in]    sizeEntry   エントリを記録するストレージのサイズ
*
* @return       マッピングの登録および拡張を必ず行うことができる回数を返します。
*
* @pre
*               - 0 < sizeEntry
*
* @details      本関数は登録と拡張の合計数を取得します。
*               登録のみの最大回数はフォーマット時のパラメータによって決定され、
*               本関数の返り値よりも少なくなる可能性があります。
*
*               マッピングの登録や拡張時に渡す引数により、
*               本関数の返り値を超える回数の登録や拡張を行える可能性があります。
*/
int64_t RemapStorage::GetMapUpdateCountLowerBound(int64_t sizeEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(sizeEntry, 0);
    // 最大で floor((lower - 1) / 2) 個のエントリが余分に消費される可能性があるため
    // lower + floor((lower - 1) / 2) == upper
    const auto upper = GetMapUpdateCountUpperBound(sizeEntry);
    return (upper * 2 + 1 + (1 - (upper % 2))) / 3;
}

/**
* @brief        指定のサイズのストレージにエントリを記録した場合に
*               マッピングの登録および拡張を行うことができる回数の最大を取得します。
*
* @param[in]    sizeEntry   エントリを記録するストレージのサイズ
*
* @return       マッピングの登録および拡張を行うことができる回数の最大を返します。
*
* @pre
*               - 0 < sizeEntry
*
* @details      本関数は登録と拡張の合計数を取得します。
*               登録のみの最大回数はフォーマット時のパラメータによって決定され、
*               本関数の返り値よりも少なくなる可能性があります。
*
*               マッピングの登録や拡張時に渡す引数により、
*               本関数の返り値未満の回数しか登録や拡張を行えない可能性があります。
*/
int64_t RemapStorage::GetMapUpdateCountUpperBound(int64_t sizeEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(sizeEntry, 0);
    return sizeEntry / sizeof(MapEntry);
}

/**
* @brief        メタデータ領域をフォーマットします。
*
* @param[in]    storageMeta     メタデータファイルを配置するレイヤー
* @param[in]    countMaxMaps    マッピングを新規登録できる回数の最大
*
* @return       関数の処理結果を返します。
*
* @pre
*               - QueryMetaSize() <= storageMeta のサイズ
*               - 0 < countMaxMaps
*
* @details      メタデータ領域をフォーマットします。
*/
Result RemapStorage::Format(
           fs::SubStorage storageMeta,
           int countMaxMaps
       ) NN_NOEXCEPT
{
    int64_t sizeMeta = 0;
    NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
    NN_SDK_REQUIRES_GREATER_EQUAL(sizeMeta, QueryMetaSize());
    NN_SDK_REQUIRES_GREATER(countMaxMaps, 0);

    ControlArea header = { 0 };
    header.magic = MagicCode;
    header.version = VersionCode;
    header.countMapEntries = 0;
    header.countMaps = 0;
    header.countHigherBits = 0;

    auto countMaps = countMaxMaps - 1;
    while( 0 < countMaps )
    {
        ++header.countHigherBits;
        countMaps /= 2;
    }
    ++header.countHigherBits;

    return storageMeta.Write(0, &header, sizeof(ControlArea));
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
RemapStorage::RemapStorage() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_IsControlAreaModified(false)
{
}

/**
* @brief        デストラクタです。
*
* @details      デストラクタです。
*/
RemapStorage::~RemapStorage() NN_NOEXCEPT
{
}

/**
* @brief        再マップレイヤーとしてマウントします。
*
* @param[in]    storageMeta     メタデータファイルを配置するレイヤー
* @param[in]    storageEntry    エントリデータを配置するレイヤー
*
* @return       関数の処理結果を返します。
*
* @pre
*               - 未初期化である
*               - QueryMetaSize() <= storageMeta.GetSize()
*               - 0 < storageEntry.GetSize()
*
* @details      再マップレイヤーとしてマウントします。
*/
Result RemapStorage::Initialize(
           fs::SubStorage storageMeta,
           fs::SubStorage storageEntry
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsInitialized);
    int64_t sizeMeta;
    NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
    NN_SDK_REQUIRES_GREATER_EQUAL(sizeMeta, QueryMetaSize());
    int64_t sizeEntry;
    NN_RESULT_DO(storageEntry.GetSize(&sizeEntry));
    NN_SDK_REQUIRES_GREATER(sizeEntry, 0);

    // メタデータ用ストレージを初期化する
    m_StorageMeta = storageMeta;
    m_StorageEntry = storageEntry;

    // 管理領域を読み込む
    NN_RESULT_DO(m_StorageMeta.Read(0, &m_ControlArea, sizeof(ControlArea)));
    m_IsControlAreaModified = false;

    if( sizeEntry < m_ControlArea.countMapEntries * static_cast<int64_t>(sizeof(MapEntry)) )
    {
        return nn::fs::ResultInvalidSize();
    }

    if( (VersionCode & MajorVersionMask) < (m_ControlArea.version & MajorVersionMask) )
    {
        return nn::fs::ResultUnsupportedVersion();
    }

    if( m_ControlArea.countMapEntries < m_ControlArea.countMaps )
    {
        return nn::fs::ResultInvalidMapEntryCount();
    }

    // ストレージエントリを初期化する
    NN_RESULT_DO(InitializeStorageEntries());

    // マッピングエントリキャッシュを初期化する
    ClearAllCaches();

    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief        アンマウントします。
*
* @details
*/
void RemapStorage::Finalize() NN_NOEXCEPT
{
    ClearAllCaches();

    m_IsInitialized = false;
}

/**
* @brief        マッピングを登録します。
*
* @param[out]   outOffsetVirtual    確定した論理オフセット
* @param[in]    size                登録するサイズ
* @param[in]    alignment           登録するマッピングのアライメント
* @param[in]    storageType         登録するストレージ種別
*
* @return       関数の処理結果を返します。
*
* @pre
*               - outOffsetVirtual != nullptr
*               - alignment == AlignmentSmall || alignment == AlignmentLarge
*
* @details      マッピングを登録します。
*/
Result RemapStorage::RegisterMap(
           int64_t* outOffsetVirtual,
           int64_t size,
           int64_t alignment,
           int storageType
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outOffsetVirtual);
    NN_SDK_REQUIRES(alignment == AlignmentSmall || alignment == AlignmentLarge);

    auto offsetVirtual = MakeVirtualOffset(m_ControlArea.countMaps, 0);

    // サイズが期待する範囲の外ならエラー
    if( size <= 0 || GetMapSizeMax() < size )
    {
        return nn::fs::ResultInvalidSize();
    }

    // 登録数があふれたらエラー
    if( m_ControlArea.countMaps != 0 && offsetVirtual <= 0 )
    {
        return nn::fs::ResultMapFull();
    }

    // 登録する
    NN_RESULT_DO(
        RegisterMapCore(
            offsetVirtual,
            size,
            0,
            alignment,
            storageType
        )
    );

    ++m_ControlArea.countMaps;
    m_IsControlAreaModified = true;
    *outOffsetVirtual = offsetVirtual;

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングを拡張します。
*
* @param[in]    virtualOffset   拡張するマッピングの論理オフセット
* @param[in]    size            拡張後のサイズ
*
* @return       関数の処理結果を返します。
*
* @details      マッピングを拡張します。
*/
Result RemapStorage::ExpandMap(int64_t offsetVirtual, int64_t size) NN_NOEXCEPT
{
    MapIterator iterator(this, offsetVirtual);
    int64_t sizeCurrent = 0;

    // 先頭のオフセットが指定されていなければエラー
    if( (offsetVirtual & GetVirtualOffsetLowerBitsMask()) != 0 )
    {
        return nn::fs::ResultInvalidOffset();
    }

    // 指定の論理オフセットに紐づけられた領域の合計サイズを取得する
    const auto sizeMax = GetMapSizeMax();

    while( iterator.MoveNext(sizeMax - sizeCurrent) )
    {
        sizeCurrent += iterator.GetMapEntry().size;
    }

    NN_RESULT_THROW_UNLESS(iterator.GetLastResult().IsSuccess(), iterator.GetLastResult());

    if( sizeCurrent < 0 )
    {
        return nn::fs::ResultInvalidMapSize();
    }

    if( sizeCurrent == 0 )
    {
        return nn::fs::ResultInvalidOffset();
    }

    if( size == sizeCurrent )
    {
        NN_RESULT_SUCCESS;
    }
    else if( size < sizeCurrent || GetMapSizeMax() < size )
    {
        return nn::fs::ResultInvalidSize();
    }

    // 登録する
    return RegisterMapCore(
               offsetVirtual,
               size,
               sizeCurrent,
               iterator.GetMapEntry().alignment,
               iterator.GetMapEntry().storageType
           );
}

/**
* @brief        ストレージを登録します。
*
* @param[in]    storage     登録するストレージ
* @param[in]    storageType 登録するストレージ種別
*
* @return       関数の処理結果を返します。
*
* @pre
*               - 0 <= storageType < CountMaxStorages
*               - 指定の storageType で RegisterStorage に成功していない
*
* @details
*/
Result RemapStorage::RegisterStorage(fs::SubStorage storage, int storageType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(storageType, 0);
    NN_SDK_REQUIRES_LESS(storageType, CountMaxStorages);

    m_StorageEntries[storageType].SetStorage(storage);

    const auto tailSmall = m_StorageEntries[storageType].GetOffsetPhysicalNextSmall();
    const auto tailLarge = m_StorageEntries[storageType].GetOffsetPhysicalNextLarge();
    const auto sizeUsed
        = nn::util::align_up(tailSmall, AlignmentLarge) < tailLarge ? tailLarge : tailSmall;
    int64_t sizeStorage = 0;
    NN_RESULT_DO(storage.GetSize(&sizeStorage));
    if( sizeStorage < sizeUsed )
    {
        return nn::fs::ResultInvalidSize();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        登録できるマッピングサイズの最大を取得します。
*
* @return       登録できるマッピングサイズの最大を返します。
*
* @details      登録できるマッピングサイズの最大を取得します。
*/
int64_t RemapStorage::GetMapSizeMax() const NN_NOEXCEPT
{
    return MakeVirtualOffset(1, 0);
}

/**
* @brief        ストレージの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      ストレージの内容をバッファに読み込みます。
*/
Result RemapStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    NN_RESULT_DO(IterateMappingEntries(
        offset,
        static_cast<int64_t>(size),
        [=](fs::IStorage* pStorage, int64_t offsetPhysical, int64_t sizeAccess, int64_t sizeAccessed) NN_NOEXCEPT -> nn::Result
        {
            NN_SDK_ASSERT_LESS_EQUAL(sizeAccess, static_cast<int64_t>(size));
            NN_SDK_ASSERT_LESS_EQUAL(sizeAccessed, static_cast<int64_t>(size));
            const size_t sizeReading = static_cast<size_t>(sizeAccess);
            const size_t sizeRead = static_cast<size_t>(sizeAccessed);

            NN_RESULT_DO(
                pStorage->Read(
                    offsetPhysical,
                    reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buffer) + sizeRead),
                    sizeReading
                )
            );
            NN_RESULT_SUCCESS;
        }));

    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファの内容をストレージに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      バッファの内容をストレージに書き込みます。
*/
Result RemapStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    NN_RESULT_DO(IterateMappingEntries(
        offset,
        static_cast<int64_t>(size),
        [=](fs::IStorage* pStorage, int64_t offsetPhysical, int64_t sizeAccess, int64_t sizeAccessed) NN_NOEXCEPT -> nn::Result
        {
            NN_SDK_ASSERT_LESS_EQUAL(sizeAccess, static_cast<int64_t>(size));
            NN_SDK_ASSERT_LESS_EQUAL(sizeAccessed, static_cast<int64_t>(size));
            const size_t sizeWriting = static_cast<size_t>(sizeAccess);
            const size_t sizeWritten = static_cast<size_t>(sizeAccessed);

            NN_RESULT_DO(
                pStorage->Write(
                    offsetPhysical,
                    reinterpret_cast<const void*>(reinterpret_cast<uintptr_t>(buffer) + sizeWritten),
                    sizeWriting
                )
            );
            NN_RESULT_SUCCESS;
        }));

    NN_RESULT_SUCCESS;
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*
* @details      フラッシュします。
*/
Result RemapStorage::Flush() NN_NOEXCEPT
{
    if( m_IsControlAreaModified )
    {
        NN_RESULT_DO(m_StorageMeta.Write(0, &m_ControlArea, sizeof(ControlArea)));
        NN_RESULT_DO(m_StorageMeta.Flush());
        m_IsControlAreaModified = false;
    }

    NN_RESULT_DO(m_StorageEntry.Flush());

    for( auto& storageEntry : m_StorageEntries )
    {
        if( storageEntry.IsRegistered() )
        {
            NN_RESULT_DO(storageEntry.GetStorage().Flush());
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        リマップストレージのサイズは取得できません。
*
* @param[out]   outValue    取得したサイズ格納先
*
* @return       関数の処理結果を返します。
*
* @details      リマップストレージのサイズは取得できません。
*/
Result RemapStorage::GetSize(int64_t*) NN_NOEXCEPT
{
    return nn::fs::ResultUnsupportedOperation();
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result RemapStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::Invalidate:
        {
            {
                int64_t sizeStorage = 0;
                NN_RESULT_DO(m_StorageMeta.GetSize(&sizeStorage));
                NN_RESULT_DO(m_StorageMeta.OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    0,
                    sizeStorage,
                    inBuffer,
                    inBufferSize));
            }

            {
                int64_t sizeStorage = 0;
                NN_RESULT_DO(m_StorageEntry.GetSize(&sizeStorage));
                NN_RESULT_DO(m_StorageEntry.OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    0,
                    sizeStorage,
                    inBuffer,
                    inBufferSize));
            }

            NN_RESULT_DO(IterateMappingEntries(
                offset,
                size,
                [=](fs::IStorage* pStorage, int64_t offsetPhysical, int64_t sizeAccess, int64_t sizeAccessed) NN_NOEXCEPT -> nn::Result
                {
                    NN_UNUSED(sizeAccessed);
                    NN_RESULT_DO(pStorage->OperateRange(
                        outBuffer,
                        outBufferSize,
                        operationId,
                        offsetPhysical,
                        sizeAccess,
                        inBuffer,
                        inBufferSize));
                    NN_RESULT_SUCCESS;
                }));

            NN_RESULT_SUCCESS;
        }

    case fs::OperationId::QueryRange:
        {
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

            fs::QueryRangeInfo outInfo;
            outInfo.Clear();

            NN_RESULT_DO(IterateMappingEntries(
                offset,
                size,
                [&](fs::IStorage* pStorage, int64_t offsetPhysical, int64_t sizeAccess, int64_t sizeAccessed) NN_NOEXCEPT -> nn::Result
                {
                    NN_UNUSED(sizeAccessed);

                    fs::QueryRangeInfo info;
                    NN_RESULT_DO(pStorage->OperateRange(
                        &info,
                        sizeof(info),
                        operationId,
                        offsetPhysical,
                        sizeAccess,
                        inBuffer,
                        inBufferSize));
                    outInfo.Merge(info);
                    NN_RESULT_SUCCESS;
                }));

            *reinterpret_cast<fs::QueryRangeInfo*>(outBuffer) = outInfo;
            NN_RESULT_SUCCESS;
        }

    default:
        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }
}

/**
* @brief        初期化します
*
* @param[in]    smallAlignedTail    小さいアライメントで配置されるアドレスの末尾
* @param[in]    largeAlignedTail    大きいアライメントで配置されるアドレスの末尾
*
* @details      初期化します
*/
void RemapStorage::StorageEntry::Initialize(int64_t smallAlignedTail, int64_t largeAlignedTail) NN_NOEXCEPT
{
    m_OffsetPhysicalNextSmall = smallAlignedTail;
    m_OffsetPhysicalNextLarge = largeAlignedTail;

    if( smallAlignedTail < largeAlignedTail
        && nn::util::is_aligned(smallAlignedTail, AlignmentLarge) )
    {
        // 大きいほうと同じ位置から
        m_OffsetPhysicalNextSmall = largeAlignedTail;
    }
    else if( largeAlignedTail < smallAlignedTail )
    {
        // 大きいほうは小さいほうより後ろから
        m_OffsetPhysicalNextLarge = nn::util::align_up(smallAlignedTail, SizeBlockLarge);
    }

    NN_SDK_ASSERT(
        nn::util::is_aligned(
            m_OffsetPhysicalNextSmall,
            AlignmentSmall
        )
    );
    NN_SDK_ASSERT(
        nn::util::is_aligned(
            m_OffsetPhysicalNextLarge,
            AlignmentLarge
        )
    );

    m_Storage = nn::fs::SubStorage();
    m_IsRegistered = false;
}

/**
* @brief        ストレージエントリを初期化します。
*
* @return       処理結果を返します。
*
* @details      ストレージエントリを初期化します。
*/
Result RemapStorage::InitializeStorageEntries() NN_NOEXCEPT
{
    int64_t offsetVirtualPrevious = 0;
    int64_t smallAlignedTails[CountMaxStorages] = { 0 };
    int64_t largeAlignedTails[CountMaxStorages] = { 0 };

    // マッピングエントリを読み込む
    for( auto entryIndex = 0; entryIndex < m_ControlArea.countMapEntries; ++entryIndex )
    {
        MapEntry entry = { 0 };

        NN_RESULT_DO(ReadMapEntry(&entry, entryIndex));

        if( entry.storageType < 0 || CountMaxStorages <= entry.storageType )
        {
            return nn::fs::ResultInvalidMapStorageType();
        }
        else if( entry.offsetPhysical + entry.size < entry.offsetPhysical
            || entry.offsetVirtual < offsetVirtualPrevious
            || entry.offsetPhysical < 0
            || entry.offsetVirtual < 0 )
        {
            return nn::fs::ResultInvalidMapOffset();
        }
        else if( entry.size < 0 )
        {
            return nn::fs::ResultInvalidMapSize();
        }

        if( entry.alignment == AlignmentSmall )
        {
            if( !nn::util::is_aligned(entry.offsetPhysical, AlignmentSmall)
                || !nn::util::is_aligned(entry.offsetVirtual, AlignmentSmall)
                || !nn::util::is_aligned(entry.size, AlignmentSmall) )
            {
                return nn::fs::ResultInvalidMapAlignment();
            }
            if( smallAlignedTails[entry.storageType] < entry.offsetPhysical + entry.size )
            {
                smallAlignedTails[entry.storageType] = entry.offsetPhysical + entry.size;
            }
        }
        else if( entry.alignment == AlignmentLarge )
        {
            if( !nn::util::is_aligned(entry.offsetPhysical, AlignmentLarge)
                || !nn::util::is_aligned(entry.offsetVirtual, AlignmentLarge)
                || !nn::util::is_aligned(entry.size, AlignmentLarge) )
            {
                return nn::fs::ResultInvalidMapAlignment();
            }
            if( largeAlignedTails[entry.storageType] < entry.offsetPhysical + entry.size )
            {
                largeAlignedTails[entry.storageType] = entry.offsetPhysical + entry.size;
            }
        }
        else
        {
            return nn::fs::ResultInvalidMapAlignment();
        }

        offsetVirtualPrevious = entry.offsetVirtual + entry.size;
    }

    // 現在のマッピング状況に基づきエントリを初期化する
    for( auto entryIndex = 0; entryIndex < CountMaxStorages; ++entryIndex )
    {
        m_StorageEntries[entryIndex].Initialize(
            smallAlignedTails[entryIndex],
            largeAlignedTails[entryIndex]
        );
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定のインデックスのマッピングエントリを読み込みます。
*
* @param[out]   outEntry    読み込んだエントリをコピーするバッファ
* @param[in]    index       読み込むインデックス
*
* @return       処理結果を返します。
*
* @pre
*               - outEntry != nullptr
*               - 0 <= index < m_ControlArea.countMapEntries
*
* @details      指定のインデックスのマッピングエントリを読み込みます。
*/
Result RemapStorage::ReadMapEntry(MapEntry* outEntry, int index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEntry);
    NN_SDK_REQUIRES_GREATER_EQUAL(index, 0);
    NN_SDK_REQUIRES_LESS(index, m_ControlArea.countMapEntries);
    return m_StorageEntry.Read(sizeof(MapEntry) * index, outEntry, sizeof(MapEntry));
}

/**
* @brief        指定のインデックスのマッピングエントリを書き込みます。
*
* @param[out]   pEntry   書き込むエントリ
* @param[in]    index    書き込むインデックス
*
* @return       処理結果を返します。
*
* @pre
*               - outEntry != nullptr
*               - 0 <= index < GetMapEntryCountMax(m_ControlArea.countMaxMaps)
*
* @details      指定のインデックスのマッピングエントリを書き込みます。
*/
Result RemapStorage::WriteMapEntry(const MapEntry* pEntry, int index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEntry);
    NN_SDK_REQUIRES_GREATER_EQUAL(index, 0);
    NN_SDK_REQUIRES_LESS(index, GetMapEntriesCountMax());
    return m_StorageEntry.Write(sizeof(MapEntry) * index, pEntry, sizeof(MapEntry));
}

/**
* @brief        仮想オフセットを作成します。
*
* @param[in]    higherBits  上位ビット
* @param[in]    lowerBits   下位ビット
*
* @return       仮想オフセットを返します。
*
* @pre
*               - lowerBits の上位ビットが全てゼロ
*
* @details      仮想オフセットを作成します。
*/
int64_t RemapStorage::MakeVirtualOffset(int64_t higherBits, int64_t lowerBits) const NN_NOEXCEPT
{
    const auto countLowerBits = 64 - m_ControlArea.countHigherBits;
    NN_SDK_REQUIRES((~((static_cast<int64_t>(1) << countLowerBits) - 1) & lowerBits) == 0);
    return (higherBits << countLowerBits) | lowerBits;
}

/**
* @brief        仮想オフセットの下位ビットのマスクを取得します。
*
* @return       マスクを返します。
*
* @details      仮想オフセットの下位ビットのマスクを取得します。
*/
int64_t RemapStorage::GetVirtualOffsetLowerBitsMask() const NN_NOEXCEPT
{
    return MakeVirtualOffset(1, 0) - 1;
}

/**
* @brief        仮想オフセットの上位ビットのマスクを取得します。
*
* @return       マスクを返します。
*
* @details      仮想オフセットの上位ビットのマスクを取得します。
*/
int64_t RemapStorage::GetVirtualOffsetHigherBitsMask() const NN_NOEXCEPT
{
    return ~GetVirtualOffsetLowerBitsMask();
}

/**
* @brief        マッピングエントリの最大数を取得します。
*
* @return       マッピングエントリの最大数を返します。
*
* @details      マッピングエントリの最大数を取得します。
*/
int64_t RemapStorage::GetMapEntriesCountMax() const NN_NOEXCEPT
{
    int64_t sizeEntry = 0;
    Result result = m_StorageEntry.GetSize(&sizeEntry);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
    return GetMapUpdateCountUpperBound(sizeEntry);
}

/**
* @brief        指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
*
* @param[in]    offsetVirtual   論理オフセット
*
* @return       指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを返します。
*
* @details      指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
*/
inline RemapStorage::MapEntryCache& RemapStorage::GetCache(int64_t) NN_NOEXCEPT
{
    return m_MapEntryCache;
}

/**
* @brief        指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
*
* @param[in]    offsetVirtual   論理オフセット
*
* @return       指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを返します。
*
* @details      指定の論理オフセットを含むエントリのキャッシュを管理するオブジェクトを取得します。
*/
inline const RemapStorage::MapEntryCache& RemapStorage::GetCache(int64_t) const NN_NOEXCEPT
{
    return m_MapEntryCache;
}

/**
* @brief        全てのキャッシュを破棄します。
*
* @details      全てのキャッシュを破棄します。
*/
inline void RemapStorage::ClearAllCaches() NN_NOEXCEPT
{
    m_MapEntryCache.Clear();
}

/**
* @brief        マッピングを新規登録します。
*
* @param[in]    offsetVirtual   登録する論理オフセット
* @param[in]    sizeNew         登録後のサイズ
* @param[in]    sizeOld         登録前のサイズ
* @param[in]    alignment       登録するアライメント
* @param[in]    storageType     登録するストレージ種別
*
* @return       処理結果を返します。
*
* @pre
*               - sizeOld < sizeNew
*               - alignment == AlignmentSmall || alignment == AlignmentLarge
*               - 0 <= storageType < CountMaxStorages
*
* @details      マッピングを新規登録します。
*/
Result RemapStorage::RegisterMapCore(
           int64_t offsetVirtual,
           int64_t sizeNew,
           int64_t sizeOld,
           int64_t alignment,
           int storageType
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(sizeNew, sizeOld);
    NN_SDK_REQUIRES(alignment == AlignmentSmall || alignment == AlignmentLarge);
    NN_SDK_REQUIRES_GREATER_EQUAL(storageType, 0);
    NN_SDK_REQUIRES_LESS(storageType, CountMaxStorages);

    auto controlArea = m_ControlArea;
    auto storageEntry = m_StorageEntries[storageType];
    MapEntry mapEntries[2] = {};

    NN_RESULT_DO(
        MakeMapEntry(
            mapEntries + 0,
            &storageEntry,
            &controlArea,
            offsetVirtual + sizeOld,
            sizeNew - sizeOld,
            alignment,
            storageType
        )
    );
    NN_SDK_ASSERT_LESS_EQUAL(sizeOld + mapEntries[0].size, sizeNew);

    if( sizeOld + mapEntries[0].size < sizeNew )
    {
        NN_RESULT_DO(
            MakeMapEntry(
                mapEntries + 1,
                &storageEntry,
                &controlArea,
                offsetVirtual + sizeOld + mapEntries[0].size,
                sizeNew - sizeOld - mapEntries[0].size,
                alignment,
                storageType
            )
        );
    }
    NN_SDK_ASSERT_EQUAL(sizeOld + mapEntries[0].size + mapEntries[1].size, sizeNew);

    NN_RESULT_DO(AddMapEntries(mapEntries, 0 < mapEntries[1].size ? 2 : 1));
    m_StorageEntries[storageType] = storageEntry;
    m_ControlArea = controlArea;
    m_IsControlAreaModified = true;

    NN_RESULT_SUCCESS;
}

/**
* @brief            マッピングエントリを新規作成します。
*
* @param[out]       outMapEntry     作成したエントリを格納する領域
* @param[in,out]    pStorageEntry   ストレージエントリ
* @param[in,out]    pControlArea    管理領域
* @param[in]        offsetVirtual   登録する論理オフセット
* @param[in]        size            登録するサイズ
* @param[in]        alignment       登録するアライメント
* @param[in]        storageType     登録するストレージ種別
*
* @return           処理結果を返します。
*
* @pre
*                   - outMapEntry != nullptr
*                   - pStorageEntry != nullptr
*                   - pControlArea != nullptr
*                   - alignment == AlignmentSmall || alignment == AlignmentLarge
*                   - 0 <= storageType < CountMaxStorages
*
* @details          ストレージエントリはマッピングが登録された後の状態に更新されます。
*/
Result RemapStorage::MakeMapEntry(
           MapEntry* outMapEntry,
           StorageEntry* pStorageEntry,
           ControlArea* pControlArea,
           int64_t offsetVirtual,
           int64_t size,
           int64_t alignment,
           int storageType
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMapEntry);
    NN_SDK_REQUIRES_NOT_NULL(pStorageEntry);
    NN_SDK_REQUIRES_NOT_NULL(pControlArea);
    NN_SDK_REQUIRES(alignment == AlignmentSmall || alignment == AlignmentLarge);
    NN_SDK_REQUIRES_GREATER_EQUAL(storageType, 0);
    NN_SDK_REQUIRES_LESS(storageType, CountMaxStorages);

    // アライメントに違反しているならばエラー
    if( (alignment == AlignmentSmall && !nn::util::is_aligned(offsetVirtual, AlignmentSmall))
        || (alignment == AlignmentLarge && !nn::util::is_aligned(offsetVirtual, AlignmentLarge)) )
    {
        return nn::fs::ResultInvalidAlignment();
    }

    if( (alignment == AlignmentSmall && size % SizeBlockSmall != 0)
        || (alignment == AlignmentLarge && size % SizeBlockLarge != 0) )
    {
        return nn::fs::ResultInvalidAlignment();
    }

    // 指定の範囲と重複する範囲が既に登録されているならエラー
    {
        MapEntry mapEntry;
        const auto countEntries = m_ControlArea.countMapEntries;

        for( auto indexMapEntry = 0; indexMapEntry < countEntries; ++indexMapEntry )
        {
            NN_RESULT_DO(ReadMapEntry(&mapEntry, indexMapEntry));
            if( offsetVirtual + size <= mapEntry.offsetVirtual )
            {
                break;
            }
            else if( offsetVirtual < mapEntry.offsetVirtual + mapEntry.size )
            {
                return nn::fs::ResultMapAddressAlreadyRegistered();
            }
        }
    }

    // 確保する余裕が無ければエラー
    if( pControlArea->countMapEntries == GetMapEntriesCountMax() )
    {
        return nn::fs::ResultMapFull();
    }

    // オフセットを確定し、マッピングを割り当てる
    {
        // 管理領域とエントリを更新する
        outMapEntry->offsetVirtual = offsetVirtual;
        outMapEntry->alignment = static_cast<int32_t>(alignment);
        outMapEntry->storageType = static_cast<int32_t>(storageType);

        int64_t offsetLarge = pStorageEntry->GetOffsetPhysicalNextLarge();
        int64_t offsetSmall = pStorageEntry->GetOffsetPhysicalNextSmall();

        switch( alignment )
        {
        case AlignmentSmall:
            {
                outMapEntry->offsetPhysical = offsetSmall;
                outMapEntry->size = size;

                if( SizeBlockLarge < offsetLarge - offsetSmall )
                {
                    const auto restSize = SizeBlockLarge - (offsetSmall % SizeBlockLarge);
                    outMapEntry->size = std::min(outMapEntry->size, restSize);
                }

                offsetSmall += outMapEntry->size;

                if( offsetSmall < offsetLarge && offsetSmall % SizeBlockLarge == 0 )
                {
                    offsetSmall = offsetLarge;
                }
                else if( offsetLarge < offsetSmall )
                {
                    offsetLarge = (offsetSmall + (SizeBlockLarge - 1)) & ~(SizeBlockLarge - 1);
                }
            }
            break;

        case AlignmentLarge:
            {
                outMapEntry->offsetPhysical = offsetLarge;
                outMapEntry->size = size;

                if( offsetSmall == offsetLarge )
                {
                    offsetSmall += outMapEntry->size;
                }
                offsetLarge += outMapEntry->size;
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
        pStorageEntry->SetOffsetPhysicalNextLarge(offsetLarge);
        pStorageEntry->SetOffsetPhysicalNextSmall(offsetSmall);

        NN_SDK_ASSERT(outMapEntry->size % outMapEntry->alignment == 0);
        NN_SDK_ASSERT(nn::util::is_aligned(offsetSmall, AlignmentSmall));
        NN_SDK_ASSERT(nn::util::is_aligned(offsetLarge, AlignmentLarge));
        NN_SDK_ASSERT_EQUAL(
            outMapEntry->offsetVirtual & GetVirtualOffsetHigherBitsMask(),
            (outMapEntry->offsetVirtual + outMapEntry->size - 1) & GetVirtualOffsetHigherBitsMask()
        );
        NN_SDK_ASSERT_LESS_EQUAL(offsetSmall, offsetLarge);

        ++pControlArea->countMapEntries;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングエントリをテーブルに追加します。
*
* @param[in]    pMapEntries 実際に登録できたサイズ
* @param[in]    count       登録する論理オフセット
*
* @return       処理結果を返します。
*
* @pre
*               - pMapEntries != nullptr
*               - 1 <= count
*               - pMapEntries[n].offsetVirtual < pMapEntries[n + 1].offsetVirtual
*
* @details      マッピングエントリをテーブルに追加します。
*/
Result RemapStorage::AddMapEntries(const MapEntry* pMapEntries, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMapEntries);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 1);

    auto indexMapEntry = m_ControlArea.countMapEntries - 1;

    // マッピングエントリキャッシュをクリアする
    ClearAllCaches();

    // エントリが論理オフセットの昇順に並ぶように後方のエントリをずらす
    while( 0 <= indexMapEntry )
    {
        MapEntry mapEntry = {};

        NN_RESULT_DO(ReadMapEntry(&mapEntry, indexMapEntry));
        if( mapEntry.offsetVirtual < pMapEntries[0].offsetVirtual )
        {
            break;
        }
        else
        {
            NN_RESULT_DO(WriteMapEntry(&mapEntry, indexMapEntry + count));
            --indexMapEntry;
        }
    }
    ++indexMapEntry;

    // エントリを新規追加する
    for( auto index = 0; index < count; ++index )
    {
        NN_SDK_REQUIRES(index == 0
            || pMapEntries[index - 1].offsetVirtual + pMapEntries[index - 1].size
            == pMapEntries[index].offsetVirtual);
        NN_RESULT_DO(WriteMapEntry(pMapEntries + index, indexMapEntry + index));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        マッピングエントリのイテレーションを行います。
*
* @param[in]    offset      イテレーションを行う開始位置
* @param[in]    size        データサイズ
* @param[in]    func        エントリごとに行う処理
*
* @return       処理結果を返します。
*
* @details      マッピングエントリをテーブルに追加します。
*/
template<typename TFunc>
Result RemapStorage::IterateMappingEntries(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT
{
    MapIterator iterator(this, offset);
    int64_t sizeAccessed = 0;

    while( sizeAccessed < size && iterator.MoveNext(size - sizeAccessed) )
    {
        const auto pStorage = iterator.GetStorage();
        const auto sizeAccess = std::min(size - sizeAccessed, iterator.GetMapEntry().size);

        if( pStorage == nullptr )
        {
            if( iterator.GetLastResult().IsSuccess() )
            {
                NN_RESULT_THROW(nn::fs::ResultMapStorageNotFound());
            }
            else
            {
                NN_RESULT_THROW(iterator.GetLastResult());
            }
        }

        NN_RESULT_DO(func(pStorage, iterator.GetMapEntry().offsetPhysical, sizeAccess, sizeAccessed));

        sizeAccessed += sizeAccess;
    }

    NN_RESULT_DO(iterator.GetLastResult());
    NN_RESULT_THROW_UNLESS(sizeAccessed != 0 || size <= 0, nn::fs::ResultInvalidOffset());

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定の論理オフセットに関するマッピングエントリに
*               アクセスするイテレータを作成します。
*
* @param[in]    pRemapStorage   ストレージ本体
* @param[in]    offsetVirtual   アクセスする論理オフセット
*
* @pre
*               - pRemapStorage != nullptr
*
* @details      指定の論理オフセットに関するマッピングエントリに
*               アクセスするイテレータを作成します。
*/
RemapStorage::MapIterator::MapIterator(
    RemapStorage* pRemapStorage,
    int64_t offsetVirtual
) NN_NOEXCEPT
    : m_pRemapStorage(pRemapStorage),
      m_MapEntry(),
      m_IndexMapEntry(-1),
      m_StorageType(CountMaxStorages),
      m_Result(ResultSuccess())
{
    NN_SDK_REQUIRES_NOT_NULL(pRemapStorage);
    m_MapEntry.offsetPhysical = 0;
    m_MapEntry.offsetVirtual = offsetVirtual;
    m_MapEntry.alignment = 0;
    m_MapEntry.size = 0;
    m_MapEntry.storageType = 0;
}

/**
* @brief        次のマッピングエントリを読み込みます。
*
* @param[in]    size    読み込むサイズの残り
*
* @return       処理結果を返します。
* @retval       true            次のマッピングエントリの読み込みに成功しました。
* @retval       false           次のマッピングエントリの読み込みに失敗したか、
*                               またはイテレーションが終了しました。
*
* @details      GetMapEntry() を呼び出す前に少なくとも一度本関数を呼び出し、
*               true が返っている必要があります。
*               処理に失敗した場合 GetlastResult() でエラー内容を取得できます。
*/
bool RemapStorage::MapIterator::MoveNext(int64_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pRemapStorage);

    auto offsetVirtualNext = m_MapEntry.offsetVirtual + m_MapEntry.size;

    if( (m_MapEntry.offsetVirtual & m_pRemapStorage->GetVirtualOffsetHigherBitsMask())
        != (offsetVirtualNext & m_pRemapStorage->GetVirtualOffsetHigherBitsMask()) )
    {
        return false;
    }

    m_Result = ResultSuccess();

    // キャッシュを探索する
    {
        MapEntry mapEntryCache;
        int indexMapEntryCache;
        const auto found = m_pRemapStorage->GetCache(offsetVirtualNext).GetMapEntry(
                               &mapEntryCache,
                               &indexMapEntryCache,
                               offsetVirtualNext
                           );

        if( found )
        {
            m_MapEntry = mapEntryCache;
            m_MapEntry.size -= offsetVirtualNext - m_MapEntry.offsetVirtual;
            m_MapEntry.offsetPhysical += offsetVirtualNext - m_MapEntry.offsetVirtual;
            m_MapEntry.offsetVirtual = offsetVirtualNext;
            m_IndexMapEntry = indexMapEntryCache;

            if( m_StorageType != CountMaxStorages && m_StorageType != m_MapEntry.storageType )
            {
                m_Result = nn::fs::ResultInvalidMapStorageType();
                return false;
            }

            m_StorageType = m_MapEntry.storageType;

            return true;
        }
    }

    // ストレージを探索する
    const auto countEntries = m_pRemapStorage->m_ControlArea.countMapEntries;
    for( auto indexMapEntry = m_IndexMapEntry + 1; indexMapEntry < countEntries; ++indexMapEntry )
    {
        MapEntry entry;

        m_Result = m_pRemapStorage->ReadMapEntry(&entry, indexMapEntry);

        if( m_Result.IsFailure() )
        {
            return false;
        }
        else if( static_cast<uint64_t>(offsetVirtualNext + size)
            <= static_cast<uint64_t>(entry.offsetVirtual) )
        {
            break;
        }
        else if( offsetVirtualNext < entry.offsetVirtual + entry.size )
        {
            if( (0 <= m_IndexMapEntry && offsetVirtualNext != entry.offsetVirtual)
                || (entry.offsetPhysical < m_MapEntry.offsetPhysical + m_MapEntry.size)
                || (entry.offsetPhysical < 0)
                || (entry.offsetVirtual < 0) )
            {
                m_Result = nn::fs::ResultInvalidMapOffset();
            }
            else if( entry.size <= 0 )
            {
                m_Result = nn::fs::ResultInvalidMapSize();
            }
            else if( (entry.alignment != AlignmentSmall && entry.alignment != AlignmentLarge)
                || !nn::util::is_aligned(entry.offsetVirtual, entry.alignment)
                || !nn::util::is_aligned(entry.offsetPhysical, entry.alignment)
                || !nn::util::is_aligned(entry.size, entry.alignment) )
            {
                m_Result = nn::fs::ResultInvalidMapAlignment();
            }
            else if( (entry.storageType < 0 || CountMaxStorages <= entry.storageType)
                || (m_StorageType != CountMaxStorages && m_StorageType != entry.storageType) )
            {
                m_Result = nn::fs::ResultInvalidMapStorageType();
            }

            if( m_Result.IsFailure() )
            {
                return false;
            }

            m_MapEntry = entry;
            m_MapEntry.size -= offsetVirtualNext - m_MapEntry.offsetVirtual;
            m_MapEntry.offsetPhysical += offsetVirtualNext - m_MapEntry.offsetVirtual;
            m_MapEntry.offsetVirtual = offsetVirtualNext;
            m_IndexMapEntry = indexMapEntry;
            m_StorageType = m_MapEntry.storageType;

            m_pRemapStorage->GetCache(offsetVirtualNext).SetMapEntry(entry, m_IndexMapEntry);

            return true;
        }
    }

    m_IndexMapEntry = m_pRemapStorage->m_ControlArea.countMapEntries;

    return false;
}

/**
* @brief        現在のマッピングエントリを取得します。
*
* @return       現在のマッピングエントリを返します。
*
* @pre
*               - MoveNext() が少なくとも一度 true を返している
*
* @details      現在のマッピングエントリを取得します。
*/
const RemapStorage::MapEntry& RemapStorage::MapIterator::GetMapEntry() const NN_NOEXCEPT
{
    return m_MapEntry;
}

/**
* @brief        アクセスしているマッピングエントリに紐付けられたストレージを取得します。
*
* @return       アクセスしているマッピングエントリに紐付けられたストレージを返します。
* @retval       nullptr ストレージ種別またはストレージが登録されていません。
*
* @pre
*               - MoveNext() が少なくとも一度 true を返している
*
* @details      アクセスしているマッピングエントリに紐付けられたストレージを取得します。
*/
fs::IStorage* RemapStorage::MapIterator::GetStorage() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pRemapStorage);
    NN_SDK_ASSERT_GREATER_EQUAL(m_StorageType, 0);
    NN_SDK_ASSERT_LESS(m_StorageType, CountMaxStorages);
    if( m_pRemapStorage->m_StorageEntries[m_StorageType].IsRegistered() )
    {
        return &m_pRemapStorage->m_StorageEntries[m_StorageType].GetStorage();
    }
    else
    {
        return nullptr;
    }
}

/**
* @brief        最後に実行された処理の結果を取得します。
*
* @return       最後に実行された処理の結果を返します。
*
* @details      最後に実行された処理の結果を取得します。
*/
Result RemapStorage::MapIterator::GetLastResult() const NN_NOEXCEPT
{
    return m_Result;
}

/**
* @brief        空のマッピングエントリキャッシュを作成します。
*/
RemapStorage::MapEntryCache::MapEntryCache() NN_NOEXCEPT
    : m_NextNodeIndex(0),
      m_Mutex(false)
{
    Clear();
}

/**
* @brief        マッピングエントリをキャッシュします。
*
* @param[in]    mapEntry        キャッシュするマッピングエントリ
* @param[in]    indexMapEntry   キャッシュするマッピングエントリのインデックス
*
* @details      同じエントリはキャッシュされていないものとし、重複があるかどうかは検査しません。
*/
void RemapStorage::MapEntryCache::SetMapEntry(
         const MapEntry& mapEntry,
         int indexMapEntry
     ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> guard(m_Mutex);

    m_Nodes[m_NextNodeIndex].mapEntry = mapEntry;
    m_Nodes[m_NextNodeIndex].indexMapEntry = indexMapEntry;
    m_NextNodeIndex = (m_NextNodeIndex + 1) % CountNodes;
}

/**
* @brief        指定の論理オフセットを含むキャッシュを取得します。
*
* @param[out]   outMapEntry         キャッシュされたエントリをコピーするバッファ
* @param[out]   outIndexmapEntry    キャッシュされたインデックスをコピーするバッファ
* @param[in]    offsetVirtual       取得するエントリの論理オフセット
*
* @return       見つかったかどうかを返します。
* @retval       true                キャッシュされたエントリが見つかりました。
* @retval       false               指定の論理オフセットを含むエントリはキャッシュされていませんでした。
*
* @pre
*               - outMapEntry != nullptr
*               - outIndexMapEntry != nullptr
*
* @details
*/
bool RemapStorage::MapEntryCache::GetMapEntry(
         MapEntry* outMapEntry,
         int* outIndexMapEntry,
         int64_t offsetVirtual
     ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMapEntry);
    NN_SDK_REQUIRES_NOT_NULL(outIndexMapEntry);

    std::lock_guard<nn::os::Mutex> guard(m_Mutex);

    for( const auto &node : m_Nodes )
    {
        if( node.indexMapEntry != InvalidMapEntryIndex
            && node.mapEntry.offsetVirtual <= offsetVirtual
            && offsetVirtual < node.mapEntry.offsetVirtual + node.mapEntry.size )
        {
            *outMapEntry = node.mapEntry;
            *outIndexMapEntry = node.indexMapEntry;
            return true;
        }
    }

    return false;
}

/**
* @brief        キャッシュを空にします。
*/
void RemapStorage::MapEntryCache::Clear() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> guard(m_Mutex);

    for( auto& node : m_Nodes )
    {
        node.indexMapEntry = InvalidMapEntryIndex;
    }
    m_NextNodeIndex = 0;
}

}}} // namespace nn::fssystem::save

