﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_HierarchicalDuplexStorage.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    // マジックコード
    const uint32_t MagicCode = 0x53465044;  // 'DPFS'

    // バージョン
    const uint32_t VersionCode = 0x00010000;
    const uint32_t VersionCodeMask = 0xFFFF0000;

    //
    const uint32_t BitmapAlignL1 = 4;

    inline int64_t AdjustAlign64(int64_t offset, int64_t align)
    {
        return ((offset + align - 1LL) & ~(align - 1LL));
    }

    inline bool IsFutureVersion(uint32_t version) NN_NOEXCEPT
    {
        return (VersionCode & VersionCodeMask) < (version & VersionCodeMask);
    }
}

/**
* @brief        二重化ストレージ用メタデータ領域を作成します。
*
* @param[out]   outSizeControl  管理領域サイズ
* @param[out]   outBodySize     マスター,L1,sizeDataの合計サイズ
* @param[out]   pMeta           メタデータ
* @param[in]    sizeBlockLevel1 L1 ビットマップ用ブロックサイズ
* @param[in]    sizeBlockLevel2 L2 ビットマップ用ブロックサイズ
* @param[in]    sizeData        実データサイズ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorageControlArea::CreateMetaInformation(
           HierarchicalDuplexMetaInformation* pMeta,
           const InputParam& inputParam,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pMeta);
    NN_SDK_ASSERT((inputParam.sizeBlockLevel[0] > 0)
        && IsPower2(static_cast<int>(inputParam.sizeBlockLevel[0])));
    NN_SDK_ASSERT((inputParam.sizeBlockLevel[1] > 0)
        && IsPower2(static_cast<int>(inputParam.sizeBlockLevel[1])));
    NN_SDK_ASSERT(0 == (sizeData % inputParam.sizeBlockLevel[1]));
    NN_SDK_ASSERT(BitmapAlignL1 <= inputParam.sizeBlockLevel[1]);
    NN_SDK_ASSERT(BitmapAlignL1 <= inputParam.sizeBlockLevel[0]);

    // 管理領域のサイズ
    std::memset(pMeta, 0, sizeof(*pMeta));

    // オプション情報の位置が確定しました。
    pMeta->magic = MagicCode;
    pMeta->version = VersionCode;

    // それぞれのデータサイズを求ます。
    int64_t sizeLevel[3];
    int64_t bitCountLevel[2];
    // L3
    sizeLevel[2] = sizeData;
    // bit単位のデータサイズ
    bitCountLevel[1]
        = ((sizeLevel[2] + inputParam.sizeBlockLevel[1] - 1) / inputParam.sizeBlockLevel[1]);
    // bitCount -> 必要バイト数に変換します
    sizeLevel[1] = DuplexBitmapHolder::QuerySize(static_cast<uint32_t>(bitCountLevel[1]));
    // サイズをL1ブロック境界に整合します
    sizeLevel[1] = AdjustAlign64(sizeLevel[1], inputParam.sizeBlockLevel[0]);
    // bit
    bitCountLevel[0]
        = ((sizeLevel[1] + inputParam.sizeBlockLevel[0] - 1) / inputParam.sizeBlockLevel[0]);
    // bitCount -> 必要バイト数に変換します
    sizeLevel[0] = DuplexBitmapHolder::QuerySize(static_cast<uint32_t>(bitCountLevel[0]));
    // サイズを BitmapAlignL1 バイト境界に整合します
    sizeLevel[0] = AdjustAlign64(sizeLevel[0], BitmapAlignL1);

    // マスター、L1は少なくともDPFS_BITMAP_ALIGNに整合しています。
    int64_t offset = 0;

    pMeta->infoDuplex.info[0].offset = offset;
    pMeta->infoDuplex.info[0].size = sizeLevel[0];
    pMeta->infoDuplex.info[0].orderBlock = 0;
    offset += sizeLevel[0] * 2;

    pMeta->infoDuplex.info[1].offset = offset;
    pMeta->infoDuplex.info[1].size = sizeLevel[1];
    pMeta->infoDuplex.info[1].orderBlock
        = ILog2(static_cast<uint32_t>(inputParam.sizeBlockLevel[0]));
    offset += sizeLevel[1] * 2;

    // 実データはL2ブロックサイズに整合しておきます。
    offset = AdjustAlign64(offset, inputParam.sizeBlockLevel[1]);
    pMeta->infoDuplex.info[2].offset = offset;
    pMeta->infoDuplex.info[2].size = sizeLevel[2];
    pMeta->infoDuplex.info[2].orderBlock
        = ILog2(static_cast<uint32_t>(inputParam.sizeBlockLevel[1]));
    offset += sizeLevel[2] * 2;

    NN_RESULT_SUCCESS;
}

/**
* @brief        二重化ストレージの総データサイズを取得します。
*
* @param[out]   outSizeControl  管理領域 + オプションデータ領域の合計サイズ
* @param[out]   outBodySize     L1,sizeDataの合計サイズ
* @param[in]    inputParam      入力パラメータ
* @param[in]    sizeData        実データサイズ (二重化元データのサイズ)
*
* @return       関数の処理結果を返します。
*
* @details      ストレージシステムを構築するために必要なそれぞれのサイズを求めます
*/
Result HierarchicalDuplexStorageControlArea::QuerySize(
           HierarchicalDuplexSizeSet* outSizeSet,
           const InputParam& inputParam,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSizeSet);
    HierarchicalDuplexMetaInformation info;
    const auto result = CreateMetaInformation(
                            &info,
                            inputParam,
                            sizeData
                        );
    if( result.IsSuccess() )
    {
        outSizeSet->layeredBitmapSizes[0] = info.infoDuplex.info[0].size;
        outSizeSet->layeredBitmapSizes[1] = info.infoDuplex.info[1].size;
        outSizeSet->bodySize = info.infoDuplex.info[2].size;
    }
    return result;
}

/**
* @brief        二重化ストレージ用メタデータ領域をフォーマットします。
*
* @param[out]   storageMeta メタデータストレージ
* @param[in]    inputParam  入力パラメータ
*
* @return       関数の処理結果を返します。
*
* @details      この関数では以下の処理を行います。
*
*              ・管理領域の生成。
*              ・ビットマップ領域、実データ領域の、それぞれのオフセット、サイズを計算し管理領域に覚えます。
*                オフセット情報はフォーマット後は基本的に読み取り専用のデータです。
*/
Result HierarchicalDuplexStorageControlArea::Format(
           fs::SubStorage storageMeta,
           const InputParam& inputParam,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    // メタデータを作成します。
    HierarchicalDuplexMetaInformation meta;

    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(meta)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    NN_RESULT_DO(
        CreateMetaInformation(
            &meta,
            inputParam,
            sizeData
        )
    );

    // パラメーター確認
    NN_SDK_ASSERT(meta.infoDuplex.info[0].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(meta.infoDuplex.info[0].orderBlock == 0);
    NN_SDK_ASSERT(meta.infoDuplex.info[1].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(meta.infoDuplex.info[1].orderBlock > 0
        && meta.infoDuplex.info[1].orderBlock < 20);
    NN_SDK_ASSERT(meta.infoDuplex.info[2].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(meta.infoDuplex.info[2].orderBlock > 0
        && meta.infoDuplex.info[2].orderBlock < 20);

    // メタデータを書き込みます。
    NN_RESULT_DO(storageMeta.Write(0, &meta, sizeof(meta)));
    NN_RESULT_DO(storageMeta.Flush());

    NN_RESULT_SUCCESS;
}

/**
* @brief        二重化ストレージ用メタデータ領域に拡張したデータの情報を保存します。
*
* @param[out]   storageMeta メタデータストレージ
* @param[in]    sizeData    拡張後の実データサイズ (二重化元データのサイズ)
*
* @return       関数の処理結果を返します。
*
* @details      メタデータのバージョン、ビットマップサイズ、実データサイズのみを更新します。
*/
Result HierarchicalDuplexStorageControlArea::Expand(
           fs::SubStorage storageMeta,
           int64_t sizeData
       ) NN_NOEXCEPT
{
    // ストレージに十分なサイズがあることを確認します。
    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(HierarchicalDuplexMetaInformation)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    // ストレージからメタデータを読み込みます。
    HierarchicalDuplexMetaInformation metaOld;
    NN_RESULT_DO(storageMeta.Read(0, &metaOld, sizeof(HierarchicalDuplexMetaInformation)));

    // 読み込んだメタデータのマジックコードとバージョンを確認します。
    if( metaOld.magic != MagicCode )
    {
        // マジックコードが一致しません。
        return nn::fs::ResultIncorrectDuplexMagicCode();
    }

    if( IsFutureVersion(metaOld.version) )
    {
        // バージョンが未来のものです。
        return nn::fs::ResultUnsupportedVersion();
    }

    // ハッシュ計算単位は同じ値にします。
    InputParam inputParam;
    inputParam.sizeBlockLevel[0] = static_cast<size_t>(1) << metaOld.infoDuplex.info[1].orderBlock;
    inputParam.sizeBlockLevel[1] = static_cast<size_t>(1) << metaOld.infoDuplex.info[2].orderBlock;

    // 拡張後のメタデータを作ります。
    HierarchicalDuplexMetaInformation metaNew;
    NN_RESULT_DO(
        CreateMetaInformation(
            &metaNew,
            inputParam,
            sizeData
        )
    );

    for( int i = 0; i < 3; ++i )
    {
        // ハッシュ計算単位を確認します。
        NN_SDK_ASSERT_EQUAL(
            metaOld.infoDuplex.info[i].orderBlock,
            metaNew.infoDuplex.info[i].orderBlock
        );

        // ビットマップのサイズを確認します。
        NN_SDK_ASSERT_GREATER_EQUAL(metaNew.infoDuplex.info[i].size, static_cast<int64_t>(0));

        // オフセットはフォーマット時のものを使います。
        metaNew.infoDuplex.info[i].offset = metaOld.infoDuplex.info[i].offset;
    }

    // メタデータを書き込みます。
    NN_RESULT_DO(storageMeta.Write(0, &metaNew, sizeof(HierarchicalDuplexMetaInformation)));
    NN_RESULT_DO(storageMeta.Flush());

    NN_RESULT_SUCCESS;
}

/**
* @brief        二重化ストレージの管理領域をマウントします。
*
* @param[in]    storageMeta メタデータストレージ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorageControlArea::Initialize(
           fs::SubStorage storageMeta
       ) NN_NOEXCEPT
{
    {
        int64_t sizeMeta = 0;
        NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
        if( sizeMeta < static_cast<int64_t>(sizeof(m_Meta)) )
        {
            return nn::fs::ResultInvalidSize();
        }
    }

    m_Storage = storageMeta;

    // メタデータを読み込みます。
    NN_RESULT_DO(m_Storage.Read(0, &m_Meta, sizeof(m_Meta)));

    // メタデータのバージョンとマジックコードを取得します。
    if( m_Meta.magic != MagicCode )
    {
        m_Storage = fs::SubStorage();
        return nn::fs::ResultIncorrectDuplexMagicCode();
    }
    if( (m_Meta.version & VersionCodeMask) != (VersionCode & VersionCodeMask) )
    {
        m_Storage = fs::SubStorage();
        return nn::fs::ResultUnsupportedVersion();
    }

    // パラメーター確認
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[0].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[0].orderBlock == 0);
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[1].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[1].orderBlock > 0
        && m_Meta.infoDuplex.info[1].orderBlock < 20);
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[2].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(m_Meta.infoDuplex.info[2].orderBlock > 0
        && m_Meta.infoDuplex.info[2].orderBlock < 20);

    NN_RESULT_SUCCESS;
}

/**
* @brief        管理領域をアンマウントします。
*/
void HierarchicalDuplexStorageControlArea::Finalize() NN_NOEXCEPT
{
    m_Storage = fs::SubStorage();
}

/**
* @brief        二重化ビットマップ用領域をフォーマットします。
*
* @param[in]    infoDuplex  二重化ストレージ情報
* @param[in]    fileBitmap  二重化ビットマップストレージ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::Format(
           const HierarchicalDuplexInformation& infoDuplex,
           fs::SubStorage storageMasterBitmapA,
           fs::SubStorage storageMasterBitmapB,
           fs::SubStorage storageBitmapL1A,
           fs::SubStorage storageBitmapL1B,
           IBufferManager* pBuffer
       ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    size_t sizeBlock = static_cast<size_t>(1) << infoDuplex.info[2].orderBlock;
    const auto cacheCount = 4;

    // サイズを確認します。
    {
        int64_t sizeMasterA = 0;
        int64_t sizeMasterB = 0;
        int64_t sizeL1A = 0;
        int64_t sizeL1B = 0;
        NN_RESULT_DO(storageMasterBitmapA.GetSize(&sizeMasterA));
        NN_RESULT_DO(storageMasterBitmapB.GetSize(&sizeMasterB));
        NN_RESULT_DO(storageBitmapL1A.GetSize(&sizeL1A));
        NN_RESULT_DO(storageBitmapL1B.GetSize(&sizeL1B));
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterA, infoDuplex.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterB, infoDuplex.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1A, infoDuplex.info[1].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1B, infoDuplex.info[1].size);
    }

    // フォーマットします。
    BufferedStorage storageBufferedMasterA;
    result = storageBufferedMasterA.Initialize(
                 storageMasterBitmapA,
                 pBuffer,
                 sizeBlock,
                 cacheCount
             );
    if( result.IsSuccess() )
    {
        BufferedStorage storageBufferedMasterB;
        result = storageBufferedMasterB.Initialize(
                     storageMasterBitmapB,
                     pBuffer,
                     sizeBlock,
                     cacheCount
                 );
        if( result.IsSuccess() )
        {
            BufferedStorage storageBufferedL1A;
            result = storageBufferedL1A.Initialize(
                         storageBitmapL1A,
                         pBuffer,
                         sizeBlock,
                         cacheCount
                     );
            if( result.IsSuccess() )
            {
                BufferedStorage storageBufferedL1B;
                result = storageBufferedL1B.Initialize(
                             storageBitmapL1B,
                             pBuffer,
                             sizeBlock,
                             cacheCount
                         );

                if( result.IsSuccess() )
                {
                    result = DuplexBitmapHolder::Format(
                        static_cast<uint32_t>(infoDuplex.info[0].size * 8),
                        fs::SubStorage(
                            &storageBufferedMasterA,
                            0,
                            infoDuplex.info[0].size
                        ),
                        fs::SubStorage(
                            &storageBufferedMasterB,
                            0,
                            infoDuplex.info[0].size
                        )
                    );
                }

                if( result.IsSuccess() )
                {
                    result = DuplexBitmapHolder::Format(
                        static_cast<uint32_t>(infoDuplex.info[1].size * 8),
                        fs::SubStorage(
                            &storageBufferedL1A,
                            0,
                            infoDuplex.info[1].size
                        ),
                        fs::SubStorage(
                            &storageBufferedL1B,
                            0,
                            infoDuplex.info[1].size
                        )
                    );
                }

                {
                    const auto resultFlush = storageBufferedL1B.Flush();
                    if( resultFlush.IsFailure() && result.IsSuccess() )
                    {
                        result = resultFlush;
                    }
                }
            }

            {
                const auto resultFlush = storageBufferedL1A.Flush();
                if( resultFlush.IsFailure() && result.IsSuccess() )
                {
                    result = resultFlush;
                }
            }
        }

        {
            const auto resultFlush = storageBufferedMasterB.Flush();
            if( resultFlush.IsFailure() && result.IsSuccess() )
            {
                result = resultFlush;
            }
        }
    }

    {
        const auto resultFlush = storageBufferedMasterA.Flush();
        if( resultFlush.IsFailure() && result.IsSuccess() )
        {
            result = resultFlush;
        }
    }

    return result;
} // NOLINT(impl/function_size)

/**
* @brief        二重化ビットマップ用領域を拡張します。
*
* @param[in]    infoDuplexOld           拡張前の二重化ストレージ情報
* @param[in]    infoDuplexNew           拡張後の二重化ストレージ情報
* @param[in]    storageMasterBitmapA    二重化ビットマップストレージ (マスター A)
* @param[in]    storageMasterBitmapB    二重化ビットマップストレージ (マスター B)
* @param[in]    storageBitmapL1A        二重化ビットマップストレージ (L1A)
* @param[in]    storageBitmapL1B        二重化ビットマップストレージ (L1B)
* @param[in]    pBuffer                 バッファマネージャ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::Expand(
           const HierarchicalDuplexInformation& infoDuplexOld,
           const HierarchicalDuplexInformation& infoDuplexNew,
           fs::SubStorage storageMasterBitmapA,
           fs::SubStorage storageMasterBitmapB,
           fs::SubStorage storageBitmapL1A,
           fs::SubStorage storageBitmapL1B,
           IBufferManager* pBuffer
       ) NN_NOEXCEPT
{
    const size_t sizeBlockNew = static_cast<size_t>(1) << infoDuplexNew.info[2].orderBlock;
    static const auto CacheCount = 4;

    // サイズを確認します。
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(infoDuplexNew.info[0].size, infoDuplexOld.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(infoDuplexNew.info[1].size, infoDuplexOld.info[1].size);

        int64_t sizeMasterA = 0;
        int64_t sizeMasterB = 0;
        int64_t sizeL1A = 0;
        int64_t sizeL1B = 0;
        NN_RESULT_DO(storageMasterBitmapA.GetSize(&sizeMasterA));
        NN_RESULT_DO(storageMasterBitmapB.GetSize(&sizeMasterB));
        NN_RESULT_DO(storageBitmapL1A.GetSize(&sizeL1A));
        NN_RESULT_DO(storageBitmapL1B.GetSize(&sizeL1B));
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterA, infoDuplexNew.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterB, infoDuplexNew.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1A, infoDuplexNew.info[1].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1B, infoDuplexNew.info[1].size);
    }

    // マスタービットマップが大きくなるなら拡張します。
    if( infoDuplexOld.info[0].size < infoDuplexNew.info[0].size )
    {
        // マスタービットマップ拡張に必要なストレージを準備します。
        BufferedStorage storageBufferedMasterA;
        NN_RESULT_DO(
            storageBufferedMasterA.Initialize(
                 storageMasterBitmapA,
                 pBuffer,
                 sizeBlockNew,
                 CacheCount
             )
         );

        BufferedStorage storageBufferedMasterB;
        NN_RESULT_DO(
            storageBufferedMasterB.Initialize(
                 storageMasterBitmapB,
                 pBuffer,
                 sizeBlockNew,
                 CacheCount
             )
         );

        // マスタービットマップを拡張します。
        NN_RESULT_DO(
            DuplexBitmapHolder::Expand(
                static_cast<uint32_t>(infoDuplexOld.info[0].size * 8),
                static_cast<uint32_t>(infoDuplexNew.info[0].size * 8),
                fs::SubStorage(
                    &storageBufferedMasterA,
                    0,
                    infoDuplexNew.info[0].size
                ),
                fs::SubStorage(
                    &storageBufferedMasterB,
                    0,
                    infoDuplexNew.info[0].size
                )
            )
        );
    }

    // L1 マップが大きくなるなら拡張します。
    if( infoDuplexOld.info[1].size < infoDuplexNew.info[1].size )
    {
        // L1 ビットマップ拡張に必要なストレージを準備します。
        BufferedStorage storageBufferedL1A;
        NN_RESULT_DO(
            storageBufferedL1A.Initialize(
                 storageBitmapL1A,
                 pBuffer,
                 sizeBlockNew,
                 CacheCount
             )
         );

        BufferedStorage storageBufferedL1B;
        NN_RESULT_DO(
            storageBufferedL1B.Initialize(
                storageBitmapL1B,
                pBuffer,
                sizeBlockNew,
                CacheCount
            )
        );

        // L1 ビットマップを拡張します。
        NN_RESULT_DO(
            DuplexBitmapHolder::Expand(
                static_cast<uint32_t>(infoDuplexOld.info[1].size * 8),
                static_cast<uint32_t>(infoDuplexNew.info[1].size * 8),
                fs::SubStorage(
                    &storageBufferedL1A,
                    0,
                    infoDuplexNew.info[1].size
                ),
                fs::SubStorage(
                    &storageBufferedL1B,
                    0,
                    infoDuplexNew.info[1].size
                )
            )
        );
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        コンストラクタ
*/
HierarchicalDuplexStorage::HierarchicalDuplexStorage() NN_NOEXCEPT
    : m_SizeData(-1)
{
}

/**
* @brief        デストラクタ
*/
HierarchicalDuplexStorage::~HierarchicalDuplexStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        二重化ストレージをマウントします。
*
* @param[in]    info            二重ストレージ情報
* @param[in]    fileBitmap      二重化ビットマップストレージ
* @param[in]    bSelectBitmap   二重化ブロックのAブロック、Bブロックのどちらを書き換えるか?
* @param[in]    pBuffer         キャッシュ
* @param[in]    pLocker         排他制御用のロックオブジェクト
*
* @return       関数の処理結果を返します。
*
*               "bSelectBitmap" はストレージをコミットする(Close)度に、
*               "0->1->0..." とトグルしながら呼ばれることを期待しています。
*               同じ "bSelectBitmap" 値をトグルせず呼ぶことで、前回書き換えた
*               状態を巻き戻してオープンすることが出来ます。
*/
Result HierarchicalDuplexStorage::Initialize(
           const HierarchicalDuplexInformation& infoDuplex,
           fs::SubStorage storageMasterBitmapA,
           fs::SubStorage storageMasterBitmapB,
           fs::SubStorage storageBitmapL1A,
           fs::SubStorage storageBitmapL1B,
           fs::SubStorage storageDataA,
           fs::SubStorage storageDataB,
           bool bSelectBitmap,
           IBufferManager* pBuffer,
           os::Mutex* pLocker
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBuffer);

    Result result;

    m_pBuffer = pBuffer;
    m_pLocker = pLocker;
    m_IsWritten = false;

    // パラメーター確認
    NN_SDK_ASSERT(infoDuplex.info[0].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(infoDuplex.info[0].orderBlock == 0);
    NN_SDK_ASSERT(infoDuplex.info[1].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(infoDuplex.info[1].orderBlock > 0);
    NN_SDK_ASSERT(infoDuplex.info[2].size >= static_cast<int64_t>(0));
    NN_SDK_ASSERT(infoDuplex.info[2].orderBlock > 0);

    // サイズを確認します。
    {
        int64_t sizeMasterA = 0;
        int64_t sizeMasterB = 0;
        int64_t sizeL1A = 0;
        int64_t sizeL1B = 0;
        int64_t sizeDataA = 0;
        int64_t sizeDataB = 0;
        NN_RESULT_DO(storageMasterBitmapA.GetSize(&sizeMasterA));
        NN_RESULT_DO(storageMasterBitmapB.GetSize(&sizeMasterB));
        NN_RESULT_DO(storageBitmapL1A.GetSize(&sizeL1A));
        NN_RESULT_DO(storageBitmapL1B.GetSize(&sizeL1B));
        NN_RESULT_DO(storageDataA.GetSize(&sizeDataA));
        NN_RESULT_DO(storageDataB.GetSize(&sizeDataB));
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterA, infoDuplex.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeMasterB, infoDuplex.info[0].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1A, infoDuplex.info[1].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeL1B, infoDuplex.info[1].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeDataA, infoDuplex.info[2].size);
        NN_SDK_REQUIRES_GREATER_EQUAL(sizeDataB, infoDuplex.info[2].size);
    }

    // バッファキャッシュを挿入します。
    m_BufferedBitmapStorage[0] = fs::SubStorage(&storageMasterBitmapA, 0, infoDuplex.info[0].size);
    m_BufferedBitmapStorage[1] = fs::SubStorage(&storageMasterBitmapB, 0, infoDuplex.info[0].size);
    m_BufferedBitmapStorage[2] = fs::SubStorage(&storageBitmapL1A, 0, infoDuplex.info[1].size);
    m_BufferedBitmapStorage[3] = fs::SubStorage(&storageBitmapL1B, 0, infoDuplex.info[1].size);
    m_BufferedBitmapStorage[4] = fs::SubStorage(&storageDataA, 0, infoDuplex.info[2].size);
    m_BufferedBitmapStorage[5] = fs::SubStorage(&storageDataB, 0, infoDuplex.info[2].size);

    for( auto index = 0; index < BufferedBitmapCount; ++index )
    {
        result = m_BufferedStorage[index].Initialize(
                     m_pBuffer,
                     m_pLocker,
                     &m_BufferedBitmapStorage[index],
                     infoDuplex.info[index / 2].size,
                     static_cast<int64_t>(1) << infoDuplex.info[index / 2].orderBlock,
                     MaxCacheEntries,
                     false,            // isReadDataCache
                     IBufferManager::BufferLevelMin,
                     true,             // isKeepBurstMode
                     fs::StorageType_SaveData
                 );
    }
    if( result.IsFailure() )
    {
        for( auto index = 0; index < BufferedBitmapCount; ++index )
        {
            m_BufferedStorage[index].Finalize();
            m_BufferedBitmapStorage[index] = fs::SubStorage();
        }
        return result;
    }

    // マスタービットマップを束ねます。
    if( !bSelectBitmap )
    {
        // 参照系
        NN_SDK_ASSERT(infoDuplex.info[0].size >= static_cast<int64_t>(0));
        m_DuplexBitmapStorage[0].Initialize(
            static_cast<uint32_t>(infoDuplex.info[0].size * 8),
            fs::SubStorage(
                m_BufferedStorage + 0,
                0,
                infoDuplex.info[0].size
            ),
            fs::SubStorage(
                m_BufferedStorage + 0,
                0,
                infoDuplex.info[0].size
            )
        );

        // 更新系 (最初は読み込みモードでマウントします)
        NN_SDK_ASSERT(infoDuplex.info[0].size >= static_cast<int64_t>(0));
        m_DuplexBitmapStorage[1].InitializeForRead(
            static_cast<uint32_t>(infoDuplex.info[0].size * 8),
            fs::SubStorage(
                m_BufferedStorage + 1,
                0,
                infoDuplex.info[0].size
            ),
            fs::SubStorage(
                m_BufferedStorage + 0,
                0,
                infoDuplex.info[0].size
            )
        );
    }
    else
    {
        // 参照系
        NN_SDK_ASSERT(infoDuplex.info[0].size >= static_cast<int64_t>(0));
        m_DuplexBitmapStorage[0].Initialize(
            static_cast<uint32_t>(infoDuplex.info[0].size * 8),
            fs::SubStorage(
                m_BufferedStorage + 1,
                0,
                infoDuplex.info[0].size
            ),
            fs::SubStorage(
                m_BufferedStorage + 1,
                0,
                infoDuplex.info[0].size
            )
        );

        // 更新系 (最初は読み込みモードでマウントします)
        NN_SDK_ASSERT(infoDuplex.info[0].size >= static_cast<int64_t>(0));
        m_DuplexBitmapStorage[1].InitializeForRead(
            static_cast<uint32_t>(infoDuplex.info[0].size * 8),
            fs::SubStorage(
                m_BufferedStorage + 0,
                0,
                infoDuplex.info[0].size
            ),
            fs::SubStorage(
                m_BufferedStorage + 1,
                0,
                infoDuplex.info[0].size
            )
        );
    }

    // L1ビットマップをマウントします。
    // L1ビットマップはマスタビットマップx2と、実データ(ストライピング済み)x2から構成されます
    int64_t sizeStripingBlock = static_cast<int64_t>(1) << infoDuplex.info[1].orderBlock;
    m_DuplexBaseStorage[0] = fs::SubStorage(
                                 m_BufferedStorage + 2,
                                 0,
                                 infoDuplex.info[1].size
                             );
    m_DuplexBaseStorage[1] = fs::SubStorage(
                                 m_BufferedStorage + 3,
                                 0,
                                 infoDuplex.info[1].size
                             );

    fs::SubStorage storageData0(&m_DuplexBaseStorage[0], 0, infoDuplex.info[1].size);
    fs::SubStorage storageData1(&m_DuplexBaseStorage[1], 0, infoDuplex.info[1].size);

    // 参照系ストレージ
    m_DuplexStorage[0].Initialize(
        &m_DuplexBitmapStorage[0],
        storageData0,
        storageData1,
        sizeStripingBlock,
        m_pBuffer
    );
    m_DuplexStorage[0].SetReadOnly(true);

    // 更新系ストレージ
    m_DuplexStorage[1].Initialize(
        &m_DuplexBitmapStorage[1],
        storageData0,
        storageData1,
        sizeStripingBlock,
        m_pBuffer
    );

    // 更新系ビットマップ
    NN_SDK_ASSERT(infoDuplex.info[1].size >= static_cast<int64_t>(0));
    m_DuplexBitmapStorage[3].Initialize(
        static_cast<uint32_t>(infoDuplex.info[1].size * 8),
        fs::SubStorage(&m_DuplexStorage[1], 0, infoDuplex.info[1].size),
        fs::SubStorage(&m_DuplexStorage[0], 0, infoDuplex.info[1].size)
    );

    // L2ビットマップをマウントします。
    // L2ビットマップはL1ビットマップx2と、実データ(ストライピング済み)x2から構成されます
    sizeStripingBlock = static_cast<int64_t>(1) << infoDuplex.info[2].orderBlock;

    m_DuplexBaseStorage[2] = fs::SubStorage(
                                 m_BufferedStorage + 4,
                                 0,
                                 infoDuplex.info[2].size
                             );

    m_DuplexBaseStorage[3] = fs::SubStorage(
                                 m_BufferedStorage + 5,
                                 0,
                                 infoDuplex.info[2].size
                             );

    fs::SubStorage storageData2(&m_DuplexBaseStorage[2], 0, infoDuplex.info[2].size);
    fs::SubStorage storageData3(&m_DuplexBaseStorage[3], 0, infoDuplex.info[2].size);

    // 更新系ストレージ
    m_DuplexStorage[2].Initialize(
        &m_DuplexBitmapStorage[3],
        storageData2,
        storageData3,
        sizeStripingBlock,
        m_pBuffer
    );

    m_SizeData = infoDuplex.info[2].size;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        二重化ストレージをアンマウントします。
*/
void HierarchicalDuplexStorage::Finalize() NN_NOEXCEPT
{
    // 初期化済みの場合のみ終了処理を行ないます。
    if( m_SizeData >= 0LL )
    {
        m_SizeData = 0;

        // 開いた順に閉じます
        m_DuplexStorage[2].Finalize();
        m_DuplexStorage[1].Finalize();
        m_DuplexStorage[0].Finalize();

        for( auto& bufferedStorage : m_BufferedStorage )
        {
            bufferedStorage.Finalize();
        }

        for( auto& bufferedBitmapStorage : m_BufferedBitmapStorage )
        {
            bufferedBitmapStorage = fs::SubStorage();
        }

        m_SizeData = -1;

        // このフラグは Close 後に参照します。クリアしないでください。
        // (Close 中に立つ可能性があるため、Close 後に見る必要があります)
        // 正しく実装されていれば、Close 中に立つことはないはずです。
        // m_isWritten = false;
    }
}

/**
* @brief        実データの総バイトサイズを取得します。
*
* @param[out]   outValue    実データの総バイトサイズ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);
    NN_SDK_ASSERT(m_SizeData >= 0);
    *outValue = m_SizeData;
    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージレイヤーからデータを読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      読み込みはブロック単位でなくて構いません。
*/
Result HierarchicalDuplexStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SizeData >= 0LL);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    NN_RESULT_DO(m_DuplexStorage[2].Read(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

/**
* @brief        実データ領域にデータを書き込み、二重化選択ビットを更新します。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      最上位の二重化選択ビットが書き換えられる前に失敗が発生した場合には
*               次回使用時にロールバックが行われます。
*/
Result HierarchicalDuplexStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_SizeData >= 0LL);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    // 一度も書き込んでいない状態の Flush 要求では CopyOnWrite をスキップ
    if( !m_IsWritten && (0 < size) )
    {
        // 初回書き込み時にCopyを行います

        auto storageDst = m_DuplexBitmapStorage[1].GetUpdateStorage();
        auto storageSrc = m_DuplexBitmapStorage[1].GetOriginalStorage();

        int64_t sizeStorage = 0;
        NN_RESULT_DO(storageDst.GetSize(&sizeStorage));

        NN_RESULT_DO(
            CopyStorage(
                storageDst,
                storageSrc,
                sizeStorage
            )
        );

        m_DuplexBitmapStorage[1].RemountForWrite();

        m_IsWritten = true;

        storageDst = fs::SubStorage();
        storageSrc = fs::SubStorage();
    }

    NN_RESULT_DO(m_DuplexStorage[2].Write(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::Flush() NN_NOEXCEPT
{
    for( auto& bufferedStorage : m_BufferedStorage )
    {
        NN_RESULT_DO(bufferedStorage.Flush());
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    return m_DuplexStorage[2].OperateRange(
        outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
}

/**
* @brief        DuplexBitmap を反転せずに再マウントします。
*/
void HierarchicalDuplexStorage::Remount() NN_NOEXCEPT
{
    fs::SubStorage storageUpdate = m_DuplexBitmapStorage[1].GetUpdateStorage();
    fs::SubStorage storageOriginal = m_DuplexBitmapStorage[1].GetOriginalStorage();

    // (参照系)
    m_DuplexBitmapStorage[0].Finalize();
    m_DuplexBitmapStorage[0].Initialize(
        m_DuplexBitmapStorage[0].GetBlockCount(),
        storageOriginal,
        storageOriginal
    );

    // (更新系)
    m_DuplexBitmapStorage[1].Finalize();
    m_DuplexBitmapStorage[1].InitializeForRead(
        m_DuplexBitmapStorage[1].GetBlockCount(),
        storageUpdate,
        storageOriginal
    );

    m_IsWritten = false;
}

/**
* @brief        ストレージの内容をコピーします。
*
* @param[in]    pDst        コピー先のストレージ
* @param[in]    offsetDst   コピー先のストレージオフセット
* @param[in]    pSrc        コピー元のストレージ
* @param[in]    offsetSrc   コピー元のストレージオフセット
* @param[in]    size        コピーするサイズ
*
* @return       関数の処理結果を返します。
*/
Result HierarchicalDuplexStorage::CopyStorage(
           fs::SubStorage dst,
           fs::SubStorage src,
           int64_t size
       ) NN_NOEXCEPT
{
    // 最上段のビットマップをコピーする用途に使用しています。
    // 多くて数十バイト程度となるため、基本的にバッファリングは不要なはずです。

    uint8_t buf[32];
    int64_t offsetSrc = 0;
    int64_t offsetDst = 0;
    int64_t sizeRemain = size;
    while( sizeRemain > 0 )
    {
        size_t sizeTx = static_cast<size_t>(size);
        if( sizeTx > sizeof(buf) )
        {
            sizeTx = sizeof(buf);
        }
        NN_RESULT_DO(src.Read(offsetSrc, buf, sizeTx));
        NN_RESULT_DO(dst.Write(offsetDst, buf, sizeTx));
        sizeRemain -= sizeTx;
        offsetSrc += sizeTx;
        offsetDst += sizeTx;
    }

    NN_RESULT_SUCCESS;
}

}}}

