﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/dbm/fs_DuplexBitmap.h>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_DuplexStorage.h>

namespace nn { namespace fssystem { namespace save {

// アラインメントされているかどうかを判定します。
#define NN_DRIVER_FS_DUPLEXSTORAGE_ISALIGNED(V)     \
    if( ((V) & ((1<<m_CountShiftBlock) - 1)) != 0)  \
    {                                               \
        NN_SDK_ASSERT(!"Invalid Parametor");        \
        return nn::save::ResultInvalidPosition();   \
    }

/**
* @brief        コンストラクタ
*/
DuplexStorage::DuplexStorage() NN_NOEXCEPT
    : m_pDuplexBitmap(nullptr),
      m_CountShiftBlock(0),
      m_SizeBlock(0),
      m_pBufferManager(nullptr),
      m_IsReadOnly(false)
{
}

/**
* @brief デストラクタ
*/
DuplexStorage::~DuplexStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        二重化選択ストレージレイヤーを初期化します。
*
* @param[in]    pBitmap     二重化選択ビットマップデータ
* @param[in]    fileData1   実データストレージ 1
* @param[in]    fileData2   実データストレージ 2
* @param[in]    sizeBlock   ブロックサイズ
* @param[in]    pBuffer     バッファマネージャ
*
* @return       関数の処理結果を返します。
*/
void DuplexStorage::Initialize(
         dbm::DuplexBitmap* pBitmap,
         fs::SubStorage storageData1,
         fs::SubStorage storageData2,
         int64_t sizeBlock,
         IBufferManager* pBuffer
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pBitmap);
    NN_SDK_ASSERT_NOT_NULL(pBuffer);

    // 二つのストレージのサイズが等しいことを確認します。
    {
        int64_t size1 = 0;
        int64_t size2 = 0;
        NN_SDK_ASSERT(storageData1.GetSize(&size1).IsSuccess());
        NN_SDK_ASSERT(storageData2.GetSize(&size2).IsSuccess());
        NN_UNUSED(size1);
        NN_UNUSED(size2);
        NN_SDK_ASSERT_EQUAL(size1, size2);
    }

    m_pDuplexBitmap = pBitmap;
    m_pBufferManager = pBuffer;

    // 二つのストレージセットを初期化します。
    m_Storage[0] = storageData1;
    m_Storage[1] = storageData2;

    // ブロックサイズからシフト値を求めておきます。
    m_CountShiftBlock = ILog2(static_cast<uint32_t>(sizeBlock));
    NN_SDK_ASSERT((1LL << m_CountShiftBlock) == sizeBlock);

    m_SizeBlock = sizeBlock;
}

/**
* @brief        二重化ストレージレイヤーを閉じます。
*/
void DuplexStorage::Finalize() NN_NOEXCEPT
{
    if( m_pDuplexBitmap != nullptr )
    {
        // このクラスはアタッチメントクラスのため、
        // Finalize のみ行う。Closeしない。
        m_Storage[0] = fs::SubStorage();
        m_Storage[1] = fs::SubStorage();
        m_pDuplexBitmap = nullptr;
        m_pBufferManager = nullptr;
    }
}

/**
* @brief        実データ領域からデータを読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result DuplexStorage::Read(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pDuplexBitmap);

    Result result;
    Result verifyHashResult = ResultSuccess();

    int64_t sizeBytesData;
    NN_RESULT_DO(GetSize(&sizeBytesData));

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    NN_RESULT_THROW_UNLESS(nn::fs::IStorage::CheckOffsetAndSize(offset, size), nn::fs::ResultInvalidOffset());

    // 上限チェック
    if( offset > sizeBytesData )
    {
        return nn::fs::ResultInvalidOffset();
    }
    if( static_cast<int64_t>(offset + size) > sizeBytesData )
    {
        size = static_cast<size_t>(sizeBytesData - offset);
    }

    // ブロック単位に分けて読み込みます。
    int64_t offsetAligned = offset & (~(m_SizeBlock - 1LL));
    int64_t offsetAlignedEnd = (offset + size + m_SizeBlock - 1LL) & (~(m_SizeBlock - 1LL));
    NN_SDK_ASSERT(offsetAligned >= 0);
    NN_SDK_ASSERT(offsetAlignedEnd
        <= ((sizeBytesData + m_SizeBlock - 1LL) & (~(m_SizeBlock - 1LL))));
    uint8_t* dst = reinterpret_cast<uint8_t *>(buffer);

    size_t bitCount = static_cast<size_t>((offsetAlignedEnd - offsetAligned) >> m_CountShiftBlock);
    uint32_t index = static_cast<uint32_t>(offsetAligned >> m_CountShiftBlock);
    dbm::DuplexBitmap::Iterator itOriginal;
    m_pDuplexBitmap->IterateOriginalBegin(&itOriginal, index, bitCount);
    dbm::DuplexBitmap::Iterator itModified;
    m_pDuplexBitmap->IterateBegin(&itModified, index, bitCount);

    size_t sizeData = 0;
    size_t totalBitCount = 0;
    size_t prevZeroCountOriginal = 0;
    size_t prevOneCountOriginal = 0;
    size_t prevZeroCountModified = 0;
    size_t prevOneCountModified = 0;
    while( totalBitCount < bitCount )
    {
        NN_SDK_ASSERT(size > 0);

        // 二重化選択ビットを更新します。
        size_t zeroCountOriginal;
        size_t oneCountOriginal;
        if( (prevZeroCountOriginal == 0) && (prevOneCountOriginal == 0) )
        {
            NN_RESULT_DO(
                m_pDuplexBitmap->IterateOriginalNext(
                    &zeroCountOriginal,
                    &oneCountOriginal,
                    &itOriginal
                )
            );
        }
        else
        {
            zeroCountOriginal = prevZeroCountOriginal;
            oneCountOriginal = prevOneCountOriginal;
        }
        prevZeroCountOriginal = zeroCountOriginal;
        prevOneCountOriginal = oneCountOriginal;

        size_t zeroCountModified;
        size_t oneCountModified;
        if( (prevZeroCountModified == 0) && (prevOneCountModified == 0) )
        {
            NN_RESULT_DO(
                m_pDuplexBitmap->IterateNext(
                    &zeroCountModified,
                    &oneCountModified,
                    &itModified
                )
            );
        }
        else
        {
            zeroCountModified = prevZeroCountModified;
            oneCountModified = prevOneCountModified;
        }
        prevZeroCountModified = zeroCountModified;
        prevOneCountModified = oneCountModified;

        // 読み込む二重化ビット、ビットの連続数を取得します。
        uint8_t bit;
        size_t readBitCount;
        if( (zeroCountOriginal > 0) && (zeroCountModified > 0) )
        {
            // original:0, modified:0
            // 二重化されたデータに対して書き換え発生していない場合は、
            // 二重化選択ビットが示すインデックスのデータを読み込みます。
            bit = 0;
            readBitCount = (zeroCountOriginal < zeroCountModified) ?
                zeroCountOriginal : zeroCountModified;
            prevZeroCountModified -= readBitCount;
            prevZeroCountOriginal -= readBitCount;
        }
        else if( zeroCountOriginal > 0 )
        {
            // original:0, modified:1
            // 二重化されたデータに対して書き換え発生済みの場合は、
            // 書き換え後側のデータを読み込みます。
            bit = 1;
            readBitCount = (zeroCountOriginal < oneCountModified) ?
                zeroCountOriginal : oneCountModified;
            prevOneCountModified -= readBitCount;
            prevZeroCountOriginal -= readBitCount;
        }
        else if( zeroCountModified > 0 )
        {
            // original:1, modified:0
            // 二重化されたデータに対して書き換え発生済みの場合は、
            // 書き換え後側のデータを読み込みます。
            bit = 0;
            readBitCount = (oneCountOriginal < zeroCountModified) ?
                oneCountOriginal : zeroCountModified;
            prevZeroCountModified -= readBitCount;
            prevOneCountOriginal -= readBitCount;
        }
        else
        {
            // original:1, modified:1
            // 二重化されたデータに対して書き換え発生していない場合は、
            // 二重化選択ビットが示すインデックスのデータを読み込みます。
            bit = 1;
            readBitCount = (oneCountOriginal < oneCountModified) ?
                oneCountOriginal : oneCountModified;
            prevOneCountModified -= readBitCount;
            prevOneCountOriginal -= readBitCount;
        }

        // 読み込むデータサイズを取得します。
        size_t sizeRead = static_cast<size_t>(readBitCount * m_SizeBlock);

        // 最大読み込み可能なバイト数を抑制します
        if( offsetAligned != offset )
        {
            NN_SDK_ASSERT(offset > offsetAligned );
            NN_SDK_ASSERT(offset < offsetAligned + m_SizeBlock );
            sizeRead -= static_cast<size_t>(offset - offsetAligned);
        }

        // 余分なサイズを切り詰めます。
        if( sizeRead > size )
        {
            sizeRead = size;
        }
        if( static_cast<int64_t>(offset + sizeRead) > sizeBytesData )
        {
            sizeRead = static_cast<size_t>(sizeBytesData - offset);
        }

        // 二重化選択ビットが示すストレージからデータを読み込みます。
        {
            // 二重化選択ビットが示すストレージからデータを読み込みます。
            result = m_Storage[bit].Read(
                         offset,
                         dst,
                         static_cast<size_t>(sizeRead)
                     );
            if( result.IsFailure()
                && !(nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result)) )
            {
                return result;
            }
            if(nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(result) )
            {
                verifyHashResult = result;
                result = ResultSuccess();
            }

            size -= sizeRead;
            dst += sizeRead;
            sizeData += sizeRead;
            offset += sizeRead;
            offsetAligned += readBitCount * m_SizeBlock;
        }

        totalBitCount += readBitCount;
    }
    NN_SDK_ASSERT(size == 0);

    return verifyHashResult;
} // NOLINT(impl/function_size)

/**
* @brief        実データ領域にデータを書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      書き込みはブロック単位でのアクセスを強制します。
*               (ブロックキャッシュレイヤーを下に敷きます)
*/
Result DuplexStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pDuplexBitmap);

    // 読み込み専用ストレージに対して書き込みを行っていないかどうか確認します。
    NN_SDK_ASSERT(! m_IsReadOnly);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return fs::ResultNullptrArgument();
    }
    NN_RESULT_THROW_UNLESS(nn::fs::IStorage::CheckOffsetAndSize(offset, size), nn::fs::ResultInvalidOffset());

    Result result;

    int64_t sizeBitmap;
    NN_RESULT_DO(GetSize(&sizeBitmap));

    if( offset > sizeBitmap )
    {
        return nn::fs::ResultInvalidOffset();
    }
    //NN_SDK_ASSERT(offset + size < sizeBytesData  + (1LL << m_CountShiftBlock));

    if( static_cast<int64_t>(offset + size) > sizeBitmap )
    {
        size = static_cast<size_t>(sizeBitmap - offset);

        // 丸められた結果 size == 0 となった場合には何もしません。
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
    }

    // ブロック単位に分けて書き込みます。
    int64_t offsetAligned = offset & (~(m_SizeBlock - 1LL));
    int64_t offsetAlignedEnd = (offset + size + m_SizeBlock - 1LL) & (~(m_SizeBlock - 1LL));
    NN_SDK_ASSERT(offsetAligned >= 0);
    NN_SDK_ASSERT(offsetAlignedEnd <= ((sizeBitmap + m_SizeBlock - 1LL) & (~(m_SizeBlock - 1LL))));

    const char* src = reinterpret_cast<const char*>(buffer);

    std::unique_ptr<char[], nn::fs::detail::Deleter> memory
        = nn::fs::detail::MakeUnique<char[]>(static_cast<size_t>(m_SizeBlock));
    if( memory == nullptr )
    {
        return  nn::fs::ResultAllocationMemoryFailedInDuplexStorageA();
    }
    char* pBufOrg = memory.get();

    size_t sizeData = 0;   // 処理済サイズ
    while( offsetAligned < offsetAlignedEnd )
    {
        NN_SDK_ASSERT(offsetAligned <= offset);

        // データサイズからビット数を求めます
        uint32_t bitCount = static_cast<uint32_t>((size + m_SizeBlock - 1) >> m_CountShiftBlock);
        if( bitCount > 32 )
        {
            bitCount = 32;
        }

        // 2 重化選択ビットを読み込みます。
        uint32_t bitsOriginal;
        uint32_t bitsModified;
        NN_RESULT_DO(
            ReadDuplexBits(
                &bitsOriginal,
                &bitsModified,
                offsetAligned,
                bitCount
            )
        );
        uint32_t bitsModified0 = bitsModified;

        int64_t offsetAlignedStart = offsetAligned;
        for( size_t i = 0, ni = (i + 1); i < bitCount; i = ni, ni = (i + 1) )
        {
            char* pBuf = pBufOrg;

            // 書き込みサイズ
            size_t sizeWrite
                = std::min(static_cast<size_t>(m_SizeBlock - (offset - offsetAligned)), size);

            // これから書こうとしているブロックが変更済みかどうか
            bool modifiedFlag
                = !((bitsOriginal & (1 << (31 - i))) == (bitsModified & (1 << (31 - i))));
            uint32_t bitOriginal = ((bitsOriginal >> (31 - i)) & 0x01);
            uint32_t bitModified = (bitOriginal ^ 0x01);

            // 連続転送可能なサイズを計算します
            if( !modifiedFlag )
            {
                // オリジナルブロックの連続
                for( size_t j = ni; j < bitCount; ++j )
                {
                    if( (bitOriginal == ((bitsOriginal >> (31 - j)) & 0x01))
                     && ((bitsOriginal & (1 << (31 - j))) == (bitsModified & (1 << (31 - j)))) )
                    {
                        sizeWrite += static_cast<size_t>(m_SizeBlock);
                        ++ni;
                    }
                    else
                    {
                        break;
                    }
                }
            }
            else
            {
                // 書き換え済みブロックの連続
                for( size_t j = ni; j < bitCount; ++j )
                {
                    if( (bitOriginal == ((bitsOriginal >> (31 - j)) & 0x01)
                     && (((sizeWrite + m_SizeBlock) <= size) ||
                         (bitsOriginal & (1 << (31 - j))) != (bitsModified & (1 << (31 - j))))) )
                    {
                        sizeWrite += static_cast<size_t>(m_SizeBlock);
                        ++ni;
                    }
                    else
                    {
                        break;
                    }
                }
            }

            // 最大書き込みサイズに切り詰める
            if( size < sizeWrite )
            {
                sizeWrite = static_cast<size_t>(size);
            }
            // ストレージサイズに切り詰める
            if( static_cast<int64_t>(offset + sizeWrite) > sizeBitmap )
            {
                sizeWrite = static_cast<size_t>(sizeBitmap - offset);
            }

            // 書き換えが発生していない場合としている場合で処理を分離
            if( !modifiedFlag )
            {
                // 二重化されたデータに対して書き換え発生していません。
                // 先頭の半端ブロックが存在する場合はコピーオンライト
                if( offsetAligned < offset )
                {
                    size_t sizeBlock = static_cast<size_t>(m_SizeBlock);
                    if( static_cast<int64_t>(offsetAligned + sizeBlock) > sizeBitmap )
                    {
                        sizeBlock = static_cast<size_t>(sizeBitmap - offsetAligned);
                    }

                    NN_RESULT_DO(
                        m_Storage[bitOriginal].Read(
                            offsetAligned,
                            pBuf,
                            sizeBlock
                        )
                    );

                    // バッファに上書き部分のみをコピーします。
                    size_t sizeReal
                        = std::min(
                              static_cast<size_t>(sizeBlock - (offset - offsetAligned)),
                              sizeWrite
                          );
                    std::memcpy(pBuf + (offset - offsetAligned), src, sizeReal);

                    // 二重化選択ビットの値の逆側にデータを書き込みます。
                    NN_RESULT_DO(
                        m_Storage[bitModified].Write(
                            offsetAligned,
                            pBuf,
                            sizeBlock
                        )
                    );

                    // オフセット、ソースポインタ、書き込みサイズを更新
                    size -= sizeReal;
                    src += sizeReal;
                    offset += sizeReal;
                    sizeData += sizeReal;
                    offsetAligned += m_SizeBlock;
                    sizeWrite -= sizeReal;
                }

                // ブロックサイズの整数倍でバースト転送
                if( 0 < (sizeWrite >> m_CountShiftBlock) )
                {
                    NN_SDK_ASSERT(offset == offsetAligned);
                    size_t sizeBurst = (sizeWrite >> m_CountShiftBlock) << m_CountShiftBlock;

                    // 二重化選択ビットの値の逆側にデータを書き込みます。
                    NN_RESULT_DO(
                        m_Storage[bitModified].Write(
                            offsetAligned,
                            src,
                            sizeBurst
                        )
                    );

                    // オフセット、ソースポインタ、書き込みサイズを更新
                    size -= sizeBurst;
                    src += sizeBurst;
                    sizeData += sizeBurst;
                    offset += sizeBurst;
                    offsetAligned += sizeBurst;
                    sizeWrite -= sizeBurst;
                }

                // 末尾の半端ブロックが存在する場合はコピーオンライト
                if( 0 < sizeWrite )
                {
                    NN_SDK_ASSERT(offset == offsetAligned);
                    NN_SDK_ASSERT(static_cast<int64_t>(sizeWrite) < m_SizeBlock);
                    NN_SDK_ASSERT(static_cast<int64_t>(offsetAligned + sizeWrite) <= sizeBitmap);

                    size_t sizeBlock = static_cast<size_t>(m_SizeBlock);
                    if( static_cast<int64_t>(offsetAligned + sizeBlock) > sizeBitmap )
                    {
                        sizeBlock = static_cast<size_t>(sizeBitmap - offsetAligned);
                    }

                    NN_RESULT_DO(
                        m_Storage[bitOriginal].Read(
                            offsetAligned,
                            pBuf,
                            sizeBlock
                        )
                    );

                    // バッファに上書き部分のみをコピーします。
                    size_t sizeReal = std::min(sizeBlock, sizeWrite);
                    std::memcpy(pBuf, src, sizeReal);

                    // 二重化選択ビットの値の逆側にデータを書き込みます。
                    NN_RESULT_DO(
                        m_Storage[bitModified].Write(
                            offsetAligned,
                            pBuf,
                            sizeBlock
                        )
                    );

                    // オフセット、ソースポインタ、書き込みサイズを更新
                    size -= sizeReal;
                    src += sizeReal;
                    offset += sizeReal;
                    sizeData += sizeReal;
                    offsetAligned += m_SizeBlock;
                    sizeWrite -= sizeReal;
                }
            }
            else
            {
                // 既に書き込み済みの場合は直接書き込みに行って構わない
                {
                    NN_RESULT_DO(
                        m_Storage[bitModified].Write(
                            offset,
                            src,
                            sizeWrite
                        )
                    );
                }

                size -= sizeWrite;
                src += sizeWrite;
                offset += sizeWrite;
                sizeData += sizeWrite;
                offsetAligned += (ni - i) * m_SizeBlock;
                sizeWrite = 0;
            }

            // 書き換え後のビットを反転します。(original の反転に強制する)
            for( size_t j = i; j < ni; ++j )
            {
                uint32_t bitMask = (1 << (31 - j));

                bitsModified = (bitsOriginal & bitMask)
                    ? (bitsModified & ~bitMask) : (bitsModified | bitMask);
            }
        }

        // 書き換えにより発生したビット反転を反映します。
        if( bitsModified0 != bitsModified )
        {
#ifndef NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK
            {
                uint32_t tOriginal;
                uint32_t tModified;
                NN_RESULT_DO(
                    ReadDuplexBits(
                        &tOriginal,
                        &tModified,
                        offsetAlignedStart,
                        bitCount
                    )
                );
                NN_SDK_ASSERT(tOriginal == bitsOriginal);
                NN_SDK_ASSERT(tModified == bitsModified0);
            }
#endif

            NN_RESULT_DO(
                UpdateModifiedBits(
                    offsetAlignedStart,
                    offsetAligned
                )
            );

#ifndef NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK
            {
                uint32_t tOriginal;
                uint32_t tModified;
                NN_RESULT_DO(
                    ReadDuplexBits(
                        &tOriginal,
                        &tModified,
                        offsetAlignedStart,
                        bitCount
                    )
                );
                NN_SDK_ASSERT(tOriginal == bitsOriginal);
                NN_SDK_ASSERT(tModified == bitsModified);
            }
#endif
        }

    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        総バイトサイズを取得します。
*
* @param[out]   outValue    総バイトサイズ
*
* @return       関数の処理結果を返します。
*/
Result DuplexStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pDuplexBitmap);
    return m_Storage[0].GetSize(outValue);
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result DuplexStorage::Flush() NN_NOEXCEPT
{
    NN_RESULT_DO(m_Storage[0].Flush());
    NN_RESULT_DO(m_Storage[1].Flush());
    NN_RESULT_DO(m_pDuplexBitmap->Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result DuplexStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::Invalidate:
        {
            NN_RESULT_DO(m_Storage[0].OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));
            NN_RESULT_DO(m_Storage[1].OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));
            return m_pDuplexBitmap->Invalidate(
                static_cast<uint32_t>(offset >> m_CountShiftBlock),
                static_cast<size_t>((size + m_SizeBlock - 1) >> m_CountShiftBlock));
        }
    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

/**
* @brief        指定したブロックでの二重化選択ビットの値を読み込みます。
*
* @param[out]   outOriginalBits オリジナルビットデータ
* @param[out]   outModifiedBits 編集後ビットデータ
* @param[in]    offset          オフセット
* @param[in]    bitCount        読み込むビット数
*
* @return       関数の処理結果を返します。
*/
Result DuplexStorage::ReadDuplexBits(
           uint32_t* outOriginalBits,
           uint32_t* outModifiedBits,
           int64_t offset,
           size_t bitCount
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outOriginalBits);
    NN_SDK_ASSERT_NOT_NULL(outModifiedBits);
    NN_SDK_ASSERT(0 == (offset & (m_SizeBlock - 1)));
    uint32_t index = static_cast<uint32_t>(offset >> m_CountShiftBlock);

    NN_SDK_ASSERT(bitCount <= 32);
    NN_RESULT_DO(m_pDuplexBitmap->ReadOriginalBitmap32(outOriginalBits, index));
    NN_RESULT_DO(m_pDuplexBitmap->ReadBitmap32(outModifiedBits, index));

    if( bitCount != 32 )
    {
        uint32_t mask = ~((1u << (32 - bitCount)) - 1);
        *outOriginalBits = (*outOriginalBits) & mask;
        *outModifiedBits = (*outModifiedBits) & mask;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        上位レイヤーでの二重化選択ビットマップデータを更新します。
*
* @param[in]    offset      オフセット
* @param[in]    offsetEnd   変更状態を示すビット列
* @param[in]    option      書き込みオプション
*
* @return       関数の処理結果を返します。
*/
Result DuplexStorage::UpdateModifiedBits(
           int64_t offset,
           int64_t offsetEnd
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(0 == (offset & (m_SizeBlock - 1)));
    NN_SDK_ASSERT(0 == (offsetEnd & (m_SizeBlock - 1)));

    uint32_t index = static_cast<uint32_t>(offset >> m_CountShiftBlock);
    uint32_t indexEnd = static_cast<uint32_t>(offsetEnd >> m_CountShiftBlock);

    NN_RESULT_DO(m_pDuplexBitmap->MarkModified(index, indexEnd - index));

    NN_RESULT_SUCCESS;
}

}}}

