﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>

namespace nn { namespace fssystem {

NN_DEFINE_STATIC_CONSTANT(const int Hash::Size);

NN_DEFINE_STATIC_CONSTANT(const uint32_t NcaHeader::Signature);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::Size);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::FsCountMax);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::HeaderSignSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::HeaderSignCount);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::EncryptedKeyAreaSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::RightIdSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::Log2SectorSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaHeader::XtsBlockSize);

NN_DEFINE_STATIC_CONSTANT(const int NcaFsHeader::Size);
NN_DEFINE_STATIC_CONSTANT(const int NcaFsHeader::HashDataOffset);

const int NcaFsHeader::HashData::IntegrityMetaInfo::MasterHashOffset =
offsetof(NcaFsHeader, hashData.integrityMetaInfo.masterHash);
const int NcaFsHeader::HashData::HierarchicalSha256Data::MasterHashOffset =
offsetof(NcaFsHeader, hashData.hierarchicalSha256Data.fsDataMasterHash);

NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::Rsa2048KeyModulusSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::Rsa2048KeyPublicExponentSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::Rsa2048KeyPrivateExponentSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::Aes128KeySize);
NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount);
NN_DEFINE_STATIC_CONSTANT(const int NcaCryptoConfiguration::EncryptionKeyCount);

NN_DEFINE_STATIC_CONSTANT(const int NcaBucketInfo::HeaderSize);
NN_DEFINE_STATIC_CONSTANT(const int NcaPatchInfo::Size);
NN_DEFINE_STATIC_CONSTANT(const int NcaPatchInfo::Offset);

NN_STATIC_ASSERT(sizeof(NcaHeader) == NcaHeader::Size);
NN_STATIC_ASSERT(sizeof(NcaFsHeader) == NcaFsHeader::Size);
NN_STATIC_ASSERT(sizeof(NcaPatchInfo) == NcaPatchInfo::Size);
NN_STATIC_ASSERT(sizeof(BucketTree::Header) == NcaBucketInfo::HeaderSize);
NN_STATIC_ASSERT(offsetof(NcaFsHeader, hashData) == NcaFsHeader::HashDataOffset);
NN_STATIC_ASSERT(offsetof(NcaFsHeader, patchInfo) == NcaPatchInfo::Offset);
NN_STATIC_ASSERT(AesXtsStorage::AesBlockSize == AesCtrStorage::BlockSize);

namespace {

template< typename T >
inline bool IsIncluded(T value, T min, T max) NN_NOEXCEPT
{
    return min <= value && value <= max;
}

template< typename T >
inline bool IsZero(const T& buffer) NN_NOEXCEPT
{
    const char Zero[sizeof(T)] = {};
    return nn::crypto::IsSameBytes(&buffer, Zero, sizeof(Zero));
}

template< int Offset, typename T >
inline bool IsZero(const T& buffer) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(Offset < sizeof(T));
    const char Zero[sizeof(T) - Offset] = {};
    const void* const ptr = &buffer;
    return nn::crypto::IsSameBytes(reinterpret_cast<const char*>(ptr) + Offset, Zero, sizeof(Zero));
}

template< int Length, typename T >
inline bool IsZero(const T (&buffer)[Length]) NN_NOEXCEPT
{
    const char Zero[sizeof(T) * Length] = {};
    return nn::crypto::IsSameBytes(buffer, Zero, sizeof(Zero));
}

template< int Offset, int Length, typename T >
inline bool IsZero(const T (&buffer)[Length]) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(Offset < sizeof(T) * Length);
    const char Zero[sizeof(T) * Length - Offset] = {};
    const void* const ptr = buffer;
    return nn::crypto::IsSameBytes(reinterpret_cast<const char*>(ptr) + Offset, Zero, sizeof(Zero));
}

}

bool NcaPatchInfo::HasIndirectTable() const NN_NOEXCEPT
{
    return this->indirectSize != 0;
}

bool NcaPatchInfo::HasAesCtrExTable() const NN_NOEXCEPT
{
    return this->aesCtrExSize != 0;
}

uint8_t NcaHeader::GetProperKeyGeneration() const NN_NOEXCEPT
{
    return std::max(this->keyGeneration, this->keyGeneration2);
}

Result NcaHeader::Verify() const NN_NOEXCEPT
{
    static const uint32_t MaskNcaSignature = 0x00FFFFFF;
    static const uint32_t MaskNumSignature = 0xFF000000;
    static const uint32_t NcaSignature = Signature & MaskNcaSignature;
    static const uint32_t NumSignature = Signature & MaskNumSignature;

    NN_RESULT_THROW_UNLESS(
        (this->signature & MaskNcaSignature) == NcaSignature &&
        (this->signature & MaskNumSignature) <= NumSignature,
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsIncluded(
            this->distributionType, DistributionType::Download, DistributionType::GameCard),
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsIncluded(this->contentType, ContentType::Program, ContentType::PublicData),
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        this->keyIndex < NcaCryptoConfiguration::KeyAreaEncryptionKeyIndexCount,
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        0 < this->programId,
        fs::ResultInvalidNcaHeader()
    );

    // contentIndex はチェックしない

    NN_RESULT_THROW_UNLESS(
        0 < this->sdkAddonVersion,
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsZero(this->reserve1),
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsZero(this->reserve2),
        fs::ResultInvalidNcaHeader()
    );

    // rightsId はチェックしない

    uint64_t sector = std::numeric_limits<uint64_t>::max();
    for( int i = 0; i < FsCountMax; ++i )
    {
        if( this->fsInfo[i].startSector != 0 || this->fsInfo[i].endSector != 0 )
        {
            if( sector == std::numeric_limits<uint64_t>::max() )
            {
                sector = this->fsInfo[i].endSector;
            }

            NN_RESULT_THROW_UNLESS(
                sector == this->fsInfo[i].endSector &&
                this->fsInfo[i].startSector < this->fsInfo[i].endSector,
                fs::ResultInvalidNcaHeader()
            );

            sector = this->fsInfo[i].startSector;

            // TODO: 現状 hashTargetSize は 固定値
            NN_RESULT_THROW_UNLESS(
                this->fsInfo[i].hashTargetSize == ByteToSector(NcaFsHeader::Size),
                fs::ResultInvalidNcaHeader()
            );
        }
        else
        {
            NN_RESULT_THROW_UNLESS(
                this->fsInfo[i].hashTargetSize == 0,
                fs::ResultInvalidNcaHeader()
            );
        }

        NN_RESULT_THROW_UNLESS(
            this->fsInfo[i].reserve == 0,
            fs::ResultInvalidNcaHeader()
        );
    }

    // 鍵のない部分は 0 埋めされている
    const auto Offset = DecryptionKey_Count * NcaCryptoConfiguration::Aes128KeySize;
    NN_RESULT_THROW_UNLESS(
        IsZero<Offset>(this->encryptedKey),
        fs::ResultInvalidNcaHeader()
    );

    NN_RESULT_SUCCESS;
}

Result NcaFsHeader::Verify() const NN_NOEXCEPT
{
    // TODO: 現在は 2 で固定
    NN_RESULT_THROW_UNLESS(
        this->version == 2, fs::ResultInvalidNcaFsHeader());

    NN_RESULT_THROW_UNLESS(
        IsIncluded(this->fsType, FsType::RomFs, FsType::PartitionFs),
        fs::ResultInvalidNcaFsHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsIncluded(this->hashType, HashType::None, HashType::HierarchicalIntegrityHash),
        fs::ResultInvalidNcaFsHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsIncluded(this->encryptionType, EncryptionType::None, EncryptionType::AesCtrEx),
        fs::ResultInvalidNcaFsHeader()
    );

    NN_RESULT_THROW_UNLESS(
        IsZero(this->reserved), fs::ResultInvalidNcaFsHeader());

    switch( this->hashType )
    {
    case HashType::HierarchicalSha256Hash:
        NN_RESULT_DO(this->hashData.hierarchicalSha256Data.Verify());

        NN_RESULT_THROW_UNLESS(
            IsZero<sizeof(this->hashData.hierarchicalSha256Data)>(this->hashData),
            fs::ResultInvalidNcaFsHeader()
        );
        break;

    case HashType::HierarchicalIntegrityHash:
        NN_RESULT_DO(this->hashData.integrityMetaInfo.Verify());

        NN_RESULT_THROW_UNLESS(
            IsZero<sizeof(this->hashData.integrityMetaInfo)>(this->hashData),
            fs::ResultInvalidNcaFsHeader()
        );
        break;

    default:
        NN_RESULT_THROW_UNLESS(
            IsZero(this->hashData), fs::ResultInvalidNcaFsHeader());
        break;
    }

    // TODO: 現状 AesCtrEx 暗号の場合は IndirectStorage/AesCtrCounterExtendedStorage を必ず含む
    if( this->encryptionType == EncryptionType::AesCtrEx )
    {
        NN_RESULT_THROW_UNLESS(
            0 <= this->patchInfo.indirectOffset, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            0 < this->patchInfo.indirectSize, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            !IsZero(this->patchInfo.indirectHeader), fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            0 <= this->patchInfo.aesCtrExOffset, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            0 < this->patchInfo.aesCtrExSize, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            !IsZero(this->patchInfo.aesCtrExHeader), fs::ResultInvalidNcaFsHeader());
    }
    else
    {
        NN_RESULT_THROW_UNLESS(
            IsZero(this->patchInfo), fs::ResultInvalidNcaFsHeader());
    }

    if( this->encryptionType != EncryptionType::AesCtr &&
        this->encryptionType != EncryptionType::AesCtrEx )
    {
        NN_RESULT_THROW_UNLESS(
            IsZero(this->aesCtrUpperIv), fs::ResultInvalidNcaFsHeader());
    }

    if( this->sparseInfo.generation != 0 )
    {
        NN_RESULT_THROW_UNLESS(0 <= this->sparseInfo.bucket.offset, fs::ResultInvalidNcaFsHeader());
        NN_RESULT_THROW_UNLESS(0 <= this->sparseInfo.bucket.size, fs::ResultInvalidNcaFsHeader());
        NN_RESULT_THROW_UNLESS(0 <= this->sparseInfo.physicalOffset, fs::ResultInvalidNcaFsHeader());
        NN_RESULT_THROW_UNLESS(IsZero(this->sparseInfo._reserved), fs::ResultInvalidNcaFsHeader());

        BucketTree::Header header;
        std::memcpy(&header, this->sparseInfo.bucket.header, sizeof(header));
        NN_RESULT_DO(header.Verify());
    }
    else
    {
        NN_RESULT_THROW_UNLESS(IsZero(this->sparseInfo), fs::ResultInvalidNcaFsHeader());
    }

    NN_RESULT_THROW_UNLESS(
        IsZero(this->reserved2), fs::ResultInvalidNcaFsHeader());

    NN_RESULT_SUCCESS;
}

Result NcaFsHeader::HashData::HierarchicalSha256Data::Verify() const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !IsZero(this->fsDataMasterHash), fs::ResultInvalidNcaFsHeader());

    NN_RESULT_THROW_UNLESS(
        0 < this->hashBlockSize && nn::util::ispow2(this->hashBlockSize),
        fs::ResultInvalidNcaFsHeader()
    );

    // TODO: 現状は 2 段固定
    NN_RESULT_THROW_UNLESS(
        this->hashLayerCount == 2, fs::ResultInvalidNcaFsHeader());

    int64_t offset = 0;
    for( int i = 0; i < this->hashLayerCount; ++i )
    {
        NN_RESULT_THROW_UNLESS(
            offset <= this->hashLayerRegion[i].offset, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            0 < this->hashLayerRegion[i].size, fs::ResultInvalidNcaFsHeader());

        offset = this->hashLayerRegion[i].offset + this->hashLayerRegion[i].size;
    }

    for( int i = this->hashLayerCount; i < HashLayerCountMax; ++i )
    {
        NN_RESULT_THROW_UNLESS(
            IsZero(this->hashLayerRegion[i]), fs::ResultInvalidNcaFsHeader());
    }

    NN_RESULT_SUCCESS;
}

Result NcaFsHeader::HashData::IntegrityMetaInfo::Verify() const NN_NOEXCEPT
{
    // TODO: 適宜修正する
    const uint32_t Magic = 0x43465649;
    const uint32_t Version = 0x00020000;

    NN_RESULT_THROW_UNLESS(
        this->magic == Magic, fs::ResultInvalidNcaFsHeader());

    NN_RESULT_THROW_UNLESS(
        this->version == Version, fs::ResultInvalidNcaFsHeader());

    NN_RESULT_THROW_UNLESS(
        this->sizeMasterHash == Hash::Size, fs::ResultInvalidNcaFsHeader());

    const auto& info = this->infoLevelHash;

    NN_RESULT_THROW_UNLESS(
        IsIncluded<size_t>(
            info.maxLayers, save::IntegrityMinLayerCount, save::IntegrityMaxLayerCount),
        fs::ResultInvalidNcaFsHeader()
    );

    int64_t offset = 0;
    for( uint32_t i = 0; i < info.maxLayers - 1; ++i )
    {
        const auto& layer = info.info[i];

        NN_RESULT_THROW_UNLESS(0 < layer.orderBlock, fs::ResultInvalidNcaFsHeader());

        const auto alignment = 1 << layer.orderBlock;

        NN_RESULT_THROW_UNLESS(
            offset <= layer.offset && util::is_aligned(layer.offset, alignment),
            fs::ResultInvalidNcaFsHeader()
        );

        NN_RESULT_THROW_UNLESS(
            0 < layer.size, fs::ResultInvalidNcaFsHeader());

        NN_RESULT_THROW_UNLESS(
            IsZero(layer.reserved), fs::ResultInvalidNcaFsHeader());

        offset = layer.offset + layer.size;
    }

    typedef InfoLevelHash::HierarchicalIntegrityVerificationLevelInformation LayerInfo;
    for( uint32_t i = info.maxLayers - 1; i < LayerInfo::IntegrityMaxLayerCount - 1; ++i )
    {
        NN_RESULT_THROW_UNLESS(
            IsZero(info.info[i]), fs::ResultInvalidNcaFsHeader());
    }

    // seed はチェックしない

    NN_RESULT_SUCCESS;
}

}}
