﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <nn/nn_Version.h>
#include <nn/fssystem/fs_AesCtrCounterExtendedStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/fs_IntegrityRomFsStorage.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/fs_NcaFileSystemDriverUtility.h>
#include <nn/fssystem/fs_SparseStorage.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/save/fs_BufferedStorage.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include <nn/fssystem/fs_SpeedEmulationConfiguration.h>
#include <nn/result/result_HandlingUtility.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/fssystem/utilTool/fs_NcaSparseStorage.h>
#include <nn/fssystem/utilTool/fs_SparseStorageBuilder.h>
#endif

#include "fs_HierarchicalSha256Storage.h"
#include "fs_ReadOnlyBlockCacheStorage.h"

namespace nn { namespace fssystem {

namespace {

typedef NcaFsHeader::Region Sha256DataRegion;
typedef NcaFsHeader::HashData::IntegrityMetaInfo::InfoLevelHash IntegrityLevelInfo;
typedef IntegrityLevelInfo::HierarchicalIntegrityVerificationLevelInformation IntegrityDataInfo;

inline const Sha256DataRegion& GetSha256DataRegion(const NcaFsHeader::HashData& data) NN_NOEXCEPT
{
    return data.hierarchicalSha256Data.hashLayerRegion[1];
}

inline const IntegrityDataInfo& GetIntegrityDataInfo(const NcaFsHeader::HashData& data) NN_NOEXCEPT
{
    return data.integrityMetaInfo.infoLevelHash.info[data.integrityMetaInfo.infoLevelHash.maxLayers - 2];
}

// AesXtsStorage 向けの IV を生成します。
inline void MakeAesXtsIv(void* pCounter, int64_t baseOffset) NN_NOEXCEPT
{
    util::BytePtr ptr(pCounter);
    ptr.Advance(sizeof(int64_t));

    util::StoreBigEndian<int64_t>(ptr.Get<int64_t>(), baseOffset / NcaHeader::XtsBlockSize);
}

// スピードエミュレーションのために HwAes を使用するかどうか (内部鍵のとき)
inline bool IsUsingHwAesCtrForSpeedEmulation() NN_NOEXCEPT
{
    // GetSpeedEmulationMode() が十分高速な実装であることを前提として毎 read チェックで switch する
    auto mode = fssystem::SpeedEmulationConfiguration::GetSpeedEmulationMode();
    return mode == fs::SpeedEmulationMode::None || mode == fs::SpeedEmulationMode::Slower;
}

inline int64_t GetFsOffset(const NcaReader& reader, int fsIndex) NN_NOEXCEPT
{
    // NOTE: GetFsOffset() の実装上、負値になることはない
    return static_cast<int64_t>(reader.GetFsOffset(fsIndex));
}

inline int64_t GetFsEndOffset(const NcaReader& reader, int fsIndex) NN_NOEXCEPT
{
    // NOTE: GetFsEndOffset() の実装上、負値になることはない
    return static_cast<int64_t>(reader.GetFsEndOffset(fsIndex));
}

// バッファのラッパクラス
class BufferHolder
{
    NN_DISALLOW_COPY(BufferHolder);

public:
    BufferHolder() NN_NOEXCEPT
        : m_pAllocator(nullptr)
        , m_pBuffer(nullptr)
        , m_BufferSize(0)
    {
    }

    BufferHolder(MemoryResource* pAllocator, size_t size) NN_NOEXCEPT
        : m_pAllocator(pAllocator)
        , m_pBuffer(reinterpret_cast<char*>(pAllocator->allocate(size)))
        , m_BufferSize(size)
    {
    }

    NN_IMPLICIT BufferHolder(BufferHolder&& obj) NN_NOEXCEPT
        : m_pAllocator(obj.m_pAllocator)
        , m_pBuffer(obj.m_pBuffer)
        , m_BufferSize(obj.m_BufferSize)
    {
        obj.m_pBuffer = nullptr;
    }

    ~BufferHolder() NN_NOEXCEPT
    {
        if( m_pBuffer != nullptr )
        {
            m_pAllocator->deallocate(m_pBuffer, m_BufferSize);
            m_pBuffer = nullptr;
        }
    }

    BufferHolder& operator=(BufferHolder&& obj) NN_NOEXCEPT
    {
        if( this != &obj )
        {
            NN_SDK_ASSERT(m_pBuffer == nullptr);

            m_pAllocator = obj.m_pAllocator;
            m_pBuffer = obj.m_pBuffer;
            m_BufferSize = obj.m_BufferSize;

            obj.m_pBuffer = nullptr;
        }
        return *this;
    }

    bool IsValid() const NN_NOEXCEPT
    {
        return m_pBuffer != nullptr;
    }

    char* Get() const NN_NOEXCEPT
    {
        return m_pBuffer;
    }

    size_t GetSize() const NN_NOEXCEPT
    {
        return m_BufferSize;
    }

private:
    MemoryResource* m_pAllocator;
    char* m_pBuffer;
    size_t m_BufferSize;
};

// ストレージのリソース管理クラス
template< typename TBase, int N >
class DerivedStorageHolder : public TBase
{
    NN_DISALLOW_COPY(DerivedStorageHolder);

public:
    typedef std::unique_ptr<fs::IStorage> StoragePtr;

public:
    DerivedStorageHolder() NN_NOEXCEPT
        : TBase()
        , m_pNcaReader()
        , m_pStorages()
    {
    }

    explicit DerivedStorageHolder(std::shared_ptr<NcaReader> pNcaReader) NN_NOEXCEPT
        : TBase()
        , m_pNcaReader(pNcaReader)
        , m_pStorages()
    {
    }

#define DRIVED_STORAGE_HOLDER_CONSTRUCTOR(n) \
    template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
    explicit DerivedStorageHolder(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x)) NN_NOEXCEPT \
        : TBase(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)) \
        , m_pNcaReader() \
        , m_pStorages() \
    { \
    } \
    template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
    explicit DerivedStorageHolder(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x), std::shared_ptr<NcaReader> pNcaReader) NN_NOEXCEPT \
        : TBase(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)) \
        , m_pNcaReader(pNcaReader) \
        , m_pStorages() \
    { \
    }

    NN_UTIL_VARIADIC_DEFINE_MACROS(DRIVED_STORAGE_HOLDER_CONSTRUCTOR)
#undef DRIVED_STORAGE_HOLDER_CONSTRUCTOR

    void Set(StoragePtr&& ptr0) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N == 1);

        m_pStorages[0] = std::move(ptr0);
    }

    void Set(StoragePtr&& ptr0, StoragePtr&& ptr1) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N == 2);

        m_pStorages[0] = std::move(ptr0);
        m_pStorages[1] = std::move(ptr1);
    }

    void Set(StoragePtr&& ptr0, StoragePtr&& ptr1, StoragePtr&& ptr2) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N == 3);

        m_pStorages[0] = std::move(ptr0);
        m_pStorages[1] = std::move(ptr1);
        m_pStorages[2] = std::move(ptr2);
    }

    void Set(StoragePtr&& ptr0, StoragePtr&& ptr1, StoragePtr&& ptr2, StoragePtr&& ptr3) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N == 4);

        m_pStorages[0] = std::move(ptr0);
        m_pStorages[1] = std::move(ptr1);
        m_pStorages[2] = std::move(ptr2);
        m_pStorages[3] = std::move(ptr3);
    }

    void Set(StoragePtr&& ptr0, StoragePtr&& ptr1, StoragePtr&& ptr2, StoragePtr&& ptr3, StoragePtr&& ptr4) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(N == 5);

        m_pStorages[0] = std::move(ptr0);
        m_pStorages[1] = std::move(ptr1);
        m_pStorages[2] = std::move(ptr2);
        m_pStorages[3] = std::move(ptr3);
        m_pStorages[4] = std::move(ptr4);
    }

private:
    std::shared_ptr<NcaReader> m_pNcaReader;
    std::array<StoragePtr, N> m_pStorages;
};

// ストレージのリソース管理クラス（N = 0 の特殊化）
template< typename TBase >
class DerivedStorageHolder<TBase, 0> : public TBase
{
    NN_DISALLOW_COPY(DerivedStorageHolder);

public:
    typedef std::unique_ptr<fs::IStorage> StoragePtr;

public:
    DerivedStorageHolder() NN_NOEXCEPT
        : TBase()
        , m_pNcaReader()
    {
    }

    explicit DerivedStorageHolder(std::shared_ptr<NcaReader> pNcaReader) NN_NOEXCEPT
        : TBase()
        , m_pNcaReader(pNcaReader)
    {
    }

#define DRIVED_STORAGE_HOLDER_CONSTRUCTOR(n) \
    template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
    explicit DerivedStorageHolder(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x)) NN_NOEXCEPT \
        : TBase(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)) \
        , m_pNcaReader() \
    { \
    } \
    template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
    explicit DerivedStorageHolder(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x), std::shared_ptr<NcaReader> pNcaReader) NN_NOEXCEPT \
        : TBase(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)) \
        , m_pNcaReader(pNcaReader) \
    { \
    }

    NN_UTIL_VARIADIC_DEFINE_MACROS(DRIVED_STORAGE_HOLDER_CONSTRUCTOR)
#undef DRIVED_STORAGE_HOLDER_CONSTRUCTOR

private:
    std::shared_ptr<NcaReader> m_pNcaReader;
};

// バッファ付きストレージのリソース管理クラス
template< typename TBase, int N >
class DerivedStorageHolderWithBuffer : public DerivedStorageHolder<TBase, N>
{
    NN_DISALLOW_COPY(DerivedStorageHolderWithBuffer);

    typedef DerivedStorageHolder<TBase, N> BaseType;

public:
    DerivedStorageHolderWithBuffer() NN_NOEXCEPT
        : BaseType()
        , m_Buffer()
    {
    }

#define DRIVED_STORAGE_HOLDER_CONSTRUCTOR(n) \
    template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (U)> \
    explicit DerivedStorageHolderWithBuffer(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (U, x)) NN_NOEXCEPT \
        : BaseType(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (U, x)) \
        , m_Buffer() \
    { \
    }

    NN_UTIL_VARIADIC_DEFINE_MACROS(DRIVED_STORAGE_HOLDER_CONSTRUCTOR)
#undef DRIVED_STORAGE_HOLDER_CONSTRUCTOR

    using BaseType::Set;

    void Set(BufferHolder&& buffer) NN_NOEXCEPT
    {
        m_Buffer = std::move(buffer);
    }

private:
    BufferHolder m_Buffer;
};

// 外部鍵で復号を行う AesCtrStorage
class AesCtrStorageExternal : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AesCtrStorageExternal);

public:
    static const size_t BlockSize = crypto::AesEncryptor128::BlockSize;
    static const size_t KeySize = 16;
    static const size_t IvSize = 16;

public:
    AesCtrStorageExternal(
        fs::IStorage* pBaseStorage,
        const void* pEncryptedKey,
        size_t keySize,
        const void* pIv,
        size_t ivSize,
        DecryptAesCtrFunction pDecryptFunction,
        int keyIndex
    ) NN_NOEXCEPT;

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }
    virtual Result GetSize(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize
    ) NN_NOEXCEPT NN_OVERRIDE;

private:
    IStorage* const m_pBaseStorage;
    char m_Iv[IvSize];
    DecryptAesCtrFunction m_pDecryptFunction;
    int m_KeyIndex;
    char m_EncryptedKey[KeySize];
};

AesCtrStorageExternal::AesCtrStorageExternal(
    fs::IStorage* pBaseStorage,
    const void* pEncryptedKey,
    size_t keySize,
    const void* pIv,
    size_t ivSize,
    DecryptAesCtrFunction pDecryptFunction,
    int keyIndex
) NN_NOEXCEPT
    : m_pBaseStorage(pBaseStorage)
    , m_pDecryptFunction(pDecryptFunction)
    , m_KeyIndex(keyIndex)
{
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_EQUAL(keySize, KeySize);
    NN_SDK_REQUIRES_NOT_NULL(pIv);
    NN_SDK_REQUIRES_EQUAL(ivSize, IvSize);
    NN_UNUSED(keySize);
    NN_UNUSED(ivSize);

    std::memcpy(m_Iv, pIv, IvSize);
    std::memcpy(m_EncryptedKey, pEncryptedKey, keySize);
}

Result AesCtrStorageExternal::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // BlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(util::is_aligned(offset, BlockSize), fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(util::is_aligned(size, BlockSize), fs::ResultInvalidArgument());

    NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));

    ScopedThreadPriorityChanger changePriority(+1, ScopedThreadPriorityChanger::Mode::Relative);

    // TORIAEZU: sf の制限で InBuffer と OutBuffer に同じバッファを指定できないためワークバッファを通す
    PooledBuffer pooledBuffer;
    pooledBuffer.AllocateParticularlyLarge(size, BlockSize);

    // PooledBuffer の最低確保サイズは BlockSize より大きいので、確保されるサイズはそれ以上になるはず
    NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), BlockSize);

    char counter[IvSize];
    std::memcpy(counter, m_Iv, IvSize);
    AddCounter(counter, IvSize, offset / BlockSize);

    size_t restSize = size;
    int64_t currentOffset = 0;

    while( restSize > 0 )
    {
        size_t readSize = std::min(pooledBuffer.GetSize(), restSize);
        char* dstBuffer = static_cast<char*>(buffer) + currentOffset;

        m_pDecryptFunction(
            pooledBuffer.GetBuffer(),
            readSize,
            m_KeyIndex,
            m_EncryptedKey,
            KeySize,
            counter,
            IvSize,
            dstBuffer,
            readSize
        );

        // TODO: size <= pooledBuffer.GetSize() の場合は pooledBuffer に Read すれば memcpy を無くせる
        std::memcpy(dstBuffer, pooledBuffer.GetBuffer(), readSize);

        currentOffset += readSize;
        restSize -= readSize;

        if( restSize > 0 )
        {
            AddCounter(counter, IvSize, readSize / BlockSize);
        }
    }

    NN_RESULT_SUCCESS;
}

Result AesCtrStorageExternal::GetSize(int64_t* pOutValue) NN_NOEXCEPT
{
    return m_pBaseStorage->GetSize(pOutValue);
}

Result AesCtrStorageExternal::OperateRange(
                                    void* outBuffer,
                                    size_t outBufferSize,
                                    fs::OperationId operationId,
                                    int64_t offset,
                                    int64_t size,
                                    const void* inBuffer,
                                    size_t inBufferSize
                                ) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::QueryRange:
        {
            NN_RESULT_THROW_UNLESS(outBuffer != nullptr, fs::ResultNullptrArgument());
            NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(fs::QueryRangeInfo), fs::ResultInvalidSize());

            NN_RESULT_DO(m_pBaseStorage->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));

            fs::QueryRangeInfo info;
            info.Clear();
            info.aesCtrKeyTypeFlag = static_cast<int32_t>(
                m_KeyIndex >= 0
                ? fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes
                : fs::AesCtrKeyTypeFlag::ExternalKeyForHardwareAes);

            reinterpret_cast<fs::QueryRangeInfo*>(outBuffer)->Merge(info);
            NN_RESULT_SUCCESS;
        }
    default:
        NN_RESULT_DO(m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize));
        NN_RESULT_SUCCESS;
    }
}

NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorageExternal::BlockSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorageExternal::KeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesCtrStorageExternal::IvSize);

// 2 つのストレージを条件によって切り替えるストレージ
template< typename TFunc >
class SwitchStorage : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(SwitchStorage);

public:
    SwitchStorage(
        std::unique_ptr<IStorage>&& pStorageForTrue,
        std::unique_ptr<IStorage>&& pStorageForFalse,
        TFunc func
    ) NN_NOEXCEPT
        : m_pStorageForTrue(std::move(pStorageForTrue))
        , m_pStorageForFalse(std::move(pStorageForFalse))
        , m_pFunc(func)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFunc() ? m_pStorageForTrue : m_pStorageForFalse)->Read(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFunc() ? m_pStorageForTrue : m_pStorageForFalse)->Flush();
    }

    virtual Result GetSize(int64_t* pOutValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFunc() ? m_pStorageForTrue : m_pStorageForFalse)->GetSize(pOutValue);
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        switch( operationId )
        {
        case fs::OperationId::Invalidate:
            NN_RESULT_DO(m_pStorageForTrue->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));
            NN_RESULT_DO(m_pStorageForFalse->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));
            NN_RESULT_SUCCESS;

        case fs::OperationId::QueryRange:
            if( m_pFunc() )
            {
                NN_RESULT_DO(m_pStorageForTrue->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize));
            }
            else
            {
                NN_RESULT_DO(m_pStorageForFalse->OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offset,
                    size,
                    inBuffer,
                    inBufferSize));
            }
            NN_RESULT_SUCCESS;

        default:
            NN_RESULT_THROW(fs::ResultUnsupportedOperation());
        }
    }

private:
    std::unique_ptr<IStorage> m_pStorageForTrue;
    std::unique_ptr<IStorage> m_pStorageForFalse;
    TFunc                     m_pFunc;
};

const int AesCtrExTableCacheBlockSize = AesCtrCounterExtendedStorage::NodeSize;
const int AesCtrExTableCacheCount = 8;
const int IndirectTableCacheBlockSize = IndirectStorage::NodeSize;
const int IndirectTableCacheCount = 8;
const int IndirectDataCacheBlockSize = 32 * 1024;
const int IndirectDataCacheCount = 16;
const int SparseTableCacheBlockSize = SparseStorage::NodeSize;
const int SparseTableCacheCount = 4;

}

NcaFileSystemDriver::StorageOption::StorageOption(NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT
    : m_FsIndex(pHeaderReader->GetFsIndex())
    , m_pHeaderReader(pHeaderReader)
    , m_pDataStorage(nullptr)
    , m_DataStorageSize(0)
    , m_pAesCtrExTableStorage(nullptr)
    , m_pAesCtrExStorageRaw(nullptr)
    , m_pAesCtrExStorage(nullptr)
    , m_pIndirectStorage(nullptr)
    , m_pSparseStorage(nullptr)
#if defined(NN_BUILD_CONFIG_OS_WIN)
    , m_pExternalSparseStorage()
#endif
{
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);
}

NcaFileSystemDriver::StorageOption::StorageOption(NcaFsHeaderReader* pHeaderReader, int index) NN_NOEXCEPT
    : m_FsIndex(index)
    , m_pHeaderReader(pHeaderReader)
    , m_pDataStorage(nullptr)
    , m_DataStorageSize(0)
    , m_pAesCtrExTableStorage(nullptr)
    , m_pAesCtrExStorageRaw(nullptr)
    , m_pAesCtrExStorage(nullptr)
    , m_pIndirectStorage(nullptr)
    , m_pSparseStorage(nullptr)
#if defined(NN_BUILD_CONFIG_OS_WIN)
    , m_pExternalSparseStorage()
#endif
{
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);
    NN_SDK_REQUIRES_RANGE(index, 0, NcaHeader::FsCountMax);
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
NcaFileSystemDriver::SparseParam::SparseParam(
                                      utilTool::SparseStorageBuilder* pBuilder,
                                      NcaFsHeaderReader* pHeaderReader,
                                      bool isModeEmpty,
                                      int fsIndex, int64_t
                                      physicalOffset
                                  ) NN_NOEXCEPT
    : m_pBuilder(pBuilder)
    , m_pHeaderReader(pHeaderReader)
    , m_IsModeEmpty(isModeEmpty)
    , m_FsIndex(fsIndex)
    , m_PhysicalOffset(physicalOffset)
    , m_pSparseStorage()
    , m_IsStorageSparsified(false)
{
    NN_SDK_REQUIRES_NOT_NULL(pBuilder);
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);
    NN_SDK_REQUIRES_GREATER_EQUAL(physicalOffset, 0);
}

NcaFileSystemDriver::SparseParam::~SparseParam() NN_NOEXCEPT
{
}
#endif

NcaFileSystemDriver::NcaFileSystemDriver(
                         std::shared_ptr<NcaReader> pNcaReader,
                         MemoryResource* pAllocator,
                         IBufferManager* pBufferManager
                     ) NN_NOEXCEPT
    : m_pReader(pNcaReader)
    , m_pOriginalReader()
    , m_pAllocator(pAllocator)
    , m_pBufferManager(pBufferManager)
{
    NN_SDK_REQUIRES_NOT_NULL(pNcaReader);
}

NcaFileSystemDriver::NcaFileSystemDriver(
                         std::shared_ptr<NcaReader> pOriginalNcaReader,
                         std::shared_ptr<NcaReader> pCurrentNcaReader,
                         MemoryResource* pAllocator,
                         IBufferManager* pBufferManager
                     ) NN_NOEXCEPT
    : m_pReader(pCurrentNcaReader)
    , m_pOriginalReader(pOriginalNcaReader)
    , m_pAllocator(pAllocator)
    , m_pBufferManager(pBufferManager)
{
    NN_SDK_REQUIRES_NOT_NULL(pCurrentNcaReader);
}

// FS n のデータ領域へ直接アクセスする IStorage を取得します。
Result NcaFileSystemDriver::OpenRawStorage(std::shared_ptr<fs::IStorage>* pOutValue, int fsIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);
    NN_SDK_REQUIRES_NOT_NULL(m_pReader);

    const auto storageOffset = GetFsOffset(*m_pReader, fsIndex);
    const auto storageSize = GetFsEndOffset(*m_pReader, fsIndex) - storageOffset;
    NN_RESULT_THROW_UNLESS(0 < storageSize, fs::ResultInvalidNcaHeader());

    typedef DerivedStorageHolder<fs::SubStorage, 0> Storage;

    *pOutValue = AllocateShared<Storage>(m_pReader->GetBodyStorage(), storageOffset, storageSize, m_pReader);
    NN_RESULT_THROW_UNLESS(*pOutValue != nullptr, fs::ResultAllocationMemoryFailedAllocateShared());

    NN_RESULT_SUCCESS;
}

// FS n のデータ領域への IStorage を取得します。
Result NcaFileSystemDriver::OpenStorage(std::shared_ptr<fs::IStorage>* pOutValue, NcaFsHeaderReader* pOutHeaderReader, int fsIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderReader);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);

    StorageOption option(pOutHeaderReader, fsIndex);
    NN_RESULT_DO(OpenStorage(pOutValue, &option));

    NN_RESULT_SUCCESS;
}
Result NcaFileSystemDriver::OpenStorage(std::shared_ptr<fs::IStorage>* pOutValue, StorageOption* pOption) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pOption);
    NN_SDK_REQUIRES_NOT_NULL(m_pReader);

    const auto fsIndex = pOption->GetFsIndex();

    // fs ヘッダ初期化（ヘッダの検証を含む）
    NN_RESULT_THROW_UNLESS(m_pReader->HasFsInfo(fsIndex), fs::ResultPartitionNotFound());

    auto& headerReader = pOption->GetHeaderReader();
    NN_RESULT_DO(headerReader.Initialize(*m_pReader, fsIndex));

    std::unique_ptr<fs::IStorage> pStorage;
    {
        BaseStorage baseStorage;
        NN_RESULT_DO(CreateBaseStorage(&baseStorage, pOption));
        NN_RESULT_DO(CreateDecryptableStorage(&pStorage, pOption, &baseStorage));
    }
    NN_RESULT_DO(CreateIndirectStorage(&pStorage, pOption, std::move(pStorage)));
    NN_RESULT_DO(CreateVerificationStorage(&pStorage, std::move(pStorage), &headerReader));

    *pOutValue = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// 検証レイヤを除いた FS n のデータ領域への IStorage を取得します。
Result NcaFileSystemDriver::OpenDecryptableStorage(std::shared_ptr<fs::IStorage>* pOutValue, StorageOption* pOption, bool isIndirectStorageNeeded) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pOption);
    NN_SDK_REQUIRES_NOT_NULL(m_pReader);

    const auto fsIndex = pOption->GetFsIndex();
    NN_RESULT_THROW_UNLESS(m_pReader->HasFsInfo(fsIndex), fs::ResultPartitionNotFound());

    auto& headerReader = pOption->GetHeaderReader();

    // fs ヘッダ初期化（ヘッダの検証を含む）
    if( !headerReader.IsInitialized() )
    {
        NN_RESULT_DO(headerReader.Initialize(*m_pReader, fsIndex));
    }

    std::unique_ptr<fs::IStorage> pStorage;
    {
        BaseStorage baseStorage;
        NN_RESULT_DO(CreateBaseStorage(&baseStorage, pOption));
        NN_RESULT_DO(CreateDecryptableStorage(&pStorage, pOption, &baseStorage));
    }

    // データストレージのサイズを取得
    {
        const auto& patchInfo = headerReader.GetPatchInfo();
        int64_t dataStorageSize = 0;

        // AesCtrEx ストレージのテーブル部分は除く
        if( headerReader.GetEncryptionType() == NcaFsHeader::EncryptionType::AesCtrEx )
        {
            dataStorageSize = patchInfo.aesCtrExOffset;
        }
        else
        {
            switch( headerReader.GetHashType() )
            {
            case NcaFsHeader::HashType::HierarchicalSha256Hash:
                {
                    const auto& region = GetSha256DataRegion(headerReader.GetHashData());
                    dataStorageSize = region.offset + region.size;
                }
                break;

            case NcaFsHeader::HashType::HierarchicalIntegrityHash:
                {
                    const auto& info = GetIntegrityDataInfo(headerReader.GetHashData());
                    dataStorageSize = info.offset + info.size;
                }
                break;

            default:
                NN_RESULT_THROW(fs::ResultInvalidNcaFsHeaderHashType());
            }

            dataStorageSize = util::align_up(dataStorageSize, NcaHeader::XtsBlockSize);
        }

        pOption->SetDataStorage(pStorage.get(), dataStorageSize);
    }

    // IndirectStorage が必要なら作成
    if( isIndirectStorageNeeded )
    {
        NN_RESULT_DO(CreateIndirectStorage(&pStorage, pOption, std::move(pStorage)));
    }

    *pOutValue = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// 元になるストレージを作成します。
Result NcaFileSystemDriver::CreateBaseStorage(BaseStorage* pOutValue, StorageOption* pOption) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pOption);

    const auto fsIndex = pOption->GetFsIndex();
    const auto& headerReader = pOption->GetHeaderReader();

    const auto storageOffset = GetFsOffset(*m_pReader, fsIndex);
    const auto storageSize = GetFsEndOffset(*m_pReader, fsIndex) - storageOffset;
    NN_RESULT_THROW_UNLESS(0 < storageSize, fs::ResultInvalidNcaHeader());

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // 外部データからスパースストレージを作成
    if( pOption->HasExternalSparseStorage() )
    {
        auto pExternal = pOption->GetExternalSparseStorage();
        const auto& sparseInfo = pExternal->GetSparseInfo(fsIndex);

        if( 0 < sparseInfo.generation )
        {
            BucketTree::Header header;
            std::memcpy(&header, sparseInfo.bucket.header, sizeof(header));
            NN_RESULT_DO(header.Verify());

            std::unique_ptr<SparseStorage> pSparseStorage(new SparseStorage());
            NN_RESULT_THROW_UNLESS(pSparseStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            std::unique_ptr<fs::IStorage> pTableStorage;

            if( header.entryCount == 0 )
            {
                pSparseStorage->Initialize(storageSize);
            }
            else
            {
                // SparseStorage のテーブル用ストレージを作成
                pTableStorage = std::unique_ptr<fs::MemoryStorage>(
                    new fs::MemoryStorage(pExternal->GetSparseData(fsIndex), sparseInfo.bucket.size));
                NN_RESULT_THROW_UNLESS(pTableStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

                const auto nodeOffset = 0;
                const auto nodeSize = SparseStorage::QueryNodeStorageSize(header.entryCount);
                const auto entryOffset = nodeOffset + nodeSize;
                const auto entrySize = SparseStorage::QueryEntryStorageSize(header.entryCount);

                NN_RESULT_DO(pSparseStorage->Initialize(
                    m_pAllocator,
                    fs::SubStorage(pTableStorage.get(), nodeOffset, nodeSize),
                    fs::SubStorage(pTableStorage.get(), entryOffset, entrySize),
                    header.entryCount
                ));

                // NOTE: 外部データからでは SparseStorage::Read() が呼び出せないので設定しない
                //pSparseStorage->SetDataStorage();
            }

            typedef DerivedStorageHolder<fs::SubStorage, 2> Storage;
            std::unique_ptr<Storage> pStorage(new Storage(m_pReader->GetBodyStorage(), storageOffset, storageSize));
            NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            pOption->SetDataStorage(pStorage.get(), storageSize);
            pOption->SetSparseStorage(pSparseStorage.get());

            pStorage->Set(std::move(pSparseStorage), std::move(pTableStorage));

            pOutValue->SetStorage(std::move(pStorage));
        }
        else
        {
            pOutValue->SetStorage(m_pReader->GetBodyStorage(), storageOffset, storageSize);
        }
    }
    // 内部データからスパースストレージを作成
    else if( headerReader.ExistsSparseLayer() )
#else
    // 内部データからスパースストレージを作成
    if( headerReader.ExistsSparseLayer() )
#endif
    {
        const auto& sparseInfo = headerReader.GetSparseInfo();

        BucketTree::Header header;
        std::memcpy(&header, sparseInfo.bucket.header, sizeof(header));
        NN_RESULT_DO(header.Verify());

        typedef DerivedStorageHolder<SparseStorage, 2> Storage;
        std::unique_ptr<Storage> pStorage(new Storage(m_pReader));
        NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

        if( header.entryCount == 0 )
        {
            pStorage->Initialize(storageSize);
        }
        else
        {
            const auto pRawStorage = m_pReader->GetBodyStorage();
            const auto rawStorageOffset = sparseInfo.physicalOffset;
            const auto rawStorageSize = sparseInfo.GetPhysicalSize();

            // SparseStorage のテーブル用ストレージを作成
            std::unique_ptr<fs::IStorage> pDecryptableStorage;
            {
                BaseStorage baseStorage(pRawStorage, rawStorageOffset, rawStorageSize);
                baseStorage.SetStorageOffset(rawStorageOffset);
                baseStorage.SetAesCtrUpperIv(sparseInfo.MakeAesCtrUpperIv(headerReader.GetAesCtrUpperIv()));
                NN_RESULT_DO(CreateAesCtrStorage(&pDecryptableStorage, &baseStorage));
            }

            // 細かいアクセスが多いのでキャッシュを敷く
            std::unique_ptr<save::BufferedStorage> pTableStorage(new save::BufferedStorage());
            NN_RESULT_THROW_UNLESS(pTableStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            NN_RESULT_DO(pTableStorage->Initialize(
                fs::SubStorage(pDecryptableStorage.get(), 0, rawStorageSize),
                m_pBufferManager,
                SparseTableCacheBlockSize,
                SparseTableCacheCount
            ));

            const auto nodeOffset = sparseInfo.bucket.offset;
            const auto nodeSize = SparseStorage::QueryNodeStorageSize(header.entryCount);
            const auto entryOffset = nodeOffset + nodeSize;
            const auto entrySize = SparseStorage::QueryEntryStorageSize(header.entryCount);

            NN_RESULT_DO(pStorage->Initialize(
                m_pAllocator,
                fs::SubStorage(pTableStorage.get(), nodeOffset, nodeSize),
                fs::SubStorage(pTableStorage.get(), entryOffset, entrySize),
                header.entryCount
            ));

            pStorage->SetDataStorage(pRawStorage, rawStorageOffset, nodeOffset);
            pStorage->Set(std::move(pDecryptableStorage), std::move(pTableStorage));
        }

        pOption->SetSparseStorage(pStorage.get());

        pOutValue->SetStorage(std::move(pStorage));
    }
    // ベースはそのまま
    else
    {
        pOutValue->SetStorage(m_pReader->GetBodyStorage(), storageOffset, storageSize);
    }

    pOutValue->SetStorageOffset(storageOffset);
    pOutValue->SetAesCtrUpperIv(headerReader.GetAesCtrUpperIv());

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// 復号機能を備えたストレージを作成します。
Result NcaFileSystemDriver::CreateDecryptableStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, BaseStorage* pBaseStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pOption);

    const auto& headerReader = pOption->GetHeaderReader();

    switch( headerReader.GetEncryptionType() )
    {
    case NcaFsHeader::EncryptionType::None:
        *pOutStorage = pBaseStorage->MakeStorage();
        NN_RESULT_THROW_UNLESS(*pOutStorage != nullptr, fs::ResultAllocationMemoryFailedNew());
        break;

    case NcaFsHeader::EncryptionType::AesXts:
        NN_RESULT_DO(CreateAesXtsStorage(pOutStorage, pBaseStorage));
        break;

    case NcaFsHeader::EncryptionType::AesCtr:
        NN_RESULT_DO(CreateAesCtrStorage(pOutStorage, pBaseStorage));
        break;

    case NcaFsHeader::EncryptionType::AesCtrEx:
        NN_RESULT_DO(CreateAesCtrExStorage(pOutStorage, pOption, pBaseStorage));
        break;

    default:
        NN_RESULT_THROW(fs::ResultInvalidNcaFsHeaderEncryptionType());
    }

    NN_RESULT_SUCCESS;
}

// AesXts 暗号を解くストレージを作成します。
Result NcaFileSystemDriver::CreateAesXtsStorage(std::unique_ptr<fs::IStorage>* pOutStorage, BaseStorage* pBaseStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);

    char iv[AesXtsStorage::IvSize] = {};
    MakeAesXtsIv(iv, pBaseStorage->GetStorageOffset());

    // TODO: AesXtsStorage が SubStorage を受け取るようになったら修正する
    std::unique_ptr<fs::IStorage> pRawStorage = pBaseStorage->MakeStorage();
    NN_RESULT_THROW_UNLESS(pRawStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    std::unique_ptr<AesXtsStorage> pDecryptStorage(new AesXtsStorage(
        pRawStorage.get(),
        m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesXts1),
        m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesXts2),
        AesXtsStorage::KeySize,
        iv,
        AesXtsStorage::IvSize,
        NcaHeader::XtsBlockSize
    ));
    NN_RESULT_THROW_UNLESS(pDecryptStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    typedef AlignmentMatchingStorage<NcaHeader::XtsBlockSize, 1> AlignedStorage;
    typedef DerivedStorageHolder<AlignedStorage, 2> Storage;

    std::unique_ptr<Storage> pStorage(new Storage(pDecryptStorage.get(), m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    pStorage->Set(std::move(pRawStorage), std::move(pDecryptStorage));

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// AesCtr 暗号を解くストレージを作成します。
Result NcaFileSystemDriver::CreateAesCtrStorage(std::unique_ptr<fs::IStorage>* pOutStorage, BaseStorage* pBaseStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);

    char iv[AesCtrStorage::IvSize] = {};
    AesCtrStorage::MakeIv(iv, sizeof(iv), pBaseStorage->GetAesCtrUpperIv().value, pBaseStorage->GetStorageOffset());

    // TODO: AesCtrStorage(External) が SubStorage を受け取るようになったら修正する
    std::unique_ptr<fs::IStorage> pRawStorage = pBaseStorage->MakeStorage();
    NN_RESULT_THROW_UNLESS(pRawStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    std::unique_ptr<fs::IStorage> pDecryptStorage;

    // 外部鍵注入済みであればそれで復号する
    const bool hasExternalKey = m_pReader->HasExternalDecryptionKey();
    if( hasExternalKey )
    {
        // 外部鍵の場合、SW 鍵を持たないので外部鍵版 HW 一択
        pDecryptStorage.reset(new AesCtrStorageExternal(
            pRawStorage.get(),
            m_pReader->GetExternalDecryptionKey(),
            AesCtrStorageExternal::KeySize,
            iv,
            AesCtrStorageExternal::IvSize,
            m_pReader->GetExternalDecryptAesCtrFunctionForExternalKey(),
            -1 // unused
        ));
        NN_RESULT_THROW_UNLESS(pDecryptStorage != nullptr, fs::ResultAllocationMemoryFailedNew());
    }
    else
    {
        // 内部鍵
        const bool hasHwAesCtrKey = m_pReader->HasInternalDecryptionKeyForAesHw();

        // SW 鍵
        std::unique_ptr<fs::IStorage> pAesCtrSwStorage(new AesCtrStorage(
            pRawStorage.get(),
            m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesCtr),
            AesCtrStorage::KeySize,
            iv,
            AesCtrStorage::IvSize
        ));
        NN_RESULT_THROW_UNLESS(pAesCtrSwStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

        if( hasHwAesCtrKey && !m_pReader->IsSwAesPrioritized() )
        {
            // AesCtrHw 鍵が埋め込まれている場合、 SpeedEmulationMode に応じて SW/HW を切り替えられるように組む
            std::unique_ptr<fs::IStorage> pAesCtrHwStorage(new AesCtrStorageExternal(
                pRawStorage.get(),
                m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesCtrHw),
                AesCtrStorageExternal::KeySize,
                iv,
                AesCtrStorageExternal::IvSize,
                m_pReader->GetExternalDecryptAesCtrFunction(),
                GetKeyTypeValue(m_pReader->GetKeyIndex(), m_pReader->GetKeyGeneration())
            ));
            NN_RESULT_THROW_UNLESS(pAesCtrHwStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            pDecryptStorage.reset(new SwitchStorage<bool(*)()>(
                std::move(pAesCtrHwStorage),
                std::move(pAesCtrSwStorage),
                IsUsingHwAesCtrForSpeedEmulation
            ));
            NN_RESULT_THROW_UNLESS(pDecryptStorage != nullptr, fs::ResultAllocationMemoryFailedNew());
        }
        else
        {
            // AesCtrHw 鍵が埋め込まれていない場合、あるいは SW 優先（ゲームカード）の場合は SW 一択
            pDecryptStorage = std::move(pAesCtrSwStorage);
        }
    }

    typedef AlignmentMatchingStorage<NcaHeader::CtrBlockSize, 1> AlignedStorage;
    typedef DerivedStorageHolder<AlignedStorage, 2> Storage;

    std::unique_ptr<Storage> pStorage(new Storage(pDecryptStorage.get(), m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    pStorage->Set(std::move(pRawStorage), std::move(pDecryptStorage));

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// AesCtr(CounterExtended) 暗号を解くストレージを作成します。
Result NcaFileSystemDriver::CreateAesCtrExStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, BaseStorage* pBaseStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pOption);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);

    const auto& headerReader = pOption->GetHeaderReader();
    const auto& patchInfo = headerReader.GetPatchInfo();

    BucketTree::Header header;
    std::memcpy(&header, patchInfo.aesCtrExHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());

    NN_RESULT_THROW_UNLESS(0 < patchInfo.indirectSize, fs::ResultInvalidNcaPatchInfoIndirectSize());
    NN_RESULT_THROW_UNLESS(0 < patchInfo.aesCtrExSize, fs::ResultInvalidNcaPatchInfoAesCtrExSize());
    NN_RESULT_THROW_UNLESS(
        patchInfo.indirectOffset + patchInfo.indirectSize <= patchInfo.aesCtrExOffset,
        fs::ResultInvalidNcaPatchInfoAesCtrExOffset()
    );

    const auto baseStorageOffset = pBaseStorage->GetStorageOffset();
    const auto baseStorageSize = util::align_up(patchInfo.aesCtrExOffset + patchInfo.aesCtrExSize, NcaHeader::XtsBlockSize);
    fs::SubStorage baseStorage(pBaseStorage->GetSubStorage(0, baseStorageSize));

    // AesCtrEx のテーブルについて初期化
    std::unique_ptr<fs::IStorage> pTableStorage;
    {
        BaseStorage aesCtrBaseStorage(&baseStorage, patchInfo.aesCtrExOffset, patchInfo.aesCtrExSize);
        aesCtrBaseStorage.SetStorageOffset(baseStorageOffset + patchInfo.aesCtrExOffset);
        aesCtrBaseStorage.SetAesCtrUpperIv(headerReader.GetAesCtrUpperIv());
        NN_RESULT_DO(CreateAesCtrStorage(&pTableStorage, &aesCtrBaseStorage));
    }

    int64_t tableSize = 0;
    NN_RESULT_DO(pTableStorage->GetSize(&tableSize));

    std::unique_ptr<save::BufferedStorage> pBufferedStorage(new save::BufferedStorage());
    NN_RESULT_THROW_UNLESS(pBufferedStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    NN_RESULT_DO(pBufferedStorage->Initialize(
        fs::SubStorage(pTableStorage.get(), 0, tableSize),
        m_pBufferManager,
        AesCtrExTableCacheBlockSize,
        AesCtrExTableCacheCount
    ));

    typedef AlignmentMatchingStorage<NcaHeader::CtrBlockSize, 1> AlignedStorage;
    std::unique_ptr<AlignedStorage> pAlignedStorage(new AlignedStorage(pBufferedStorage.get()));
    NN_RESULT_THROW_UNLESS(pAlignedStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    const auto entryCount = header.entryCount;
    const int64_t dataOffset = 0;
    const int64_t dataSize = patchInfo.aesCtrExOffset;
    const int64_t nodeOffset = 0;
    const int64_t nodeSize = AesCtrCounterExtendedStorage::QueryNodeStorageSize(entryCount);
    const int64_t entryOffset = nodeOffset + nodeSize;
    const int64_t entrySize = AesCtrCounterExtendedStorage::QueryEntryStorageSize(entryCount);
    fs::SubStorage dataStorage(&baseStorage, dataOffset, dataSize);
    fs::SubStorage nodeStorage(pAlignedStorage.get(), nodeOffset, nodeSize);
    fs::SubStorage entryStorage(pAlignedStorage.get(), entryOffset, entrySize);
    const auto secureValue = headerReader.GetAesCtrUpperIv().part.secureValue;

    typedef AesCtrCounterExtendedStorage SingleStorage;
    typedef SwitchStorage<bool(*)()> SwitchStorage;
    std::unique_ptr<fs::IStorage> pAesCtrExStorage;

    // 外部鍵注入済みであればそれで復号する
    const bool hasExternalKey = m_pReader->HasExternalDecryptionKey();
    if( hasExternalKey )
    {
        std::unique_ptr<AesCtrCounterExtendedStorage::IDecryptor> pDecryptor;
        NN_RESULT_DO(AesCtrCounterExtendedStorage::CreateExternalDecryptor(
            &pDecryptor,
            m_pReader->GetExternalDecryptAesCtrFunctionForExternalKey(),
            -1 // unused
        ));

        // 外部鍵の場合、SW 鍵を持たないので外部鍵版 HW 一択
        std::unique_ptr<SingleStorage> pSingleStorage(new SingleStorage());
        NN_RESULT_THROW_UNLESS(pSingleStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

        NN_RESULT_DO(pSingleStorage->Initialize(
            m_pAllocator,
            m_pReader->GetExternalDecryptionKey(),
            AesCtrStorage::KeySize,
            secureValue,
            baseStorageOffset,
            dataStorage,
            nodeStorage,
            entryStorage,
            entryCount,
            std::move(pDecryptor)
        ));

        pOption->SetAesCtrExStorageRaw(pSingleStorage.get());

        pAesCtrExStorage = std::move(pSingleStorage);
    }
    else
    {
        // 内部鍵
        const bool hasHwAesCtrKey = m_pReader->HasInternalDecryptionKeyForAesHw();

        // SW 鍵
        std::unique_ptr<AesCtrCounterExtendedStorage::IDecryptor> pSwDecryptor;
        NN_RESULT_DO(AesCtrCounterExtendedStorage::CreateSoftwareDecryptor(&pSwDecryptor));

        if( hasHwAesCtrKey && !m_pReader->IsSwAesPrioritized() )
        {
            std::unique_ptr<AesCtrCounterExtendedStorage> pSwStorage(new AesCtrCounterExtendedStorage());
            NN_RESULT_THROW_UNLESS(pSwStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            NN_RESULT_DO(pSwStorage->Initialize(
                m_pAllocator,
                m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesCtr),
                AesCtrStorage::KeySize,
                secureValue,
                baseStorageOffset,
                dataStorage,
                nodeStorage,
                entryStorage,
                entryCount,
                std::move(pSwDecryptor)
            ));

            pOption->SetAesCtrExStorageRaw(pSwStorage.get());

            // HW 鍵
            std::unique_ptr<AesCtrCounterExtendedStorage::IDecryptor> pHwDecryptor;
            NN_RESULT_DO(AesCtrCounterExtendedStorage::CreateExternalDecryptor(
                &pHwDecryptor,
                m_pReader->GetExternalDecryptAesCtrFunction(),
                GetKeyTypeValue(m_pReader->GetKeyIndex(), m_pReader->GetKeyGeneration())
            ));

            // AesCtrHw 鍵が埋め込まれている場合、 SpeedEmulationMode に応じて SW/HW を切り替えられるように組む
            std::unique_ptr<AesCtrCounterExtendedStorage> pHwStorage(new AesCtrCounterExtendedStorage());
            NN_RESULT_THROW_UNLESS(pHwStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            NN_RESULT_DO(pHwStorage->Initialize(
                m_pAllocator,
                m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesCtrHw),
                AesCtrStorage::KeySize,
                secureValue,
                baseStorageOffset,
                dataStorage,
                nodeStorage,
                entryStorage,
                entryCount,
                std::move(pHwDecryptor)
            ));

            std::unique_ptr<SwitchStorage> pSwitchStorage(new SwitchStorage(
                std::move(pHwStorage),
                std::move(pSwStorage),
                IsUsingHwAesCtrForSpeedEmulation
            ));
            NN_RESULT_THROW_UNLESS(pSwitchStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            pAesCtrExStorage = std::move(pSwitchStorage);
        }
        else
        {
            // AesCtrHw 鍵が埋め込まれていない場合、あるいは SW 優先（ゲームカード）の場合は SW 一択
            std::unique_ptr<SingleStorage> pSingleStorage(new SingleStorage());
            NN_RESULT_THROW_UNLESS(pSingleStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

            NN_RESULT_DO(pSingleStorage->Initialize(
                m_pAllocator,
                m_pReader->GetDecryptionKey(NcaHeader::DecryptionKey_AesCtr),
                AesCtrStorage::KeySize,
                secureValue,
                baseStorageOffset,
                dataStorage,
                nodeStorage,
                entryStorage,
                entryCount,
                std::move(pSwDecryptor)
            ));

            pOption->SetAesCtrExStorageRaw(pSingleStorage.get());

            pAesCtrExStorage = std::move(pSingleStorage);
        }
    }

    typedef DerivedStorageHolder<AlignedStorage, 5> Storage;
    std::unique_ptr<Storage> pStorage(new Storage(pAesCtrExStorage.get(), m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    pOption->SetAesCtrExTableStorage(pTableStorage.get());
    pOption->SetAesCtrExStorage(pStorage.get());

    pStorage->Set(pBaseStorage->GetStorage(), std::move(pTableStorage), std::move(pBufferedStorage), std::move(pAlignedStorage), std::move(pAesCtrExStorage));

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// IndirectStorage を作成します。
Result NcaFileSystemDriver::CreateIndirectStorage(std::unique_ptr<fs::IStorage>* pOutStorage, StorageOption* pOption, std::unique_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pOption);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);

    const auto& headerReader = pOption->GetHeaderReader();
    const auto& patchInfo = headerReader.GetPatchInfo();

    // パッチなし
    if( !patchInfo.HasIndirectTable() )
    {
        *pOutStorage = std::move(pBaseStorage);
        NN_RESULT_SUCCESS;
    }

    BucketTree::Header header;
    std::memcpy(&header, patchInfo.indirectHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());

    const auto nodeSize = IndirectStorage::QueryNodeStorageSize(header.entryCount);
    const auto entrySize = IndirectStorage::QueryEntryStorageSize(header.entryCount);
    NN_RESULT_THROW_UNLESS(nodeSize + entrySize <= patchInfo.indirectSize, fs::ResultInvalidIndirectStorageSize());

    // 元データのストレージ取得
    std::unique_ptr<fs::IStorage> pOriginalStorage;
    {
        const int fsIndex = headerReader.GetFsIndex();

        if( m_pOriginalReader != nullptr && m_pOriginalReader->HasFsInfo(fsIndex) )
        {
            NcaFsHeaderReader originalHeaderReader;
            NN_RESULT_DO(originalHeaderReader.Initialize(*m_pOriginalReader, fsIndex));

            NcaFileSystemDriver original(m_pOriginalReader, m_pAllocator, m_pBufferManager);
            StorageOption option(&originalHeaderReader, fsIndex);
#if defined(NN_BUILD_CONFIG_OS_WIN)
            option.SetExternalSparseStorage(pOption->GetOriginalSparseStorage());
#endif
            BaseStorage baseStorage;
            NN_RESULT_DO(original.CreateBaseStorage(&baseStorage, &option));
            NN_RESULT_DO(original.CreateDecryptableStorage(&pOriginalStorage, &option, &baseStorage));
        }
        else
        {
            // 元データに存在しない Fs なので空のストレージを設定
            pOriginalStorage.reset(new fs::MemoryStorage(nullptr, 0));
            NN_RESULT_THROW_UNLESS(pOriginalStorage != nullptr, fs::ResultAllocationMemoryFailedNew());
        }
    }

    int64_t oldDataSize = 0;
    NN_RESULT_DO(pOriginalStorage->GetSize(&oldDataSize));

    const auto newDataSize = patchInfo.indirectOffset;
    NN_SDK_ASSERT_ALIGNED(newDataSize, NcaHeader::XtsBlockSize);

    // 細かいアクセスが多いのでキャッシュを敷く
    std::unique_ptr<save::BufferedStorage> pIndirectTableStorage(new save::BufferedStorage());
    NN_RESULT_THROW_UNLESS(pIndirectTableStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    NN_RESULT_DO(pIndirectTableStorage->Initialize(
        fs::SubStorage(pBaseStorage.get(), newDataSize, nodeSize + entrySize),
        m_pBufferManager,
        IndirectTableCacheBlockSize,
        IndirectTableCacheCount
    ));

    // 断片化対策でデータの方にキャッシュを敷く
    std::unique_ptr<save::BufferedStorage> pIndirectDataStorage(new save::BufferedStorage());
    NN_RESULT_THROW_UNLESS(pIndirectDataStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    NN_RESULT_DO(pIndirectDataStorage->Initialize(
        fs::SubStorage(pBaseStorage.get(), 0, newDataSize),
        m_pBufferManager,
        IndirectDataCacheBlockSize,
        IndirectDataCacheCount
    ));

    typedef DerivedStorageHolder<IndirectStorage, 4> Storage;

    std::unique_ptr<Storage> pStorage(new Storage(m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    NN_RESULT_DO(pStorage->Initialize(
        m_pAllocator,
        fs::SubStorage(pIndirectTableStorage.get(), 0, nodeSize),
        fs::SubStorage(pIndirectTableStorage.get(), nodeSize, entrySize),
        header.entryCount
    ));

    pStorage->SetStorage(0, pOriginalStorage.get(), 0, oldDataSize);
    pStorage->SetStorage(1, pIndirectDataStorage.get(), 0, newDataSize);

    pStorage->Set(
        std::move(pBaseStorage),
        std::move(pOriginalStorage),
        std::move(pIndirectTableStorage),
        std::move(pIndirectDataStorage)
    );

    pOption->SetIndirectStorage(pStorage.get());

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// 検証機能を備えたストレージを作成します。
Result NcaFileSystemDriver::CreateVerificationStorage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);

    // 完全性検証の選択
    switch( pHeaderReader->GetHashType() )
    {
    case NcaFsHeader::HashType::HierarchicalSha256Hash:
        NN_RESULT_DO(CreateSha256Storage(pOutStorage, std::move(pBaseStorage), pHeaderReader));
        break;

    case NcaFsHeader::HashType::HierarchicalIntegrityHash:
        NN_RESULT_DO(CreateIntegrityVerificationStorage(pOutStorage, std::move(pBaseStorage), pHeaderReader));
        break;

    default:
        NN_RESULT_THROW(fs::ResultInvalidNcaFsHeaderHashType());
    }

    NN_RESULT_SUCCESS;
}

// HierarchicalSha256Storage を作成します。
Result NcaFileSystemDriver::CreateSha256Storage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);

    typedef HierarchicalSha256Storage VerificationStorage;
    typedef ReadOnlyBlockCacheStorage CacheStorage;
    typedef AlignmentMatchingStoragePooledBuffer<1> AlignedStorage;
    typedef DerivedStorageHolderWithBuffer<AlignedStorage, 4> Storage;

    auto& hashData = pHeaderReader->GetHashData().hierarchicalSha256Data;

    // TODO: 2 段固定
    NN_RESULT_THROW_UNLESS(
        util::ispow2(hashData.hashBlockSize),
        fs::ResultInvalidHierarchicalSha256BlockSize()
    );
    NN_RESULT_THROW_UNLESS(
        hashData.hashLayerCount == HierarchicalSha256Storage::LayerCount - 1,
        fs::ResultInvalidHierarchicalSha256LayerCount()
    );

    const auto& hashRegion = hashData.hashLayerRegion[0];
    const auto& dataRegion = hashData.hashLayerRegion[1];

    static const int CacheBlockCount = 2;

    // ハッシュ領域のバッファ + ブロックキャッシュ用バッファ
    const auto hashBufferSize = static_cast<size_t>(hashRegion.size);
    const auto cacheBufferSize = CacheBlockCount * hashData.hashBlockSize;
    const auto totalBufferSize = hashBufferSize + cacheBufferSize;

    BufferHolder buffer(m_pAllocator, totalBufferSize);
    NN_RESULT_THROW_UNLESS(buffer.IsValid(), fs::ResultAllocationMemoryFailedInNcaFileSystemDriverI());

    // TODO: HierarchicalSha256Storage::Initialize() が SubStorage を受け取れば new する必要はないのだが...
    std::unique_ptr<fs::SubStorage> pDataStorage(new fs::SubStorage(pBaseStorage.get(), dataRegion.offset, dataRegion.size));
    NN_RESULT_THROW_UNLESS(pDataStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    std::unique_ptr<VerificationStorage> pVerificationStorage(new VerificationStorage());
    NN_RESULT_THROW_UNLESS(pVerificationStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    // TODO: データ領域以外は Initialize() 以降参照されないのでスタックに配置。変更されたら修正
    fs::MemoryStorage masterHashStorage(&hashData.fsDataMasterHash, sizeof(Hash));
    fs::SubStorage layerHashStorage(pBaseStorage.get(), hashRegion.offset, hashRegion.size);
    fs::IStorage* pStorages[VerificationStorage::LayerCount] =
    {
        &masterHashStorage,
        &layerHashStorage,
        pDataStorage.get(),
    };

    NN_RESULT_DO(pVerificationStorage->Initialize(
        pStorages,
        VerificationStorage::LayerCount,
        hashData.hashBlockSize,
        buffer.Get(),
        hashBufferSize
    ));

    std::unique_ptr<CacheStorage> pCacheStorage(new CacheStorage(
        pVerificationStorage.get(),
        hashData.hashBlockSize,
        buffer.Get() + hashBufferSize,
        cacheBufferSize,
        CacheBlockCount
    ));
    NN_RESULT_THROW_UNLESS(pCacheStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    std::unique_ptr<Storage> pStorage(new Storage(pCacheStorage.get(), hashData.hashBlockSize, m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    pStorage->Set(
        std::move(pBaseStorage),
        std::move(pDataStorage),
        std::move(pVerificationStorage),
        std::move(pCacheStorage)
    );
    pStorage->Set(std::move(buffer));

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

// HierarchicalIntegrityVerificationStorage を作成します。
Result NcaFileSystemDriver::CreateIntegrityVerificationStorage(std::unique_ptr<fs::IStorage>* pOutStorage, std::unique_ptr<fs::IStorage> pBaseStorage, NcaFsHeaderReader* pHeaderReader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStorage);
    NN_SDK_REQUIRES_NOT_NULL(pBaseStorage);
    NN_SDK_REQUIRES_NOT_NULL(pHeaderReader);

    typedef save::HierarchicalIntegrityVerificationStorage VerificationStorage;
    typedef VerificationStorage::HierarchicalStorageInformation StorageInfo;
    typedef DerivedStorageHolder<IntegrityRomFsStorage, 1> Storage;

    const auto& hashData = pHeaderReader->GetHashData().integrityMetaInfo;

    save::HierarchicalIntegrityVerificationInformation infoLevelHash;
    std::memcpy(&infoLevelHash, &hashData.infoLevelHash, sizeof(infoLevelHash));

    // 階層数のチェック
    NN_RESULT_THROW_UNLESS(
        save::IntegrityMinLayerCount <= infoLevelHash.maxLayers && infoLevelHash.maxLayers <= save::IntegrityMaxLayerCount,
        fs::ResultInvalidHierarchicalIntegrityVerificationLayerCount()
    );

    StorageInfo storageInfo;

    // 階層ハッシュ設定
    const auto layerCount = static_cast<int>(infoLevelHash.maxLayers - 2);
    for( int layer = 0; layer < layerCount; ++layer )
    {
        const auto& layerInfo = infoLevelHash.info[layer];
        storageInfo[1 + layer] = fs::SubStorage(pBaseStorage.get(), layerInfo.offset, layerInfo.size);
    }

    // 本体データ設定
    const auto& layerInfo = infoLevelHash.info[infoLevelHash.maxLayers - 2];
    storageInfo.SetDataStorage(fs::SubStorage(pBaseStorage.get(), layerInfo.offset, layerInfo.size));

    std::unique_ptr<Storage> pStorage(new Storage(m_pReader));
    NN_RESULT_THROW_UNLESS(pStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    // 階層署名ストレージ初期化
    NN_RESULT_DO(pStorage->Initialize(infoLevelHash, hashData.masterHash, storageInfo, m_pBufferManager));

    pStorage->Set(std::move(pBaseStorage));

    *pOutStorage = std::move(pStorage);

    NN_RESULT_SUCCESS;
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
// SparseStorage を埋め込みます。
Result NcaFileSystemDriver::SparsifyStorage(SparseParam* pParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParam);

    NN_RESULT_THROW_UNLESS(m_pOriginalReader != nullptr, fs::ResultInvalidArgument());

    const auto fsIndex = pParam->GetFsIndex();
    NN_RESULT_THROW_UNLESS(m_pOriginalReader->HasFsInfo(fsIndex), fs::ResultInvalidArgument());

    if( pParam->IsModeEmpty() )
    {
        NN_RESULT_DO(SparsifyEmptyStorage(pParam));
    }
    else
    {
        NN_RESULT_THROW_UNLESS(m_pReader->HasFsInfo(fsIndex), fs::ResultInvalidArgument());

        NN_RESULT_DO(SparsifyBasicStorage(pParam));
    }

    NN_RESULT_SUCCESS;
}

// 基本的な SparseStorage を埋め込みます。
Result NcaFileSystemDriver::SparsifyBasicStorage(SparseParam* pParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParam);

    pParam->SetStorageSparsified(true);

    const int fsIndex = pParam->GetFsIndex();

    NcaFsHeaderReader patchHeaderReader;
    NN_RESULT_DO(patchHeaderReader.Initialize(*m_pReader, fsIndex));

    const auto& patchInfo = patchHeaderReader.GetPatchInfo();
    NN_RESULT_THROW_UNLESS(patchInfo.HasIndirectTable(), fs::ResultInvalidNcaFsHeader());

    auto& originalHeaderReader = pParam->GetHeaderReader();
    NN_RESULT_DO(originalHeaderReader.Initialize(*m_pOriginalReader, fsIndex));

    std::unique_ptr<fs::IStorage> pDataStorage;

    // 外部データのスパース情報をコピー
    if( pParam->HasSparseStorage() )
    {
        std::memcpy(&originalHeaderReader.GetSparseInfo(), &pParam->GetSparseStorage()->GetSparseInfo(fsIndex), sizeof(NcaSparseInfo));
    }

    auto& storageBuilder = pParam->GetBuilder();

    // SparseStorage を再配置する
    if( originalHeaderReader.ExistsSparseLayer() )
    {
        SparseStorage* pSparseStorage = nullptr;
        BaseStorage baseStorage;
        {
            StorageOption option(&originalHeaderReader, fsIndex);
            {
                NcaFileSystemDriver original(m_pOriginalReader, m_pAllocator, m_pBufferManager);
                option.SetExternalSparseStorage(pParam->GetSparseStorage());
                NN_RESULT_DO(original.CreateBaseStorage(&baseStorage, &option));
            }

            pSparseStorage = option.GetSparseStorage();
            NN_RESULT_THROW_UNLESS(pSparseStorage != nullptr, fs::ResultInvalidNcaHeader());
        }
        NN_RESULT_DO(storageBuilder.Import(pSparseStorage));

        pDataStorage = baseStorage.MakeStorage();
        NN_RESULT_THROW_UNLESS(pDataStorage != nullptr, fs::ResultAllocationMemoryFailedNew());
    }

    BucketTree::Header header;
    std::memcpy(&header, patchInfo.indirectHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());
    NN_RESULT_THROW_UNLESS(0 < header.entryCount, fs::ResultInvalidNcaFsHeader());

    const auto storageOffset = GetFsOffset(*m_pOriginalReader, fsIndex);
    const auto storageSize = GetFsEndOffset(*m_pOriginalReader, fsIndex) - storageOffset;
    NN_RESULT_THROW_UNLESS(0 < storageSize, fs::ResultInvalidNcaHeader());
    {
        // パッチ側 nca から IndirectTable 解析用のストレージを作成する
        std::unique_ptr<fs::IStorage> pBaseStorage;
        {
            StorageOption option(&patchHeaderReader, fsIndex);
            BaseStorage baseStorage;
            NN_RESULT_DO(CreateBaseStorage(&baseStorage, &option));
            NN_RESULT_DO(CreateDecryptableStorage(&pBaseStorage, &option, &baseStorage));
        }

        // 細かいアクセスが多いのでキャッシュを敷く
        std::unique_ptr<save::BufferedStorage> pTableStorage(new save::BufferedStorage());
        NN_RESULT_THROW_UNLESS(pTableStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

        static const auto CacheBlockSize = IndirectStorage::NodeSize;
        static const auto CacheCount = 4;
        const auto nodeSize = IndirectStorage::QueryNodeStorageSize(header.entryCount);
        const auto entrySize = IndirectStorage::QueryEntryStorageSize(header.entryCount);

        NN_RESULT_DO(pTableStorage->Initialize(
            fs::SubStorage(pBaseStorage.get(), patchInfo.indirectOffset, nodeSize + entrySize),
            m_pBufferManager,
            CacheBlockSize,
            CacheCount
        ));

        // スパース化
        if( pDataStorage == nullptr )
        {
            NN_RESULT_DO(storageBuilder.Build(
                m_pAllocator,
                fs::SubStorage(pTableStorage.get(), 0, nodeSize),
                fs::SubStorage(pTableStorage.get(), nodeSize, entrySize),
                header.entryCount,
                storageSize
            ));
        }
        // 再スパース化
        else
        {
            NN_RESULT_DO(storageBuilder.Rebuild(
                m_pAllocator,
                fs::SubStorage(pTableStorage.get(), 0, nodeSize),
                fs::SubStorage(pTableStorage.get(), nodeSize, entrySize),
                header.entryCount,
                storageSize
            ));
        }
    }

    const auto dataSize = util::align_up(storageBuilder.QueryDataStorageSize(), 1 << NcaHeader::Log2SectorSize);
    const auto nodeSize = storageBuilder.QueryTableNodeStorageSize();
    const auto entrySize = storageBuilder.QueryTableEntryStorageSize();
    auto& sparseInfo = originalHeaderReader.GetSparseInfo();

    if( pDataStorage == nullptr )
    {
        // SparseStorage にしても小さくならないので埋め込まない
        if( storageSize <= dataSize + nodeSize + entrySize )
        {
            pParam->SetStorageSparsified(false);
            NN_RESULT_SUCCESS;
        }

        // ストレージをそのままコピーするため生のストレージを設定
        storageBuilder.SetDataStorage(m_pOriginalReader->GetBodyStorage(), storageOffset, storageSize);
    }
    else
    {
        NN_RESULT_THROW_UNLESS(dataSize <= sparseInfo.bucket.offset, fs::ResultInvalidNcaHeader());

        storageBuilder.SetDataStorage(std::move(pDataStorage), 0, storageSize);
    }

    // SparseStorage の情報を設定
    storageBuilder.WriteHeader(sparseInfo.bucket.header, NcaBucketInfo::HeaderSize);
    sparseInfo.bucket.offset = dataSize;
    sparseInfo.bucket.size = nodeSize + entrySize;
    sparseInfo.physicalOffset = pParam->GetPhysicalOffset();
    ++sparseInfo.generation;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

// 元データをすべて参照しない SparseStorage を埋め込みます。
Result NcaFileSystemDriver::SparsifyEmptyStorage(SparseParam* pParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParam);

    const auto fsIndex = pParam->GetFsIndex();
    auto& originalHeaderReader = pParam->GetHeaderReader();
    NN_RESULT_DO(originalHeaderReader.Initialize(*m_pOriginalReader, fsIndex));

    // 外部データのスパース情報をコピー
    if( pParam->HasSparseStorage() )
    {
        std::memcpy(&originalHeaderReader.GetSparseInfo(), &pParam->GetSparseStorage()->GetSparseInfo(fsIndex), sizeof(NcaSparseInfo));
    }

    auto& storageBuilder = pParam->GetBuilder();
    storageBuilder.Build(m_pOriginalReader->GetFsSize(fsIndex));

    // SparseStorage の情報を設定
    auto& sparseInfo = originalHeaderReader.GetSparseInfo();
    storageBuilder.WriteHeader(sparseInfo.bucket.header, NcaBucketInfo::HeaderSize);
    sparseInfo.bucket.offset = 0;
    sparseInfo.bucket.size = 0;
    sparseInfo.physicalOffset = pParam->GetPhysicalOffset();
    ++sparseInfo.generation; // NOTE: データの更新はないが世代番号は上げておく

    NN_RESULT_SUCCESS;
}
#endif

Result NcaFileSystemDriver::SetupFsHeaderReader(NcaFsHeaderReader* pOutHeaderReader, const NcaReader& ncaReader, int fsIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHeaderReader);
    NN_SDK_REQUIRES_RANGE(fsIndex, 0, NcaHeader::FsCountMax);

    NN_RESULT_THROW_UNLESS(
        ncaReader.GetSignature() == NcaHeader::Signature, fs::ResultUnsupportedVersion());

    NN_RESULT_THROW_UNLESS(ncaReader.HasFsInfo(fsIndex), fs::ResultPartitionNotFound());

    // fs ヘッダ初期化（ヘッダの検証を含む）
    NN_RESULT_DO(pOutHeaderReader->Initialize(ncaReader, fsIndex));

    NN_RESULT_SUCCESS;
}

}}
