﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ビルドスイッチのため、例外的に先頭でインクルードする
#include <nn/fs/detail/fs_ResultHandlingUtilitySuppressRecordingEventOnUnsupportedPlatforms.h>

#include <nn/fssystem/fs_BucketTreeImpl.h>
#include <nn/fssystem/fs_IndirectStorageImpl.h>
#include <nn/fs/fs_QueryRange.h>

namespace nn { namespace fssystem {

NN_DEFINE_STATIC_CONSTANT(const int IndirectStorage::StorageCount);
NN_DEFINE_STATIC_CONSTANT(const size_t IndirectStorage::NodeSize);
NN_DEFINE_STATIC_CONSTANT(const int IndirectStorage::ContinuousReadingEntry::FragmentSizeMax);

// テンプレート関数の実体化
template Result
BucketTree::Visitor::ScanContinuousReading<IndirectStorage::ContinuousReadingEntry>(
    BucketTree::ContinuousReadingInfo* pOutInfo, int64_t offset, size_t size
) const NN_NOEXCEPT;

/**
 * @brief   コンストラクタです。
 */
IndirectStorage::IndirectStorage() NN_NOEXCEPT
    : m_Table()
    , m_DataStorage()
{
}

/**
 * @brief   初期化します。（テスト向け）
 *
 * @param[in]   pAllocator      アロケータのポインタ
 * @param[in]   tableStorage    論物変換テーブルを格納したストレージ
 *
 * @retval  ResultSuccess                   正常に処理が終了しました。
 * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
 * @retval  上記以外                        ストレージの読み込みに失敗しました。
 *
 * @pre
 *      - pAllocator != nullptr
 *      - 未初期化
 */
Result IndirectStorage::Initialize(
                            IAllocator* pAllocator,
                            fs::SubStorage tableStorage
                        ) NN_NOEXCEPT
{
    BucketTree::Header header;
    NN_RESULT_DO(tableStorage.Read(0, &header, sizeof(header)));
    NN_RESULT_DO(header.Verify());

    const auto nodeStorageSize =
        BucketTree::QueryNodeStorageSize(NodeSize, sizeof(Entry), header.entryCount);
    const auto entryStorageSize =
        BucketTree::QueryEntryStorageSize(NodeSize, sizeof(Entry), header.entryCount);

    const auto nodeStorageOffset = BucketTree::QueryHeaderStorageSize();
    const auto entryStorageOffset = nodeStorageOffset + nodeStorageSize;

    // 引数の事前検証はこっちに任せる
    return Initialize(
               pAllocator,
               fs::SubStorage(&tableStorage, nodeStorageOffset, nodeStorageSize),
               fs::SubStorage(&tableStorage, entryStorageOffset, entryStorageSize),
               header.entryCount
           );
}

/**
 * @brief   終了処理をします。
 */
void IndirectStorage::Finalize() NN_NOEXCEPT
{
    if( IsInitialized() )
    {
        m_Table.Finalize();

        for( int i = 0; i < StorageCount; ++i )
        {
            m_DataStorage[i] = fs::SubStorage();
        }
    }
}

/**
 * @brief   データを読み込みます。
 *
 * @param[in]   offset  読み込むオフセット
 * @param[out]  buffer  読み込んだデータを格納するバッファ
 * @param[in]   size    読み込むサイズ
 *
 * @return  関数の実行結果を返します。
 *
 *      - 0 <= offset
 *      - size == 0 || buffer != nullptr
 *      - 初期化済み
 */
Result IndirectStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
    NN_SDK_REQUIRES(IsInitialized());

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    NN_RESULT_DO(OperatePerEntry<true>(
        offset,
        size,
        [=](fs::SubStorage* pStorage,
            int64_t dataOffset,
            int64_t processOffset,
            int64_t processSize) NN_NOEXCEPT -> Result
        {
            NN_RESULT_DO(pStorage->Read(
                dataOffset,
                reinterpret_cast<char*>(buffer) + (processOffset - offset),
                static_cast<size_t>(processSize)));
            NN_RESULT_SUCCESS;
        }
    ));

    NN_RESULT_SUCCESS;
}

/**
 * @brief   指定した範囲に対応するエントリリストを取得します。（AuthoringTool 向け）
 *
 * @param[out]  outEntries     エントリリストを格納するポインタ
 * @param[out]  outEntryCount  エントリリストの長さ
 * @param[in]   entryCount     取得するエントリリストの長さ
 * @param[in]   offset         範囲のオフセット
 * @param[in]   size           範囲のサイズ
 *
 * @return  関数の実行結果を返します。
 *
 * @pre
 *      - 0 <= offset
 *      - outEntryCount != nullptr
 *      - entryCount == 0 || outEntries != nullptr
 *      - outEntries が指す格納領域のサイズが sizeof(Entry) * entryCount 以上
 *      - 初期化済み
 */
Result IndirectStorage::GetEntryList(Entry* outEntries, int* outEntryCount, int entryCount, int64_t offset, int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
    NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    NN_SDK_REQUIRES(IsInitialized());
    NN_RESULT_THROW_UNLESS(outEntryCount != nullptr, fs::ResultNullptrArgument());

    if( size == 0 )
    {
        *outEntryCount = 0;
        NN_RESULT_SUCCESS;
    }

    if (entryCount != 0)
    {
        NN_RESULT_THROW_UNLESS(outEntries != nullptr, fs::ResultNullptrArgument());
    }

    NN_RESULT_THROW_UNLESS(m_Table.IsInclude(offset, size), fs::ResultOutOfRange());

    // Find() で offset < 0 を弾くので、0 <= offset は保障される
    BucketTree::Visitor visitor;
    NN_RESULT_DO(m_Table.Find(&visitor, offset));
    {
        const auto entryOffset = visitor.Get<Entry>()->GetVirtualOffset();
        NN_RESULT_THROW_UNLESS(
            0 <= entryOffset && m_Table.IsInclude(entryOffset),
            fs::ResultInvalidIndirectEntryOffset()
        );
    }

    const auto endOffset = offset + size;

    int count = 0;
    auto entry = *visitor.Get<Entry>();

    while( entry.GetVirtualOffset() < endOffset )
    {
        if (entryCount != 0)
        {
            if( count >= entryCount )
            {
                break;
            }
            std::memcpy(&outEntries[count], &entry, sizeof(Entry));
        }

        count++;

        if( visitor.CanMoveNext() )
        {
            NN_RESULT_DO(visitor.MoveNext());
            entry = *visitor.Get<Entry>();
        }
        else
        {
            break;
        }
    }

    *outEntryCount = count;

    NN_RESULT_SUCCESS;
}

/**
 * @brief       範囲指定処理を行います。
 *
 * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
 * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
 * @param[in]   operationId      範囲指定処理の種類
 * @param[in]   offset           範囲指定処理開始位置
 * @param[in]   size             範囲指定処理を行うデータサイズ
 * @param[in]   inBuffer         範囲指定処理に渡すバッファ
 * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
 *
 * @return      関数の処理結果を返します。
 */
Result IndirectStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, offset);
    NN_SDK_REQUIRES_LESS_EQUAL(0, size);
    NN_SDK_REQUIRES(IsInitialized());

    switch( operationId )
    {
    case fs::OperationId::Invalidate:
        if( 0 < size )
        {
            NN_RESULT_THROW_UNLESS(m_Table.IsInclude(offset, size), fs::ResultOutOfRange());

            if( !m_Table.IsEmpty() )
            {
                NN_RESULT_DO(m_Table.InvalidateCache());

                NN_RESULT_DO(OperatePerEntry<false>(
                    offset,
                    size,
                    [=](fs::SubStorage* pStorage,
                        int64_t dataOffset,
                        int64_t processOffset,
                        int64_t processSize) NN_NOEXCEPT -> Result
                    {
                        NN_UNUSED(processOffset);
                        NN_RESULT_DO(pStorage->OperateRange(
                            outBuffer,
                            outBufferSize,
                            operationId,
                            dataOffset,
                            processSize,
                            inBuffer,
                            inBufferSize
                        ));
                        NN_RESULT_SUCCESS;
                    }
                ));
            }
        }
        break;

    case fs::OperationId::QueryRange:
        NN_RESULT_THROW_UNLESS(outBuffer != nullptr, fs::ResultNullptrArgument());
        NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(fs::QueryRangeInfo), fs::ResultInvalidArgument());
        if( 0 < size )
        {
            NN_RESULT_THROW_UNLESS(m_Table.IsInclude(offset, size), fs::ResultOutOfRange());

            if( !m_Table.IsEmpty() )
            {
                fs::QueryRangeInfo infoMerged;
                infoMerged.Clear();

                NN_RESULT_DO(OperatePerEntry<false>(
                    offset,
                    size,
                    [=, &infoMerged](
                        fs::SubStorage* pStorage,
                        int64_t dataOffset,
                        int64_t processOffset,
                        int64_t processSize) NN_NOEXCEPT -> Result
                    {
                        NN_UNUSED(processOffset);
                        fs::QueryRangeInfo infoEntry;
                        NN_RESULT_DO(pStorage->OperateRange(
                            &infoEntry,
                            sizeof(infoEntry),
                            operationId,
                            dataOffset,
                            processSize,
                            inBuffer,
                            inBufferSize
                        ));
                        infoMerged.Merge(infoEntry);
                        NN_RESULT_SUCCESS;
                    }
                ));

                *reinterpret_cast<fs::QueryRangeInfo*>(outBuffer) = infoMerged;
            }
        }
        break;

    case fs::OperationId::DestroySignature:
    case fs::OperationId::FillZero:
        NN_RESULT_THROW(fs::ResultUnsupportedOperation());

    default:
        NN_RESULT_THROW(fs::ResultInvalidArgument());
    }

    NN_RESULT_SUCCESS;
}

}}
