﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_StaticAssert.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/crypto/crypto_Sha256Generator.h>


namespace nn { namespace fssystem {


// 暫定で 2 階層固定の階層化ハッシュレイヤ
class HierarchicalSha256Storage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(HierarchicalSha256Storage);

public:
    static const int LayerCount = 3;
    static const size_t HashSize = crypto::Sha256Generator::HashSize;

private:

    IStorage* m_pBaseStorage; // 完全性検証対象
    int64_t m_BaseStorageSize;

    char* m_HashBuffer;
    size_t m_HashBufferSize;

    int m_HashTargetBlockSize;
    int m_Log2SizeRatio;

    // バッファを排他利用する
    os::Mutex m_Mutex;

public:
    HierarchicalSha256Storage() NN_NOEXCEPT;

    /**
        @brief データ領域、階層化ハッシュ領域、マスターハッシュ領域を受け取り、階層化ハッシュでデータ領域を完全性検証するストレージレイヤを構築します。
        @pre
            - ppBaseStorage[0] はマスターハッシュ領域
            - ppBaseStorage[1] はハッシュ領域 (1段目)
            - ppBaseStorage[2] はデータ領域

            - hashBufferSize >= ppBaseStorage[1] のサイズ
            - hashTargetBlockSize == 512KB (暫定の制約)
    */
    Result Initialize(IStorage** ppBaseStorage, int layerCount, size_t hashTargetBlockSize, void* hashBuffer, size_t hashBufferSize) NN_NOEXCEPT;

    /**
        @pre
            - is_aligned(offset, BlockSize)
            - is_aligned(size,   BlockSize)
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    // TORIAZU: テスト目的のみ。
    // データ領域とハッシュバッファを更新するのみで、更新したハッシュを永続化しない
    // マスターハッシュも更新しない
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;
};


}}
