﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_ApplicationTemporaryFileSystem.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/fs_IMacGenerator.h>

#include "fs_SaveDataFileSystemResultHandlingUtility.h"

using namespace nn;
using namespace nn::fs;

namespace
{
    static_assert(sizeof(nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData) == sizeof(nn::fs::SaveDataExtraData),
                  "sizeof(nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData) must be equal sizeof(nn::fs::SaveDataExtraData).");
}

#define NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(r) \
{ \
    NN_RESULT_DO(detail::ConvertSaveFsDriverPrivateResult(r)); \
}

namespace nn { namespace fssystem {

    namespace
    {
        class ApplicationTemporaryFile: public nn::fs::fsa::IFile, public nn::fs::detail::Newable
        {
        public:
            ApplicationTemporaryFile(nn::fssystem::save::IFileSystem* pFileSystem, nn::fssystem::save::IFile* pFile, nn::fs::OpenMode mode) NN_NOEXCEPT
                : m_FileSystem(pFileSystem),
                  m_File(pFile),
                  m_Mode(mode)
            {
                NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
                NN_SDK_REQUIRES_NOT_NULL(pFile);
            }

            virtual ~ApplicationTemporaryFile() NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(m_FileSystem);
                NN_SDK_REQUIRES_NOT_NULL(m_File);
                m_FileSystem->CloseFile(m_File);
            }

            virtual Result DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoGetSize(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE;
            virtual Result DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

        private:
            nn::fssystem::save::IFileSystem* m_FileSystem;
            nn::fssystem::save::IFile* m_File;
            const nn::fs::OpenMode m_Mode;
        };

        Result ApplicationTemporaryFile::DoRead(size_t* outValue, int64_t offset, void *buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());

            size_t readSize = 0;
            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));

            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_File->ReadBytes(offset, buffer, readSize));

            *outValue = readSize;

            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryFile::DoWrite(int64_t offset, const void *buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());

            NN_FSP_REQUIRES((m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode());

            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_File->WriteBytes(offset, buffer, size));

            if ((option.flags & nn::fs::WriteOptionFlag_Flush) != 0)
            {
                return DoFlush();
            }

            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryFile::DoFlush() NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());
            if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_File->Flush());
            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryFile::DoSetSize(int64_t size) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());
            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(DrySetSize(size, m_Mode));
            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_File->SetSize(size));
            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryFile::DoGetSize(int64_t *outValue) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());

            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_File->GetSize(outValue));
            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryFile::DoOperateRange(
                void* outBuffer,
                size_t outBufferSize,
                OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize) NN_NOEXCEPT
        {
            if( operationId == nn::fs::OperationId::QueryRange )
            {
                NN_FSP_REQUIRES(m_File != nullptr, nn::fs::ResultNullptrArgument());
                NN_RESULT_DO(m_File->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize));
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
        }

        class ApplicationTemporaryDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
        {
        public:
            ApplicationTemporaryDirectory(
                    nn::fssystem::save::IFileSystem* pFileSystem,
                    nn::fssystem::save::IDirectory* pDirectory,
                    const nn::fssystem::save::Path& path,
                    nn::fs::OpenDirectoryMode mode
                ) NN_NOEXCEPT
                : m_FileSystem(pFileSystem),
                  m_Directory(pDirectory),
                  m_Path(path),
                  m_Mode(mode)
            {
                NN_SDK_REQUIRES_NOT_NULL(pFileSystem);
                NN_SDK_REQUIRES_NOT_NULL(pDirectory);
            }

            virtual ~ApplicationTemporaryDirectory() NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(m_FileSystem);
                NN_SDK_REQUIRES_NOT_NULL(m_Directory);
                m_FileSystem->CloseDirectory(m_Directory);
            }

            virtual Result DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT;
            virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT;

        private:
            nn::fssystem::save::IFileSystem* m_FileSystem;
            nn::fssystem::save::IDirectory* m_Directory;
            nn::fssystem::save::Path m_Path;
            nn::fs::OpenDirectoryMode m_Mode;
        };

        Result ApplicationTemporaryDirectory::DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(m_Directory != nullptr, nn::fs::ResultNullptrArgument());

            int32_t count = 0;
            while( entryBufferCount > count )
            {
                nn::fs::DirectoryEntry entry;
                int32_t readCount = 0;
                NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(
                    m_Directory->Read(&readCount, &entry, 1)
                );
                if( readCount == 0 )
                {
                    break;
                }
                if( m_Mode == nn::fs::OpenDirectoryMode::OpenDirectoryMode_Directory )
                {
                    if( entry.directoryEntryType != nn::fs::DirectoryEntryType_Directory )
                    {
                        continue;
                    }
                }
                if( m_Mode == nn::fs::OpenDirectoryMode::OpenDirectoryMode_File )
                {
                    if( entry.directoryEntryType != nn::fs::DirectoryEntryType_File )
                    {
                        continue;
                    }
                }
                entryBuffer[count] = entry;
                ++count;
            }
            *outValue = count;

            NN_RESULT_SUCCESS;
        }

        Result ApplicationTemporaryDirectory::DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT
        {
            nn::fssystem::save::IDirectory* pDirectory;
            NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem->OpenDirectory(&pDirectory, m_Path));

            int64_t entryCount = 0;
            for( ; ; )
            {
                nn::fs::DirectoryEntry entry;
                int32_t readCount;
                NN_RESULT_TRY(pDirectory->Read(&readCount, &entry, 1))
                    NN_RESULT_CATCH_ALL
                    {
                        m_FileSystem->CloseDirectory(pDirectory);
                        NN_RESULT_RETHROW;
                    }
                NN_RESULT_END_TRY;

                if( readCount == 0 )
                {
                    m_FileSystem->CloseDirectory(pDirectory);
                    *outValue = entryCount;
                    NN_RESULT_SUCCESS;
                }
                if( m_Mode == nn::fs::OpenDirectoryMode::OpenDirectoryMode_Directory )
                {
                    if( entry.directoryEntryType != nn::fs::DirectoryEntryType_Directory )
                    {
                        continue;
                    }
                }
                if( m_Mode == nn::fs::OpenDirectoryMode::OpenDirectoryMode_File )
                {
                    if( entry.directoryEntryType != nn::fs::DirectoryEntryType_File )
                    {
                        continue;
                    }
                }
                ++entryCount;
            }
        }
    }

    ApplicationTemporaryFileSystem::ApplicationTemporaryFileSystem() NN_NOEXCEPT : m_Observer(nullptr)
    {
    }

    ApplicationTemporaryFileSystem::~ApplicationTemporaryFileSystem() NN_NOEXCEPT
    {
        m_FileSystem.Commit();
        m_FileSystem.Finalize();
        if (m_Observer)
        {
            m_Observer->Unregister(m_SpaceId, m_SaveDataId, this);
        }
    }

    Result ApplicationTemporaryFileSystem::Initialize(nn::fs::IStorage* pStorage, IBufferManager* pBufferManager, IMacGenerator* pMacGenerator) NN_NOEXCEPT
    {
        int64_t size;
        NN_RESULT_DO(pStorage->GetSize(&size));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.Initialize(nn::fs::SubStorage(pStorage, 0, size), pBufferManager, pMacGenerator));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::Initialize(std::shared_ptr<nn::fs::IStorage> pStorage, IBufferManager* pBufferManager, IMacGenerator* pMacGenerator) NN_NOEXCEPT
    {
        m_Storage = std::move(pStorage);
        NN_RESULT_THROW(Initialize(m_Storage.get(), pBufferManager, pMacGenerator));
    }

    Result ApplicationTemporaryFileSystem::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(option);
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.CreateFile(pathBuf, size));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.DeleteFile(pathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.CreateDirectory(pathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.DeleteDirectory(pathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.DeleteDirectoryRecursively(pathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.CleanDirectoryRecursively(pathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        nn::fssystem::save::Path currentPathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(currentPathBuf.Initialize(currentPath));
        nn::fssystem::save::Path newPathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(newPathBuf.Initialize(newPath));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.RenameFile(currentPathBuf, newPathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        nn::fssystem::save::Path currentPathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(currentPathBuf.Initialize(currentPath));
        nn::fssystem::save::Path newPathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(newPathBuf.Initialize(newPath));
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.RenameDirectory(currentPathBuf, newPathBuf));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char *path) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));
        bool isFile;
        bool isDirectory;
        NN_RESULT_DO(m_FileSystem.HasFile(&isFile, pathBuf));
        if( isFile )
        {
            *outValue = nn::fs::DirectoryEntryType_File;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(m_FileSystem.HasDirectory(&isDirectory, pathBuf));
        if( isDirectory )
        {
            *outValue = nn::fs::DirectoryEntryType_Directory;
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW(nn::fs::ResultPathNotFound());
    }

    Result ApplicationTemporaryFileSystem::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));

        nn::fssystem::save::IFile* pFile;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.OpenFile(&pFile, pathBuf, static_cast<int>(mode)));

        std::unique_ptr<ApplicationTemporaryFile> file(new ApplicationTemporaryFile(&m_FileSystem, pFile, mode));
        NN_RESULT_THROW_UNLESS(file, nn::fs::ResultAllocationMemoryFailedNew());

        *outValue = std::move(file);

        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        nn::fssystem::save::Path pathBuf;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(pathBuf.Initialize(path));

        nn::fssystem::save::IDirectory* pDirectory;
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.OpenDirectory(&pDirectory, pathBuf));

        std::unique_ptr<ApplicationTemporaryDirectory> directory(new ApplicationTemporaryDirectory(&m_FileSystem, pDirectory, pathBuf, mode));
        NN_RESULT_THROW_UNLESS(directory, nn::fs::ResultAllocationMemoryFailedNew());

        *outValue = std::move(directory);

        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoCommit() NN_NOEXCEPT
    {
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.Commit());

        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoCommitProvisionally(int64_t counter) NN_NOEXCEPT
    {
        NN_UNUSED(counter);
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoRollback() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoGetFreeSpaceSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);

        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.GetFreeBytes(outValue));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::DoGetTotalSpaceSize(int64_t* outValue, const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);

        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.GetDataAreaBytes(outValue));

        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::WriteExtraData(const nn::fs::SaveDataExtraData& extraData) NN_NOEXCEPT
    {
        NN_FS_APPLICATION_TEMPORARY_FS_RESULT_DO(m_FileSystem.WriteExtraData(*reinterpret_cast<const nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData*>(&extraData)));
        NN_RESULT_SUCCESS;
    }

    Result ApplicationTemporaryFileSystem::CommitExtraData(bool updateTimeStamp) NN_NOEXCEPT
    {
        NN_UNUSED(updateTimeStamp);
        NN_RESULT_THROW(DoCommit());
    }

    Result ApplicationTemporaryFileSystem::ReadExtraData(nn::fs::SaveDataExtraData* outValue) NN_NOEXCEPT
    {
        m_FileSystem.ReadExtraData(reinterpret_cast<nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData*>(outValue));
        NN_RESULT_SUCCESS;
    }

    void ApplicationTemporaryFileSystem::RegisterCacheObserver(ISaveDataExtraDataAccessorCacheObserver* observer, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        m_Observer = observer;
        m_SpaceId = spaceId;
        m_SaveDataId = saveDataId;
    }

}}
