﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_BitUtil.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorageImpl.h>

namespace nn { namespace fssystem {


namespace {

template< typename T >
size_t GetRoundDownDiff(T x, size_t align) NN_NOEXCEPT
{
    return static_cast<size_t>(x - util::align_down(x, align));
}

template< typename T >
size_t GetRoundUpDiff(T x, size_t align) NN_NOEXCEPT
{
    return static_cast<size_t>(util::align_up(x, align) - x);
}

template< typename T >
size_t GetRoundUpDiff(T* x, size_t align) NN_NOEXCEPT
{
    return GetRoundUpDiff(reinterpret_cast<uintptr_t>(x), align);
}

} // namespace



Result AlignmentMatchingStorageImpl::Read(fs::IStorage* pBaseStorage, char* subBuffer, size_t subBufferSize, size_t dataAlignment, size_t bufferAlignment, int64_t offset, char* buffer, size_t size)
{
    NN_SDK_REQUIRES(subBufferSize >= dataAlignment);
    NN_UNUSED(subBufferSize);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    char*  alignedBodyBuffer;
    int64_t bodyOffset;
    size_t bodySize;
    size_t bufferGap;
    size_t offsetGap;
    int64_t coveredOffset;

    if( util::is_aligned(reinterpret_cast<uintptr_t>(buffer) + GetRoundUpDiff(offset, dataAlignment), bufferAlignment) )
    {
        // アクセスする領域をデータアラインに切り上げた結果、
        // 対象バッファがバッファアラインに載る場合は、その構成でバッファを使う (memmove しなくて済むため)
        alignedBodyBuffer = buffer + GetRoundUpDiff(offset, dataAlignment);

        bodySize = size < GetRoundUpDiff(offset, dataAlignment) ? 0 : util::align_down(size - GetRoundUpDiff(offset, dataAlignment), dataAlignment);
        bodyOffset = util::align_up(offset, dataAlignment);
        offsetGap = 0;
        bufferGap = 0;

        coveredOffset = bodySize > 0 ? bodyOffset : offset;
    }
    else
    {
        // さもなくば、バッファアラインで切り上げたバッファを利用可能バッファとして使う
        alignedBodyBuffer = buffer + GetRoundUpDiff(buffer, bufferAlignment);

        bodySize = size < GetRoundUpDiff(buffer, bufferAlignment) ? 0 : util::align_down(size - GetRoundUpDiff(buffer, bufferAlignment), dataAlignment);
        bodyOffset = util::align_down(offset, dataAlignment);
        offsetGap = GetRoundDownDiff(offset, dataAlignment);
        bufferGap = GetRoundUpDiff(buffer, bufferAlignment);

        coveredOffset = offset;
    }


    // ユーザバッファの利用可能領域で body を読む
    if(bodySize > 0)
    {
        NN_RESULT_DO(pBaseStorage->Read(bodyOffset, alignedBodyBuffer, bodySize));

        // TODO: 途中で失敗した際にバッファをゼロ埋めする

        // 正しいオフセットにずらす
        if( offsetGap != 0 || bufferGap != 0 )
        {
            std::memmove(alignedBodyBuffer - bufferGap, alignedBodyBuffer + offsetGap, bodySize - offsetGap);
            bodySize -= offsetGap;
        }
    }

    // head
    if( offset < coveredOffset )
    {
        int64_t headBlockOffset = util::align_down(offset, dataAlignment);
        size_t  headSize = static_cast<size_t>(coveredOffset - offset);

        NN_SDK_ASSERT(GetRoundDownDiff(offset, dataAlignment) + headSize <= subBufferSize);

        NN_RESULT_DO(pBaseStorage->Read(headBlockOffset, subBuffer, dataAlignment));
        std::memcpy(buffer, subBuffer + GetRoundDownDiff(offset, dataAlignment), headSize);
    }

    // tail
    int64_t tailOffset = coveredOffset + bodySize;
    size_t tailRestSize = static_cast<size_t>((offset + size) - tailOffset);
    while( tailRestSize > 0 )
    {
        auto copySize = std::min(static_cast<size_t>(util::align_down(tailOffset, dataAlignment) + dataAlignment - tailOffset), tailRestSize);
        int64_t tailBlockOffset = util::align_down(tailOffset, dataAlignment);
        NN_RESULT_DO(pBaseStorage->Read(tailBlockOffset, subBuffer, dataAlignment));

        NN_SDK_ASSERT((tailOffset - offset) + copySize <= size);
        NN_SDK_ASSERT((tailOffset - tailBlockOffset) + copySize <= dataAlignment);
        std::memcpy(static_cast<char*>(buffer) + (tailOffset - offset), subBuffer + (tailOffset - tailBlockOffset), copySize);

        tailRestSize -= copySize;
        tailOffset   += copySize;
    }

    NN_RESULT_SUCCESS;
}

Result AlignmentMatchingStorageImpl::Write(fs::IStorage* pBaseStorage, char* subBuffer, size_t subBufferSize, size_t dataAlignment, size_t bufferAlignment, int64_t offset, const char* buffer, size_t size)
{
    NN_SDK_REQUIRES(subBufferSize >= dataAlignment);
    NN_UNUSED(subBufferSize);

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    const char* alignedBodyBuffer;
    int64_t bodyOffset;
    size_t bodySize;
    int64_t coveredOffset;

    if( util::is_aligned(reinterpret_cast<uintptr_t>(buffer) + GetRoundUpDiff(offset, dataAlignment), bufferAlignment) )
    {
        // アクセスする領域をデータアラインに切り上げた結果、
        // 対象バッファがバッファアラインに載る場合は、その構成でバッファを使う
        alignedBodyBuffer = buffer + GetRoundUpDiff(offset, dataAlignment);

        bodySize = size < GetRoundUpDiff(offset, dataAlignment) ? 0 : util::align_down(size - GetRoundUpDiff(offset, dataAlignment), dataAlignment);
        bodyOffset = util::align_up(offset, dataAlignment);

        coveredOffset = bodySize > 0 ? bodyOffset : offset;
    }
    else
    {
        // write 時は const バッファなのでワークバッファとして使わない
        alignedBodyBuffer = nullptr;

        bodySize = 0;
        bodyOffset = util::align_down(offset, dataAlignment);

        coveredOffset = offset;
    }


    // ユーザバッファの利用可能領域で body を書く
    if(bodySize > 0)
    {
        NN_RESULT_DO(pBaseStorage->Write(bodyOffset, alignedBodyBuffer, bodySize));
    }

    // head
    if( offset < coveredOffset )
    {
        int64_t headBlockOffset = util::align_down(offset, dataAlignment);
        size_t  headSize = static_cast<size_t>(coveredOffset - offset);

        NN_SDK_ASSERT((offset - headBlockOffset) + headSize <= dataAlignment);

        // 既存データとマージ
        NN_RESULT_DO(pBaseStorage->Read(headBlockOffset, subBuffer, dataAlignment));
        std::memcpy(subBuffer + (offset - headBlockOffset), buffer, headSize);

        NN_RESULT_DO(pBaseStorage->Write(headBlockOffset, subBuffer, dataAlignment));
    }

    // tail
    int64_t tailOffset = coveredOffset + bodySize;
    size_t tailRestSize = static_cast<size_t>((offset + size) - tailOffset);
    while( tailRestSize > 0 )
    {
        NN_SDK_ASSERT(static_cast<size_t>(tailOffset - offset) < size);

        auto copySize = std::min(static_cast<size_t>(util::align_down(tailOffset, dataAlignment) + dataAlignment - tailOffset), tailRestSize);
        int64_t tailBlockOffset = util::align_down(tailOffset, dataAlignment);

        // 既存データとマージ
        NN_RESULT_DO(pBaseStorage->Read(tailBlockOffset, subBuffer, dataAlignment));
        std::memcpy(subBuffer + GetRoundDownDiff(tailOffset, dataAlignment), buffer + (tailOffset - offset), copySize);

        NN_RESULT_DO(pBaseStorage->Write(tailBlockOffset, subBuffer, dataAlignment));


        tailRestSize -= copySize;
        tailOffset   += copySize;
    }

    NN_RESULT_SUCCESS;
}

template<>
Result AlignmentMatchingStorageInBulkRead<1>::Read(
    int64_t offset,
    void* buffer,
    size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    if( buffer == nullptr )
    {
        return fs::ResultNullptrArgument();
    }
    int64_t baseStorageSize = 0;
    NN_RESULT_DO(GetSize(&baseStorageSize));
    NN_RESULT_THROW_UNLESS(fs::IStorage::CheckAccessRange(offset, size, baseStorageSize), fs::ResultOutOfRange());

    const auto offsetEnd = offset + static_cast<int64_t>(size);
    const auto offsetAligned = nn::util::align_down(offset, m_DataAlignment);
    const auto offsetEndAligned = nn::util::align_up(offsetEnd, m_DataAlignment);
    const auto sizeAligned = static_cast<size_t>(offsetEndAligned - offsetAligned);

    PooledBuffer pooledBuffer;

    // オフセットまたはサイズがアラインされていない場合のみバッファを確保する
    if( offsetAligned != offset || sizeAligned != size )
    {
        if( sizeAligned <= pooledBuffer.GetAllocatableSizeMax() )
        {
            // 可能な場合は指定サイズ分のバッファを確保して一括読み込みする
            // 十分なサイズを確保できなかった場合はアライメント調整用バッファとして利用する
            pooledBuffer.Allocate(sizeAligned, m_DataAlignment);

            if( sizeAligned <= pooledBuffer.GetSize() )
            {
                NN_RESULT_DO(m_pBaseStorage->Read(
                    offsetAligned,
                    pooledBuffer.GetBuffer(),
                    sizeAligned));
                std::memcpy(buffer, pooledBuffer.GetBuffer() + (offset - offsetAligned), size);
                NN_RESULT_SUCCESS;
            }
            else
            {
                pooledBuffer.Shrink(m_DataAlignment);
            }
        }
        else
        {
            // 指定されたサイズが大きすぎる場合はアライメント調整用バッファのみ確保する
            pooledBuffer.Allocate(m_DataAlignment, m_DataAlignment);
        }

        NN_SDK_ASSERT_GREATER_EQUAL(pooledBuffer.GetSize(), static_cast<size_t>(m_DataAlignment));
    }

    const auto offsetBody = nn::util::align_up(offset, m_DataAlignment);
    const auto offsetBodyEnd = nn::util::align_down(offsetEnd, m_DataAlignment);

    // 先頭にアラインされていない領域があればアライメント調整用バッファに読み込んでからコピーする
    if( offset < offsetBody )
    {
        const auto sizeHead = static_cast<size_t>(offsetBody - offset);
        NN_SDK_ASSERT_LESS(sizeHead, size);

        NN_RESULT_DO(m_pBaseStorage->Read(
            offsetAligned,
            pooledBuffer.GetBuffer(),
            m_DataAlignment));
        std::memcpy(buffer, pooledBuffer.GetBuffer() + (offset - offsetAligned), sizeHead);
    }

    // 中央のアラインされた領域をバッファに直接読み込む
    if( offsetBody < offsetBodyEnd )
    {
        const auto bufferBody = reinterpret_cast<char*>(buffer) + (offsetBody - offset);
        const auto sizeBody = static_cast<size_t>(offsetBodyEnd - offsetBody);

        NN_RESULT_DO(m_pBaseStorage->Read(offsetBody, bufferBody, sizeBody));

        // TODO: 途中で失敗した際にバッファをゼロ埋めする
    }

    // 末尾にアラインされていない領域があればアライメント調整用バッファに読み込んでからコピーする
    if( offsetBodyEnd < offsetEnd )
    {
        const auto bufferTail = reinterpret_cast<char*>(buffer) + (offsetBodyEnd - offset);
        const auto sizeTail = static_cast<size_t>(offsetEnd - offsetBodyEnd);

        NN_RESULT_DO(m_pBaseStorage->Read(
            offsetBodyEnd,
            pooledBuffer.GetBuffer(),
            m_DataAlignment));
        std::memcpy(bufferTail, pooledBuffer.GetBuffer(), sizeTail);
    }

    NN_RESULT_SUCCESS;
}

}}
