﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/dbm/fs_DbmUtils.h>
#include <nn/fssystem/dbm/fs_Bitmap.h>

namespace nn { namespace fssystem { namespace dbm {

namespace {

    //! フォーマット時のキャッシュサイズ
    static const uint32_t FormatCacheSize = 128;

    //! イテレート時のキャッシュサイズ
    static const uint32_t IterateCacheSize = 128;
}

/**
* @brief        ビットマップ用ストレージをフォーマットします。
*
* @param[in]    bitCount      総ブロック数
* @param[in]    storage         ビットマップ用ストレージ
*
* @return       関数の処理結果を返します。
*/
Result Bitmap::Format(
           uint32_t bitCount,
           fs::SubStorage storage
       ) NN_NOEXCEPT
{
    // ビットマップ領域を0クリアします。
    uint8_t bits[FormatCacheSize] = {};

    const size_t sizeTotal = static_cast<size_t>(QuerySize(bitCount));
    size_t size = sizeTotal;
    int64_t offsetCurr = 0;
    while( size > 0 )
    {
        const size_t length = std::min<size_t>(sizeof(bits), size);
        NN_RESULT_DO(storage.Write(offsetCurr, bits, length));
        size -= length;
        offsetCurr += length;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        ビットマップ用ストレージをフォーマットします。
*
* @param[in]    bitCountOld   拡張以前のブロック数
* @param[in]    bitCountNew   拡張後のブロック数
* @param[in]    storage       ビットマップ用ストレージ
*
* @return       関数の処理結果を返します。
*
* @pre
*               - bitCountNew > bitCountOld
*/
Result Bitmap::Expand(
           uint32_t bitCountOld,
           uint32_t bitCountNew,
           fs::SubStorage storage
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(bitCountNew, bitCountOld);

    const size_t oldSize = static_cast<size_t>(QuerySize(bitCountOld));
    const size_t newSize = static_cast<size_t>(QuerySize(bitCountNew));
    if( oldSize < newSize )
    {
        // 拡張領域を0クリアします。
        uint8_t bits[FormatCacheSize] = {};

        size_t expandSize = newSize - oldSize;
        int64_t offsetBytes = oldSize;
        while( expandSize > 0 )
        {
            const size_t length = std::min<size_t>(sizeof(bits), expandSize);
            NN_RESULT_DO(storage.Write(offsetBytes, bits, length));
            expandSize -= length;
            offsetBytes += length;
        }
    }

    NN_RESULT_SUCCESS;
}

//! コンストラクタ
Bitmap::Bitmap() NN_NOEXCEPT
: m_BitCount(0)
, m_Storage()
{
}

/**
* @brief        ビットマップ用ストレージをマウントします。
*
* @param[in]    bitCount  総ブロック数
* @param[in]    storage     ビットマップ用ストレージ
*
* @return       関数の処理結果を返します。
*/
void Bitmap::Initialize(
         uint32_t bitCount,
         fs::SubStorage storage
     ) NN_NOEXCEPT
{
    m_BitCount = bitCount;
    m_Storage = storage;
}

/**
* @brief ビットマップに対してイテレーションを開始します。
*
* @param[out]   it      イテレータ
* @param[in]    index   インデックス
*
* @return 関数の処理結果を返します。
*
* @pre
*               - it != nullptr
*               - index <= m_BitCount
*/
Result Bitmap::IterateBegin(
           Iterator* it,
           uint32_t index
       ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(it);
    NN_SDK_REQUIRES_LESS_EQUAL(index, m_BitCount);

    it->index = index;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ストレージからデータを読み込みします。
*
* @param[out]   buffer      読み込んだデータを格納するバッファ
* @param[in]    maxBytes    読み込み先のバッファサイズ。4 の倍数である必要があります。
* @param[in]    index       読み込み開始インデックス
*
* @return       関数の処理結果を返します。
*
* @details
*               読み込みは 32 ビット単位で行われます。
*               読込先のバッファサイズは 4 の倍数にしておく必要があります。
*
* @pre
*               - buffer != nullptr
*/
Result Bitmap::ReadBlock(
           void* buffer,
           uint32_t maxBytes,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    // 読み込み開始アドレスは 32 ビット境界に配置し、
    // 読み込みは 32 ビット単位で行います。
    NN_SDK_ASSERT((maxBytes & 0x3) == 0);
    //NN_SDK_ASSERT((index & 0x1F) == 0);

    NN_RESULT_DO(
        m_Storage.Read(
            nn::util::align_down(index, 32) / 8,
            buffer,
            maxBytes
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        ビットマップに対してイテレーションを行います。
*
* @param[out]   outZeroCount    0 が連続している個数
* @param[out]   outOneCount     1 が連続している個数
* @param[in,out]it              イテレータ
*
* @return       関数の処理結果を返します。
*
* @details
*               outZeroCount または outOneCount のどちらかは必ず 0 になります。
*               イテレーション完了時には両方とも 0 が格納されます。
*
* @pre
*               - outZeroCount != nullptr
*               - outOneCount != nullptr
*               - it != nullptr
*/
Result Bitmap::IterateNext(
           uint32_t* outZeroCount,
           uint32_t* outOneCount,
           Iterator* it
       ) NN_NOEXCEPT
{
    return LimitedIterateNext(outZeroCount, outOneCount, it, 0, 0);
}

/**
* @brief        ビットマップに対して読み込むサイズを制限したイテレーションを行います。
*
* @param[out]       outZeroCount        0 が連続している個数
* @param[out]       outOneCount         1 が連続している個数
* @param[in,out]    it                  イテレータ
* @param[in]        countMaxIterateZero 0 が連続しているかチェックする最大個数（0 で無制限）
* @param[in]        countMaxIterateOne  1 が連続しているかチェックする最大個数（0 で無制限）
*
* @return       関数の処理結果を返します。
*
* @details
*               outZeroCount または outOneCount のどちらかは必ず 0 になります。
*               イテレーション完了時には両方とも 0 が格納されます。
*
* @pre
*               - outZeroCount != nullptr
*               - outOneCount != nullptr
*               - it != nullptr
*/
Result Bitmap::LimitedIterateNext(
           uint32_t* outZeroCount,
           uint32_t* outOneCount,
           Iterator* it,
           uint32_t countMaxIterateZero,
           uint32_t countMaxIterateOne
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outZeroCount);
    NN_SDK_ASSERT_NOT_NULL(outOneCount);
    NN_SDK_ASSERT_NOT_NULL(it);

    *outZeroCount = 0;
    *outOneCount = 0;

    // イテレーションが終了したかどうかをチェックします。
    NN_SDK_ASSERT(it->index <= m_BitCount);
    if( it->index >= m_BitCount )
    {
        // イテレーションは終了しました。
        NN_RESULT_SUCCESS;
    }

    uint8_t buffer[IterateCacheSize];

    uint32_t totalCount = 0;
    bool bFindZero = true;
    bool bFindFinished = false;
    while( it->index < m_BitCount )
    {
        // バッファに次のデータを読み込みます。
        size_t startOffset = nn::util::align_down(it->index, 32);
        size_t endOffset = nn::util::align_up(m_BitCount, 32);
        uint32_t readRequestBytes = static_cast<uint32_t>((endOffset - startOffset) / 8);
        readRequestBytes = std::min(readRequestBytes, IterateCacheSize);
        NN_RESULT_DO(ReadBlock(buffer, readRequestBytes, it->index));

        // 4 バイト単位でビットの連続出現数を計測します。
        uint32_t indexRead = 0;
        while( indexRead < readRequestBytes )
        {
            // ビット列先頭から目的のビットが連続している個数を取得します。
            uint32_t x = ReadU32(buffer, indexRead);
            uint32_t shiftCount = it->index % 32;
            x <<= shiftCount;
            uint32_t count = 0;

            // 連続する 0/1 どちらの個数を検出するか決定します。
            if( totalCount == 0 )
            {
                count = CountLeadingZeros(x);
                if( count > 0 )
                {
                    bFindZero = true;
                }
                else
                {
                    count = CountLeadingOnes(x);
                    if( count > 0 )
                    {
                        bFindZero = false;
                    }
                    else
                    {
                        NN_SDK_ASSERT(false);
                    }
                }
            }
            else
            {
                // 2 回目以降の探索処理
                if( bFindZero )
                {
                    count = CountLeadingZeros(x);
                    if( count == 0 )
                    {
                        *outZeroCount = totalCount;
                        bFindFinished = true;
                        break;
                    }
                }
                else
                {
                    count = CountLeadingOnes(x);
                    if( count == 0 )
                    {
                        *outOneCount = totalCount;
                        bFindFinished = true;
                        break;
                    }
                }
            }

            // 実際見るべき範囲は以下のとおりです。
            bFindFinished = false;
            if( count >= 32 - shiftCount )
            {
                // データの末尾まで等しいデータが並んでいました
                count = 32 - shiftCount;
            }
            else
            {
                bFindFinished = true;
            }

            // 個数を残りのビット数に丸め込みます。
            uint32_t leftBitCount = m_BitCount - it->index;
            if( count >= leftBitCount )
            {
                count = leftBitCount;
                bFindFinished = true;
            }

            totalCount += count;
            it->index += count;

            if( bFindZero )
            {
                if( countMaxIterateZero > 0 && totalCount >= countMaxIterateZero )
                {
                    it->index -= totalCount - countMaxIterateZero;
                    totalCount = countMaxIterateZero;
                    bFindFinished = true;
                }
            }
            else
            {
                if( countMaxIterateOne > 0 && totalCount >= countMaxIterateOne )
                {
                    it->index -= totalCount - countMaxIterateOne;
                    totalCount = countMaxIterateOne;
                    bFindFinished = true;
                }
            }

            if( bFindFinished )
            {
                if( bFindZero )
                {
                    *outZeroCount = totalCount;
                }
                else
                {
                    *outOneCount = totalCount;
                }
                break;
            }

            // 次の 4 バイトに対して連続出現数を計測します。
            indexRead += 4;
        }

        if( bFindFinished )
        {
            break;
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief        指定範囲に対してビットデータを反転します。
*
* @param[in]    index   インデックス
* @param[in]    count   ビット数
*
* @return       関数の処理結果を返します。
*/
Result Bitmap::Reverse(
           uint32_t index,
           uint32_t count
       ) NN_NOEXCEPT
{
    // 範囲チェックを行います。
    if( index + count > m_BitCount )
    {
        return nn::fs::ResultOutOfRange();
    }

    uint32_t offset = index % 32;

    // 4 バイト単位で書き込みを行います。
    while( count > 0 )
    {
        // ビットマップデータを読み込みます。
        uint8_t bits[sizeof(uint32_t)];
        NN_RESULT_DO(
            m_Storage.Read(
                ((index / 32) * sizeof(uint32_t)),
                bits,
                sizeof(uint32_t)
            )
        );

        const uint32_t orgX = ReadU32(bits, 0);

        // 対象範囲をビットマップを反転したものを取得します。
        uint32_t mask = (count < 32) ? ~((1 << (32 - count)) - 1) : 0xffffffffu;
        if( offset > 0 )
        {
            mask >>= offset;
        }
        uint32_t x = orgX ^ mask;

        WriteU32(bits, 0, x);

        // 反転したデータを保存します。
        NN_RESULT_DO(
            m_Storage.Write(
                ((index / 32) * sizeof(uint32_t)),
                bits,
                sizeof(uint32_t)
            )
        );
        // 書き込み位置、残り書き込み個数を更新します。
        index += (32 - offset);
        if( count + offset > 32 )
        {
            count -= (32 - offset);
        }
        else
        {
            count = 0;
        }
        offset = 0;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        ビットデータを0クリアします。
*
* @return       関数の処理結果を返します。
*/
Result Bitmap::Clear() NN_NOEXCEPT
{
    uint8_t bits[FormatCacheSize] = {};
    uint8_t bitsOriginal[FormatCacheSize];

    const size_t sizeTotal = static_cast<size_t>(QuerySize(m_BitCount));
    size_t size = sizeTotal;
    int64_t offsetCurr = 0;
    while( size > 0 )
    {
        const size_t length = std::min<size_t>(sizeof(bits), size);
        NN_RESULT_DO(m_Storage.Read(offsetCurr, bitsOriginal, length));
        static const auto IsZero = [](uint8_t value) NN_NOEXCEPT
        {
            return value == 0;
        };
        if( !std::all_of(bitsOriginal, bitsOriginal + length, IsZero) )
        {
            NN_RESULT_DO(m_Storage.Write(offsetCurr, bits, length));
        }
        size -= length;
        offsetCurr += length;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したインデックスから 32 ビット分のビットマップを取得します。
*
* @param[out]   outValue    ビットマップ
* @param[in]    index       インデックス
*
* @return       関数の処理結果を返します。
*
* @details      テスト用の機能です。インデックスには 32 の倍数のみ指定できます。
*
* @pre
*               - outZeroCount != nullptr
*               - (index & 0x1F) == 0
*/
Result Bitmap::GetBitmap32(uint32_t* outValue, uint32_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    // index は 32 の倍数のみ
    NN_SDK_REQUIRES((index & 0x1F) == 0);

    Bit32 bits;
    NN_RESULT_DO(ReadBlock(&bits, sizeof(Bit32), index));

    *outValue = bits;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したインデックスのビットが 0 か 1 かを取得します。
*
* @param[out]   outValue    1 ならば true、0 ならば false
* @param[in]    index       インデックス
*
* @return       関数の処理結果を返します。
*
* @details      テスト用の機能です。
*
* @pre
*               - outValue != nullptr
*/
Result Bitmap::GetBit(bool* outValue, uint32_t index) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    if( index >= m_BitCount )
    {
        return nn::fs::ResultOutOfRange();
    }

    Bit32 bits;
    NN_RESULT_DO(ReadBlock(&bits, sizeof(Bit32), index));

    *outValue = ((bits << (index & 31)) & 0x80000000) ? true : false;

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::fssystem::dbm

