﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>

namespace nn { namespace fssystem { namespace dbm {

NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::IndexEnd );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorFreeListHeader );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorListHead );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorMaskValid );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorMaskPreviousIsListHead );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorReservedCount );
NN_DEFINE_STATIC_CONSTANT( const uint32_t AllocationTable::SectorEmpty );

/**
* @brief        指定したブロック数のアロケーションテーブルに必要なデータサイズを取得します。
*
* @param[in]    blockCount      ブロック数
*
* @return       指定したブロック数のアロケーションテーブルに必要なデータサイズ。
*
* @details      指定したブロック数のアロケーションテーブルに必要なデータサイズを取得します。
*/
int64_t AllocationTable::QuerySize(uint32_t blockCount) NN_NOEXCEPT
{
    // フリーリスト管理領域分増やしたサイズを要求します。
    return (sizeof(TableElement) * (blockCount + SectorReservedCount));
}

/**
* @brief        アロケーションテーブルをフォーマットします。
*
* @param[in]    storage         アロケーションテーブルに割り当てる領域
* @param[in]    offsetStorage   アロケーションテーブル開始オフセット(バイト)
* @param[in]    blockCount      ブロック数(管理領域は含まない)
*
* @return       関数の処理結果を返します。
* @return       ResultSuccess   正常に初期化できました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          blockCount >= 1
*
* @details      アロケーションテーブルをフォーマットします。
*               初めてマウントする前に 1 度だけ呼び出してください。
*/
Result AllocationTable::Format(
           fs::SubStorage storage,
           uint32_t blockCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(1U, blockCount);

    // フリーリスト管理領域は次のセクタを指すように初期化します。
    {
        TableElement blockFreeListHeader;
        blockFreeListHeader.SetListFreeHeader();
        NN_RESULT_DO(
            storage.Write(
                SectorFreeListHeader * sizeof(TableElement),
                &blockFreeListHeader,
                sizeof(TableElement)
            )
        );
    }

    // フリーリストの先頭ブロックを初期化します。
    {
        TableElement blockFirstListParent;
        blockFirstListParent.SetListHead(SectorFreeListHeader, ( blockCount > 1 ));
        NN_RESULT_DO(
            storage.Write(
                SectorReservedCount * sizeof(TableElement),
                &blockFirstListParent,
                sizeof(TableElement)
            )
        );
    }

    // フリーリストの 2 番目以降のブロックを連続ブロックの子として初期化します。
    if( blockCount > 1 )
    {
        TableElement block;
        block.SetChainChild(SectorReservedCount, blockCount);

        // 処理簡略化のため、2番目と最後のエントリーのみ初期化します。
        NN_RESULT_DO(
            storage.Write(
                (SectorReservedCount + 1) * sizeof(TableElement),
                &block,
                sizeof(TableElement)
            )
        );
        NN_RESULT_DO(
            storage.Write(
                (SectorReservedCount + blockCount - 1) * sizeof(TableElement),
                &block,
                sizeof(TableElement)
            )
        );
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        アロケーションテーブル用ストレージを拡張します。
*
* @param[in]    storage         アロケーションテーブルに割り当てる領域
* @param[in]    offsetStorage   アロケーションテーブル開始オフセット(バイト)
* @param[in]    oldBlockCount   今までのブロック数(要素数)
* @param[in]    newBlockCount   新しいブロック数(要素数)
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に初期化できました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @pre          アンマウントしてから実行する必要があります。
* @pre          newBlockCount > oldBlockCount
*
* @details      アロケーションテーブル用ストレージを拡張します。
*/
Result AllocationTable::Expand(
           fs::SubStorage storage,
           uint32_t oldBlockCount,
           uint32_t newBlockCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(oldBlockCount, newBlockCount);

    // フリーリストを読み込みます。
    TableElement freeList;
    NN_RESULT_DO(
        storage.Read(
            SectorFreeListHeader * sizeof(TableElement),
            &freeList,
            sizeof(TableElement)
        )
    );

    uint32_t firstSector = SectorReservedCount + oldBlockCount;
    uint32_t endSector = SectorReservedCount + newBlockCount - 1;

    // 新規に追加した部分の先頭ブロックをフリーリストの先頭として初期化します。
    TableElement block;
    block.SetListHead(freeList.sectorNext, (newBlockCount - oldBlockCount > 1));
    NN_RESULT_DO(
        storage.Write(
            firstSector * sizeof(TableElement),
            &block,
            sizeof(TableElement)
        )
    );

    // 新規に追加した部分を連続ブロックの子として初期化します。
    block.SetChainChild(firstSector, endSector);
    for( uint32_t sector = firstSector + 1; sector <= endSector; ++sector )
    {
        NN_RESULT_DO(
            storage.Write(
                sector * sizeof(TableElement),
                &block,
                sizeof(TableElement)
            )
        );
    }

    // 今までフリーリストの先頭だった箇所へのリンクを更新します。
    if( !freeList.IsFreeListEmpty() )
    {
        NN_RESULT_DO(
            storage.Read(
                freeList.sectorNext * sizeof(TableElement),
                &block,
                sizeof(TableElement)
            )
        );
        block.sectorPrevious = firstSector;
        NN_RESULT_DO(
            storage.Write(
                freeList.sectorNext * sizeof(TableElement),
                &block,
                sizeof(TableElement)
            )
        );
    }

    // フリーリストを更新します。
    freeList.sectorNext = firstSector;
    NN_RESULT_DO(
        storage.Write(
            SectorFreeListHeader * sizeof(TableElement),
            &freeList,
            sizeof(TableElement)
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタです。
*
* @details      コンストラクタです。
*/
AllocationTable::AllocationTable() NN_NOEXCEPT
    : m_Table(),
      m_BlockCount(0)
{
}

/**
* @brief        ストレージをマウントします。
*
* @param[in]    storage         アロケーションテーブルに割り当てるストレージ
* @param[in]    offsetStorage   ストレージにおけるオフセット
* @param[in]    blockCount      ブロック数
*
* @details      ストレージをマウントします。
*/
void AllocationTable::Initialize(
         fs::SubStorage storage,
         uint32_t blockCount
     ) NN_NOEXCEPT
{
    m_Table = storage;
    m_BlockCount = blockCount;
}

/**
* @brief        ストレージをアンマウントします。
*
* @details      ストレージをアンマウントします。
*/
void AllocationTable::Finalize() NN_NOEXCEPT
{
    m_BlockCount = 0;
}

/**
* @brief        指定したサイズ分のデータ領域をテーブルから確保します。
*
* @param[out]   outIndex                    確保したデータ領域の先頭インデックス
* @param[in]    blockCount                  確保するブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess               正常にデータを読み込みました。
* @retval       ResultAllocationTableFull   フリーリストに空きが無い。
* @retval       上記以外                    ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          outIndex が NULL ではない。
* @pre          blockCount が 0 以下ではない。
*
* @details      指定したサイズ分のデータ領域をテーブルから確保します。
*/
Result AllocationTable::Allocate(
           uint32_t* outIndex,
           uint32_t blockCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);
    NN_SDK_REQUIRES_LESS(0U, blockCount);

    // フリーリストを読み込みます。
    TableElement freelist;
    NN_RESULT_DO(ReadFreeList(&freelist));

    uint32_t headSector = SectorEmpty;
    uint32_t previousSector = SectorEmpty;
    uint32_t parentSector = SectorEmpty;
    uint32_t previousParentSector = SectorEmpty;
    uint32_t lastSector = SectorEmpty;
    uint32_t lastBlock = blockCount;
    while( lastBlock > 0 )
    {
        // フリーリストに空きがなくなりました。
        if( freelist.IsFreeListEmpty() )
        {
            break;
        }

        // フリーリストから空いているデータ領域を取得します。
        uint32_t sector = 0;
        uint32_t chainBlockCount = 0;
        NN_RESULT_DO(AllocateFreeSector(&sector, &chainBlockCount, &freelist, lastBlock));

        if( headSector == SectorEmpty )
        {
            // 初回データ領域へのセクタを覚えておきます。
            headSector = sector;

            // 現在のセクタを連続領域の先頭とします。
            previousParentSector = SectorEmpty;
            parentSector = sector;
            if( chainBlockCount > 1 )
            {
                lastSector = sector + chainBlockCount - 1;
            }
            else
            {
                lastSector = SectorEmpty;
            }

            previousSector = sector;
        }
        else if( sector - 1 == previousSector )
        {
            // 前回の領域と連続しています。

            // 連続領域の末尾だけを更新します。
            lastSector = previousSector + chainBlockCount;

            previousSector += chainBlockCount;
        }
        else
        {
            // 連続領域が分断されました。

            // 前回の連続領域を書き換えます。
            // また、連続領域の親ブロックの次セクタを sector に設定します。
            NN_RESULT_DO(
                UpdateChain(
                    headSector,
                    parentSector,
                    lastSector,
                    previousParentSector,
                    sector
                )
            );

            // 現在の連続領域の先頭、末尾のセクタを更新します。
            previousParentSector = parentSector;
            parentSector = sector;
            if( chainBlockCount > 1 )
            {
                lastSector = sector + chainBlockCount - 1;
            }
            else
            {
                lastSector = SectorEmpty;
            }

            previousSector = sector;
        }

        // 書き込む残りデータサイズを更新します。
        if( lastBlock > chainBlockCount )
        {
            lastBlock -= chainBlockCount;
        }
        else
        {
            lastBlock = 0;
        }
    }

    // 最後の連続領域を書き換えます。
    // 最後の連続領域での親ブロックの次セクタは SectorFreeListHeader です。
    NN_RESULT_DO(
        UpdateChain(
            headSector,
            parentSector,
            lastSector,
            previousParentSector,
            SectorFreeListHeader
        )
    );

    if( lastBlock > 0 )
    {
        // 空きが足りなかった場合は今回確保したセクタを解放します。
        if( headSector != SectorEmpty )
        {
            // フリーリストが空の状態でしかここにはきません。
            NN_SDK_ASSERT(freelist.IsFreeListEmpty());

            // 今回確保したリストをフリーリストにします。
            freelist.sectorNext = headSector;

            // フリーリストを更新します。
            NN_RESULT_DO(UpdateFreeList(&freelist));
        }
        return nn::fs::ResultAllocationTableFull();
    }

    // フリーリストを更新します。
    NN_RESULT_DO(UpdateFreeList(&freelist));

    *outIndex = ConvertSectorToIndex(headSector);

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したデータ領域を解放します。
*
* @param[in]    index           解放対象ブロックリストに含まれるインデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に解放できました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
*
* @details      指定したデータ領域を解放します。
*/
Result AllocationTable::Free(uint32_t index) NN_NOEXCEPT
{
    // フリーリストを読み込みます。
    TableElement freelist;
    NN_RESULT_DO(ReadFreeList(&freelist));

    // 今までのフリーリストの先頭を今回解放するリストの後方に移動します。
    if( !freelist.IsFreeListEmpty() )
    {
        NN_RESULT_DO(Concat(index, ConvertSectorToIndex(freelist.sectorNext)));
    }

    // 今回解放したリストをフリーリストの先頭とします。
    freelist.sectorNext = ConvertIndexToSector(index);

    // フリーリストを更新します。
    NN_RESULT_DO(UpdateFreeList(&freelist));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したデータ領域の次のデータ領域を取得します。
*
* @param[out]   outNextIndex        次の連続ブロックチェインの先頭インデックス
*                                   最後の連続ブロックチェインであれば INDEX_EMPTY
* @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
* @param[in]    index               現在の連続ブロックチェインの先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       処理が終了しました。
* @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outNextIndex が NULL ではない。
* @pre          outBlockCount が NULL ではない。
* @pre          index が 空のセクタではない。
*
* @details      連続ブロックチェイン単位でイテレーションします。
*/
Result AllocationTable::ReadNext(
           uint32_t* outNextIndex,
           uint32_t* outBlockCount,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outNextIndex);
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);
    NN_SDK_REQUIRES_NOT_EQUAL(SectorEmpty, ConvertIndexToSector(index));

    // 指定した位置から 2 ブロック取得します。
    // (連続していた時にすぐブロック数を求められるようにしておきます)
    TableElement blocks[2];
    uint32_t readCount = GetReadBlockCount(ConvertIndexToSector(index));

    NN_RESULT_DO(ReadBlock(blocks, ConvertIndexToSector(index), readCount));

    if( blocks[0].IsChainParent() )
    {
        NN_SDK_ASSERT(!blocks[1].IsChainParent());

        // 連続領域の先頭であれば次のエントリーから個数を取得します。
        *outBlockCount = blocks[1].sectorNext - index;
    }
    else if( blocks[0].IsListHead() || blocks[0].IsSingleParent() )
    {
        *outBlockCount = 1;
    }
    else
    {
        // 連続領域の子ブロックです。
        NN_SDK_ASSERT(blocks[0].IsChainChild());

        // 連続領域の子ブロック対する ReadNext は未実装です。
        NN_SDK_ASSERT(false);
        return nn::fs::ResultNotImplemented();
    }
    if( blocks[0].IsTailParent() )
    {
        // 最後の連続領域であれば終端を示す IndexEnd を設定します。
        *outNextIndex = IndexEnd;
    }
    else
    {
        // 次の連続領域へのインデックスを設定します。
        *outNextIndex = ConvertSectorToIndex(blocks[0].sectorNext & SectorMaskValid);
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したデータ領域の前データ領域を取得します。
*
* @param[out]   outPrevIndex    前の連続ブロックチェインの先頭インデックス
*                               最初の連続ブロックチェインであれば INDEX_EMPTY
* @param[out]   outBlockCount   現在の連続ブロックチェインのデータ領域数
* @param[in]    index           現在の連続ブロックチェインの先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   処理が終了しました。
* @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          pPrevIndex が NULL ではない。
* @pre          outBlockCount が NULL ではない。
* @pre          index が 空のセクタではない。
*
* @details      連続ブロックチェイン単位でイテレーションします。
*/
Result AllocationTable::ReadPrevious(
           uint32_t* outPrevIndex,
           uint32_t* outBlockCount,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPrevIndex);
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);
    NN_SDK_REQUIRES_NOT_EQUAL(SectorEmpty, index);

    // 指定した位置から 2 ブロック取得します。
    // (連続していた時にすぐブロック数を求められるようにしておきます)
    TableElement blocks[2];
    uint32_t readCount = GetReadBlockCount(ConvertIndexToSector(index));

    NN_RESULT_DO(ReadBlock(blocks, ConvertIndexToSector(index), readCount));

    if( blocks[0].IsChainParent() )
    {
        NN_SDK_ASSERT(!blocks[1].IsChainParent());

        // 連続領域の先頭であれば次のエントリーから個数を取得します。
        *outBlockCount = blocks[1].sectorNext - index;

        if( blocks[0].IsListHead() )
        {
            // ブロックリストの先頭であれば終端を示す IndexEnd を設定します。
            *outPrevIndex = IndexEnd;
        }
        else
        {
            // 手前の連続領域への先頭インデックスを設定します。
            *outPrevIndex = ConvertSectorToIndex(blocks[0].sectorPrevious & SectorMaskValid);
        }
    }
    else if( blocks[0].IsListHead() )
    {
        *outBlockCount = 1;

        // ブロックリストの先頭であれば終端を示す IndexEnd を設定します。
        *outPrevIndex = IndexEnd;
    }
    else if( blocks[0].IsSingleParent() )
    {
        *outBlockCount = 1;

        // 単一領域の場合は手前の連続領域への先頭インデックスを設定します。
        *outPrevIndex = ConvertSectorToIndex(blocks[0].sectorPrevious);
    }
    else
    {
        // 連続領域の子ブロックです。
        NN_SDK_ASSERT(blocks[0].IsChainChild());

        // 連続領域の子ブロック対する ReadPrevious は未実装です。
        NN_SDK_ASSERT(false);
        return nn::fs::ResultNotImplemented();
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        総ブロック数を取得します。
*
* @param[out]   outBlockCount   総ブロック数
* @param[in]    index           データ領域の先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に検出できました。
* @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outBlockCount が NULL ではない。
*
* @details      総ブロック数を取得します。
*/
Result AllocationTable::CalcTotalBlockCount(
           uint32_t* outBlockCount,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);

    uint32_t totalCount = 0;
    uint32_t currentIndex = index;
    while( currentIndex != IndexEnd )
    {
        uint32_t nextIndex;
        uint32_t blockCount;
        NN_RESULT_DO(ReadNext(&nextIndex, &blockCount, currentIndex));
        totalCount += blockCount;
        currentIndex = nextIndex;
    }

    *outBlockCount = totalCount;

    NN_RESULT_SUCCESS;
}

/**
* @brief        最後の領域での先頭インデックスを探索します。
*
* @param[out]   outTailIndex    最後の領域での先頭インデックス
* @param[in]    index           先頭インデックス
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に検出できました。
* @retval       上記以外        ストレージからのデータ読み込みに失敗しました。
*
* @pre          outTailIndex が NULL ではない。
*
* @details      最後の領域での先頭インデックスを探索します。
*/
Result AllocationTable::LookupTailParentIndex(
           uint32_t* outTailIndex,
           uint32_t index
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTailIndex);

    uint32_t tailIndex = index;
    for( ; ; )
    {
        uint32_t nextIndex;
        uint32_t blockCount;
        NN_RESULT_DO(ReadNext(&nextIndex, &blockCount, tailIndex));
        if( nextIndex == IndexEnd )
        {
            break;
        }
        tailIndex = nextIndex;
    }

    *outTailIndex = tailIndex;

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定した 2 つのデータ領域を連結します。
*
* @param[in]    indexBefore         連結後、前側に来るデータ領域
* @param[in]    indexAfter          連結後、後ろ側に来るデータ領域
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常にデータを読み込みました。
* @retval       上記以外            ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
*
* @details      指定した 2 つのデータ領域を連結します。
*/
Result AllocationTable::Concat(
           uint32_t indexBefore,
           uint32_t indexAfter
       ) NN_NOEXCEPT
{
    // 後方リストの先頭を読み込みます。
    TableElement blockAfter;
    NN_RESULT_DO(ReadBlock(&blockAfter, ConvertIndexToSector(indexAfter), 1));

    // 前方リストの最後の領域での先頭インデックスを取得します。
    uint32_t tailIndex = 0;
    NN_RESULT_DO(LookupTailParentIndex(&tailIndex, indexBefore));

    // 前方リストの最後の領域での先頭ブロックを読み込みます。
    TableElement blockBeforeTail;
    NN_RESULT_DO(ReadBlock(&blockBeforeTail, ConvertIndexToSector(tailIndex), 1));

    // 前方リストの前を更新します。
    NN_RESULT_DO(UpdatePrevious(ConvertIndexToSector(indexBefore), blockAfter.sectorPrevious));

    // 前方リストの最後と後方リストの先頭を接続します。
    blockBeforeTail.SetChainParent(
        blockBeforeTail.sectorPrevious,
        ConvertIndexToSector(indexAfter),
        blockBeforeTail.IsChainParent()
    );
    NN_RESULT_DO(
        WriteBlock(
            &blockBeforeTail,
            ConvertIndexToSector(tailIndex)
        )
    );

    // 後方リストの先頭ブロックの前セクタを更新します。
    blockAfter.sectorPrevious = ConvertIndexToSector(tailIndex);
    NN_RESULT_DO(
        WriteBlock(
            &blockAfter,
            ConvertIndexToSector(indexAfter)
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief        データ領域を指定した位置で 2 分割します。
*
* @param[out]   outIndexAfter       分割後のデータ領域の先頭インデックス
* @param[in]    index               分割するデータ領域の先頭インデックス
* @param[in]    blockCount          分割後に前半部分になるブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess       正常に分割できました。
* @retval       ResultOutOfRange    blockCount が 範囲外です。
* @retval       上記以外            ストレージでの読み書きに失敗しました。
*
* @pre          outIndexAfter が NULL ではない。
*
* @details      データ領域を指定した位置で 2 分割します。
*/
Result AllocationTable::Split(
           uint32_t* outIndexAfter,
           uint32_t index,
           uint32_t blockCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndexAfter);


    // 総ブロック数を取得します。
    uint32_t totalBlockCount = 0;
    NN_RESULT_DO(CalcTotalBlockCount(&totalBlockCount, index));

    // 範囲チェック
    NN_FSP_REQUIRES(blockCount != 0, nn::fs::ResultOutOfRange());
    NN_FSP_REQUIRES(blockCount <= totalBlockCount, nn::fs::ResultOutOfRange());

    // 分断位置を取得します。
    TableElement blocks[2];

    uint32_t sector = ConvertIndexToSector(index);
    uint32_t newSector = SectorEmpty;
    uint32_t blockOffset = blockCount;
    while( sector != SectorFreeListHeader )
    {
        uint32_t readCount = GetReadBlockCount(sector);

        NN_RESULT_DO(ReadBlock(blocks, sector, readCount));

        if( blocks[0].IsChainParent() )
        {
            uint32_t chainCount = blocks[1].sectorNext - sector + 1;
            if( chainCount > blockOffset )
            {
                // 連続ブロックの途中で分断します。

                // 前半部分の接続を分断します。
                if( blockOffset > 0 )
                {
                    NN_RESULT_DO(
                        UpdateChain(
                            ConvertIndexToSector(index),    // headSector
                            sector,                         // parentSector
                            sector + blockOffset - 1,       // lastSector
                            blocks[0].sectorPrevious,       // prevSector
                            SectorFreeListHeader            // nextSector
                        )
                    );
                }
                else
                {
                    NN_RESULT_DO(
                        UpdateNext(blocks[0].sectorPrevious, SectorFreeListHeader)
                    );
                }

                // 後半部分を新しいリストとして分断します。
                newSector = sector + blockOffset;
                NN_RESULT_DO(
                    UpdateChain(
                        sector + blockOffset,                       // headSector
                        sector + blockOffset,                       // parentSector
                        sector + chainCount - 1,                    // lastSector
                        SectorMaskPreviousIsListHead,               // prevSector
                        (blocks[0].sectorNext & SectorMaskValid)    // nextSector
                    )
                );
                break;
            }
            else
            {
                // 次の連続領域に移動します。
                blockOffset -= chainCount;
                sector = ( blocks[0].sectorNext & SectorMaskValid );
            }
        }
        else
        {
            NN_SDK_ASSERT(blocks[0].IsListHead() || blocks[0].IsSingleParent());
            if( blockOffset == 0 )
            {
                // 単一領域を基準として分断します

                // 前半のリストを分断します。
                NN_RESULT_DO(UpdateNext(blocks[0].sectorPrevious, SectorFreeListHeader));

                // 後半部分を新しいリストとして分断します。
                newSector = sector;
                NN_RESULT_DO(UpdatePrevious(sector, SectorMaskPreviousIsListHead));

                break;
            }
            else
            {
                // 次の連続領域に移動します。
                blockOffset--;
                sector = blocks[0].sectorNext;
            }
        }
    }

    NN_SDK_ASSERT_NOT_EQUAL(SectorEmpty, newSector);
    *outIndexAfter = ConvertSectorToIndex(newSector);

    NN_RESULT_SUCCESS;
}

/**
* @brief        フリーリストの先頭インデックスを取得します。
*
* @param[out]   outIndex        フリーリストの先頭インデックス
*                               フリーリストが空であれば IndexEnd
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          outIndex が NULL ではない。
*
* @details      フリーリストの先頭インデックスを取得します。
*/
Result AllocationTable::ReadFreeListHead(uint32_t* outIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outIndex);

    TableElement freelist;
    NN_RESULT_DO(ReadBlock(&freelist, SectorFreeListHeader, 1));

    if( !freelist.IsFreeListEmpty() )
    {
        *outIndex = ConvertSectorToIndex(freelist.sectorNext);
    }
    else
    {
        // フリーリストが空であれば IndexEnd を返します。
        *outIndex = IndexEnd;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        テーブルの空きブロック数を取得します。
*
* @param[out]   outCount        テーブルの空きブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          outCount が NULL ではない。
*
* @details      テーブルの空きブロック数を取得します。
*/
Result AllocationTable::CalcFreeListLength(uint32_t* outCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    uint32_t count = 0;

    uint32_t index = 0;
    Result result = ReadFreeListHead(&index);

    if( !result.IsFailure() )
    {
        while( index != IndexEnd )
        {
            uint32_t nextIndex;
            uint32_t blockCount;
            NN_RESULT_DO(ReadNext(&nextIndex, &blockCount, index));
            count += blockCount;
            index = nextIndex;
        }
    }
    *outCount = count;

    return result;
}

/**
* @brief        インデックスをセクタ番号に変換します。
*
* @param[in]    index   インデックス
*
* @return       セクタ番号。
*
* @pre          インデックスが範囲外を指していない。
*
* @details      インデックスをセクタ番号に変換します。
*/
uint32_t AllocationTable::ConvertIndexToSector(uint32_t index) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(index + SectorReservedCount, m_BlockCount);
    return ( index + SectorReservedCount );
}

/**
* @brief        セクタ番号をインデックスに変換します。
*
* @param[in]    sector  セクタ番号
*
* @return       インデックス。
*
* @pre          セクタ番号が範囲外を指していない。
*
* @details      セクタ番号をインデックスに変換します。
*/
uint32_t AllocationTable::ConvertSectorToIndex(uint32_t sector) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(SectorReservedCount, sector);
    NN_SDK_REQUIRES_EQUAL(0U, sector & ~SectorMaskValid);
    NN_SDK_REQUIRES_GREATER_EQUAL(m_BlockCount, sector);
    //sector &= SectorMaskValid;
    return ( sector - SectorReservedCount );
}

/**
* @brief        ブロック情報を読み込む際の個数を取得します。
*
* @param[in]    sector  セクタ番号
*
* @return       読み込むべきブロックの個数。
*
* @details      読み込むブロックが連続ブロックの場合、次ブロックの情報が
*               必要になるため、基本的に 2 個読み込む。
*               最終ブロックの場合は次ブロックが無いため 1 個。
*/
uint32_t AllocationTable::GetReadBlockCount(uint32_t sector) const NN_NOEXCEPT
{
    return (sector == m_BlockCount) ? 1 : 2;
}

/**
* @brief        指定したセクタ以降のブロック複数個ストレージから読み込みます。
*
* @param[out]   outDst          読み込んだブロック
* @param[in]    sector          セクタ
* @param[in]    blockCount      ブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常にデータを読み込みました。
* @retval       上記以外        ストレージからのデータ読み込みに失敗しました
*
* @pre          マウントしている。
* @pre          outDst が NULL ではない。
* @pre          blockCount が 0 ではない。
*
* @details      指定したセクタ以降のブロック複数個ストレージから読み込みます。
*/
Result AllocationTable::ReadBlock(
           TableElement* outDst,
           uint32_t sector,
           uint32_t blockCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outDst);
    NN_SDK_REQUIRES_LESS(0U, blockCount);

    return m_Table.Read(
               (sector & SectorMaskValid) * sizeof(TableElement),
               outDst,
               blockCount * sizeof(TableElement)
           );
}

/**
* @brief        指定したセクタに書き込みます。
*
* @param[in]    pSrc            書き込むブロック
* @param[in]    sector          セクタ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常にデータを書き込めました。
* @retval       上記以外        ストレージへのデータ書きこみに失敗しました
*
* @pre          マウントしている。
* @pre          pSrc が NULL ではない。
*
* @details      指定したセクタに書き込みます。
*/
Result AllocationTable::WriteBlock(
           const TableElement* pSrc,
           uint32_t sector
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    return m_Table.Write(
               (sector & SectorMaskValid) * sizeof(TableElement),
               pSrc,
               sizeof(TableElement)
           );
}

/**
* @brief        指定したブロックの次セクタを更新します。
*
* @param[in]    sector          更新するセクタ
* @param[in]    nextSector      次連続領域の先頭セクタ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常にデータを書き込みました。
* @retval       上記以外        ストレージへのデータ書きこみに失敗しました。
*
* @pre          マウントしている。
*
* @details      指定したブロックの次セクタを更新します。
*/
Result AllocationTable::UpdateNext(
           uint32_t sector,
           uint32_t nextSector
       ) NN_NOEXCEPT
{
    uint32_t validSector = sector & SectorMaskValid;
    NN_SDK_ASSERT_NOT_EQUAL(SectorFreeListHeader, validSector);

    TableElement block;
    NN_RESULT_DO(ReadBlock(&block, validSector, 1));
    if( block.IsListHead() )
    {
        block.SetListHead(nextSector, block.IsChainParent());
    }
    else
    {
        block.SetChainParent(block.sectorPrevious, nextSector, block.IsChainParent());
    }
    return WriteBlock(&block, validSector);
}

/**
* @brief        指定したブロックの前セクタを更新します。
*
* @param[in]    sector          更新するセクタ
* @param[in]    sectorPrevious  前連続領域の先頭セクタ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に処理が終了しました。
* @retval       上記以外        ストレージでのデータ読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          sector がフリーリストの先頭ではない。
*
* @details      指定したブロックの前セクタを更新します。
*/
Result AllocationTable::UpdatePrevious(
           uint32_t sector,
           uint32_t sectorPrevious
       ) NN_NOEXCEPT
{
    uint32_t validSector = sector & SectorMaskValid;
    NN_SDK_REQUIRES_NOT_EQUAL(SectorFreeListHeader, validSector);

    TableElement block;
    NN_RESULT_DO(ReadBlock(&block, validSector, 1));

    block.sectorPrevious = sectorPrevious;

    return WriteBlock(&block, validSector);
}

/**
* @brief        連続領域内のリンク情報を更新します。
*
* @param[in]    headSector      ブロックリストの先頭セクタ
* @param[in]    parentSector    連続ブロックの先頭セクタ
* @param[in]    lastSector      連続ブロックの末尾セクタ
* @param[in]    previousSector  前の連続ブロックの先頭セクタ
* @param[in]    nextSector      次の連続ブロックの先頭セクタ
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   リンク情報を正常に更新しました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
*
* @details      parentSector から始まる連続ブロックを
*               headSector から始まる連続ブロックにつなげます。
*/
Result AllocationTable::UpdateChain(
           uint32_t headSector,
           uint32_t parentSector,
           uint32_t lastSector,
           uint32_t previousSector,
           uint32_t nextSector
       ) NN_NOEXCEPT
{
    TableElement block;
    if( parentSector == headSector )
    {
        // リストの先頭として初期化します。
        bool hasChild = ((parentSector != lastSector) && (lastSector != SectorEmpty));
        block.SetListHead(nextSector, hasChild);
    }
    else
    {
        // 連続ブロックの先頭として初期化します。
        bool hasChild = ((parentSector != lastSector) && (lastSector != SectorEmpty));
        block.SetChainParent(previousSector, nextSector, hasChild);
    }

    // 先頭ブロック情報を書き込みます。
    NN_RESULT_DO(WriteBlock(&block, parentSector));

    // 次領域の先頭ブロックの情報を更新します。
    if( (nextSector & SectorMaskValid ) != SectorFreeListHeader )
    {
        NN_RESULT_DO(UpdatePrevious(nextSector & SectorMaskValid, parentSector));
    }

    // 連続領域の子ブロックを書き込みます。
    if( (lastSector != SectorEmpty) && (parentSector != lastSector) )
    {
        NN_SDK_ASSERT_LESS(parentSector, lastSector);

        block.SetChainChild(parentSector, lastSector);

        // 連続領域の 2 番目と末尾のブロックにのみ書き込みます。
        NN_RESULT_DO(WriteBlock(&block, parentSector + 1));
        if( parentSector + 1 != lastSector )
        {
            NN_RESULT_DO(WriteBlock(&block, lastSector));
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        フリーリストを取得します。
*
* @param[out]   outFreeList     フリーリスト
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   フリーリストが取得できました。
* @retval       上記以外        ストレージからの読み込みに失敗しました。
*
* @pre          マウントしている。
* @pre          outFreeList が NULL ではない。
*
* @details      フリーリストを取得します。
*/
Result AllocationTable::ReadFreeList(TableElement* outFreeList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outFreeList);

    NN_RESULT_DO(ReadBlock(outFreeList, SectorFreeListHeader, 1));

    NN_RESULT_SUCCESS;
}

/**
* @brief        フリーリストから空いているデータ領域を取得します。
*
* @param[out]   outSector       確保した領域の先頭セクタ
* @param[out]   outBlockCount   確保できたブロック数
* @param[in]    pFreeListHeader フリーリスト
* @param[in]    requiredCount   確保するブロック数
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常に取得できました。
* @retval       上記以外        ストレージでの読み書きに失敗しました。
*
* @pre          マウントしている。
* @pre          outSector が NULL ではない。
* @pre          outBlockCount が NULL ではない。
* @pre          pFreeListHeader が NULL ではない。
*
* @details      フリーリストから空いているデータ領域を取得します。
*/
Result AllocationTable::AllocateFreeSector(
           uint32_t* outSector,
           uint32_t* outBlockCount,
           TableElement* pFreeListHeader,
           uint32_t requiredCount
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSector);
    NN_SDK_REQUIRES_NOT_NULL(outBlockCount);
    NN_SDK_REQUIRES_NOT_NULL(pFreeListHeader);

    // フリーリストの先頭から 2 ブロック取得します。
    // (連続していた時にすぐブロック数を求められるようにしておきます)
    TableElement blocks[2];
    uint32_t readCount = GetReadBlockCount(pFreeListHeader->sectorNext);

    NN_RESULT_DO(ReadBlock(blocks, pFreeListHeader->sectorNext, readCount));

    *outSector = pFreeListHeader->sectorNext;

    if( !blocks[0].IsChainParent() )
    {
        // 連続フリー領域でなければ 1 ブロックだけ取得します。
        *outBlockCount = 1;
        pFreeListHeader->sectorNext = blocks[0].sectorNext;

        // 次領域の前をセクタフリーリストに付け替えます。
        if( !pFreeListHeader->IsFreeListEmpty() )
        {
            NN_RESULT_DO(
                UpdatePrevious(
                    pFreeListHeader->sectorNext,
                    SectorMaskPreviousIsListHead
                )
            );
        }
    }
    else
    {
        NN_SDK_ASSERT_EQUAL(2U, readCount);
        NN_SDK_ASSERT(blocks[1].IsChainChild());

        // 連続フリー領域であれば、連続したブロックを取得します。
        uint32_t count = blocks[1].sectorNext - pFreeListHeader->sectorNext + 1;
        if( count > requiredCount )
        {
            *outBlockCount = requiredCount;

            // 抽出後の連続領域の先頭、末尾ブロックのリンクを更新します。
            NN_RESULT_DO(
                UpdateChain(
                    pFreeListHeader->sectorNext + requiredCount,    // headSector
                    pFreeListHeader->sectorNext + requiredCount,    // parentSector
                    blocks[1].sectorNext,                           // lastSector
                    blocks[0].sectorPrevious,                       // prevSector
                    ( blocks[0].sectorNext & SectorMaskValid )      // nextSector
                )
            );

            // フリーリストの先頭位置を抽出した分ずらします。
            pFreeListHeader->sectorNext += requiredCount;
        }
        else
        {
            // 連続領域をすべて割り当てます。
            *outBlockCount = count;

            // 次の領域をフリーリストの先頭に割り当てます。
            *outSector = pFreeListHeader->sectorNext;
            pFreeListHeader->sectorNext = ( blocks[0].sectorNext & SectorMaskValid );

            // 次領域の前セクタをフリーリストに付け替えます。
            if( pFreeListHeader->sectorNext != SectorFreeListHeader )
            {
                NN_RESULT_DO(
                    UpdatePrevious(
                        pFreeListHeader->sectorNext,
                        SectorMaskPreviousIsListHead
                    )
                );
            }
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        更新後のフリーリストを保存します。
*
* @param[in]    pFreeListHeader フリーリスト
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess   正常にフリーリストを更新しました。
* @retval       上記以外        ストレージへの書き込みに失敗しました。
*
* @pre          マウントしている。
* @pre          pFreeListHeader が NULL ではない。
*
* @details      更新後のフリーリストを保存します。
*/
Result AllocationTable::UpdateFreeList(const TableElement* pFreeList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFreeList);

    return WriteBlock(pFreeList, SectorFreeListHeader);
}

}}}

