﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Allocator.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/fscreator/fssrv_StorageOnNcaCreator.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include "../detail/fssrv_RuntimeConfiguration.h"

using namespace nn::fs;

namespace {
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include "fssrv_AcidSignKey-hardware.nx.inc"
#else   // if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    #include "fssrv_AcidSignKey-hardware.default.inc"
#endif  // if defined(NN_BUILD_CONFIG_HARDWARE_NX) else
}

namespace nn { namespace fssrv { namespace fscreator {


    StorageOnNcaCreator::StorageOnNcaCreator(nn::MemoryResource* pAllocator, const fssystem::NcaCryptoConfiguration& cryptoConfig, bool isProd, nn::fssystem::IBufferManager* pBufferManager) NN_NOEXCEPT
        : m_Allocator(pAllocator)
        , m_CryptoConfig(cryptoConfig)
        , m_IsProd(isProd)
        , m_pBufferManager(pBufferManager)

    {
    }

    Result StorageOnNcaCreator::VerifyAcid(fs::fsa::IFileSystem* pFileSystem, fssystem::NcaReader* ncaReader) NN_NOEXCEPT
    {
        std::unique_ptr<fsa::IFile> file;
        const char NpdmFilePath[] = "/main.npdm";
        NN_RESULT_DO(pFileSystem->OpenFile(&file, NpdmFilePath, OpenMode_Read));

        size_t size;

        // npdm から ACID の読み込み
        const int64_t AcidOffsetOffset = 0x78;
        const int64_t AcidSizeOffset   = AcidOffsetOffset + 4;
        int32_t acidOffset;
        NN_RESULT_DO(file->Read(&size, AcidOffsetOffset, &acidOffset, sizeof(acidOffset), ReadOption()));
        NN_RESULT_THROW_UNLESS(size == sizeof(acidOffset), ResultInvalidAcidFileSize());

        int32_t acidSize;
        NN_RESULT_DO(file->Read(&size, AcidSizeOffset, &acidSize, sizeof(acidSize), ReadOption()));
        NN_RESULT_THROW_UNLESS(size == sizeof(acidSize), ResultInvalidAcidFileSize());

        char* pAcid = static_cast<char*>(m_Allocator->allocate(acidSize));
        NN_RESULT_THROW_UNLESS(pAcid != nullptr, ResultAllocationMemoryFailedInStorageOnNcaCreatorA());
        NN_UTIL_SCOPE_EXIT
        {
            m_Allocator->deallocate(pAcid, acidSize);
        };

        NN_RESULT_DO(file->Read(&size, acidOffset, pAcid, acidSize, ReadOption()));
        NN_RESULT_THROW_UNLESS(size == static_cast<size_t>(acidSize), ResultInvalidAcidSize());

        const int AcidSignOffset             = 0x0;
        const int AcidSignSize               = 0x100;
        const int HeaderSign2KeyOffset       = 0x100;
        const int HeaderSign2KeySize         = 0x100;
        const int AcidSignTargetOffset       = 0x100;
        const int AcidSignTargetSizeOffset   = 0x204;

        NN_RESULT_THROW_UNLESS(acidSize >= static_cast<int32_t>(AcidSignTargetSizeOffset + sizeof(int32_t))
                               , ResultInvalidAcidSize());
        int32_t acidSignTargetSize = *reinterpret_cast<int32_t*>(pAcid + AcidSignTargetSizeOffset);
        NN_RESULT_THROW_UNLESS(acidSize >= static_cast<int32_t>(acidSignTargetSize + sizeof(int32_t)), ResultInvalidAcidSize());
        NN_RESULT_THROW_UNLESS(acidSize >= AcidSignTargetOffset + acidSignTargetSize, ResultInvalidAcidSize());

        const unsigned char* const pAcidSignKeyModulus = m_IsProd ? AcidSignKeyModulusProd
                                                                  : AcidSignKeyModulusDev;
        const size_t AcidSignKeyModulusSize = 0x100;

        // ACID の検証
        bool acidSignatureVerified = crypto::VerifyRsa2048PssSha256(
            pAcid + AcidSignOffset, AcidSignSize,
            pAcidSignKeyModulus, AcidSignKeyModulusSize,
            AcidSignKeyPublicExponent, sizeof(AcidSignKeyPublicExponent),
            pAcid + AcidSignTargetOffset, acidSignTargetSize);
        if( !acidSignatureVerified )
        {
            NN_SDK_LOG("[fs] Error %s: Failed ACID signature verification.\n", nn::fssrv::detail::GetEnabledProgramVerification() ? "" : "(ignored)");
            // 暫定実装
            // nn::fs::GetEnabledProgramVerification() == false のときは、検証エラーを無視して以降の検証をスキップする。
            if( nn::fssrv::detail::GetEnabledProgramVerification() )
            {
                return nn::fs::ResultAcidVerificationFailed();
            }
            else
            {
                NN_RESULT_SUCCESS;
            }
        }

        if (ncaReader)
        {
            NN_RESULT_THROW_UNLESS(acidSize >= HeaderSign2KeyOffset + HeaderSign2KeySize, ResultInvalidAcidSize());
            // ゼロ鍵の場合 crypto::VerifyRsa2048PssSha256() 内の Assert に引っかかってしまう対策
            NN_RESULT_THROW_UNLESS(*(pAcid + HeaderSign2KeyOffset + HeaderSign2KeySize - 1) != 0x00, ResultInvalidAcid());

            // ヘッダ署名 2 による nca header の検証
            return ncaReader->VerifyHeaderSign2(pAcid + HeaderSign2KeyOffset, HeaderSign2KeySize);
        }

        NN_RESULT_SUCCESS;
    }

    Result StorageOnNcaCreator::VerifyNcaHeaderSign2(fssystem::NcaReader* ncaReader, fs::IStorage* storage) NN_NOEXCEPT
    {
        fssystem::PartitionFileSystem partFs;
        NN_RESULT_DO(partFs.Initialize(storage));
        return VerifyAcid(&partFs, ncaReader);
    }

    Result StorageOnNcaCreator::Create(std::shared_ptr<fs::IStorage>* outValue, fssystem::NcaFsHeaderReader *outFsHeaderReader, std::shared_ptr<fssystem::NcaReader> ncaReader, int index, bool verifyHeaderSign2) NN_NOEXCEPT
    {
        fssystem::NcaFileSystemDriver ncaFsDriver(ncaReader, m_Allocator, m_pBufferManager);

        std::shared_ptr<fs::IStorage> storage;
        NN_RESULT_DO(ncaFsDriver.OpenStorage(&storage, outFsHeaderReader, index));

        // ヘッダ署名2の検証
        if( verifyHeaderSign2 )
        {
            NN_RESULT_DO(VerifyNcaHeaderSign2(ncaReader.get(), storage.get()));
        }

        *outValue = std::move(storage);

        NN_RESULT_SUCCESS;
    }

    Result StorageOnNcaCreator::CreateWithPatch(
        std::shared_ptr<fs::IStorage>* outValue,
        fssystem::NcaFsHeaderReader* outFsHeaderReader,
        std::shared_ptr<fssystem::NcaReader> originalNcaReader,
        std::shared_ptr<fssystem::NcaReader> currentNcaReader,
        int index,
        bool verifyHeaderSign2) NN_NOEXCEPT
    {
        fssystem::NcaFileSystemDriver ncaFsDriver(
            originalNcaReader,
            currentNcaReader,
            m_Allocator,
            m_pBufferManager);

        std::shared_ptr<fs::IStorage> storage;
        NN_RESULT_DO(ncaFsDriver.OpenStorage(
            &storage,
            outFsHeaderReader,
            index));

        // ヘッダ署名2の検証
        if( verifyHeaderSign2 )
        {
            NN_RESULT_DO(VerifyNcaHeaderSign2(currentNcaReader.get(), storage.get()));
        }

        *outValue = std::move(storage);

        NN_RESULT_SUCCESS;
    }

    Result StorageOnNcaCreator::CreateNcaReader(std::shared_ptr<fssystem::NcaReader>* outValue, std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT
    {
        std::shared_ptr<fssystem::NcaReader> ncaReader = fssystem::AllocateShared<fssystem::NcaReader>();
        NN_RESULT_THROW_UNLESS(ncaReader, ResultAllocationMemoryFailedInStorageOnNcaCreatorB());
        NN_RESULT_DO(ncaReader->Initialize(std::move(storage), m_CryptoConfig));

        *outValue = std::move(ncaReader);
        NN_RESULT_SUCCESS;
    }

}}}
