﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>

#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageCreator.h>
#include <nn/fssystem/fs_Assert.h>
#include "../detail/fssrv_SdmmcStorageService.h"
#include "../detail/fssrv_BuiltInStorageService.h"

using namespace nn::fssrv::detail;
using namespace nn::fssystem;
using namespace nn::fs;


namespace nn { namespace fssrv { namespace fscreator {

namespace {

const int MmcAccessBlockSize = 512;
const size_t EncryptionBlockSize = 16 * 1024;

// TODO: プラットフォーム固有の設定値として、外部で管理する
// TODO: InitialImage と共有
const Guid BisGuids[] =
{
#include "../detail/fssrv_BuiltInStorageServiceGuidConfig.h"
};


const int BisGptPartitionCount = sizeof(BisGuids) / sizeof(BisGuids[0]);

const Guid* GetBisPartitionGuid(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    static_assert(sizeof(BisGuids) / sizeof(BisGuids[0]) == (static_cast<size_t>(nn::fs::BisPartitionId::System) - static_cast<size_t>(nn::fs::BisPartitionId::UserDataRoot)), "BisGuids[] is invalid.");

    switch(id)
    {
    case nn::fs::BisPartitionId::CalibrationBinary:
        return &BisGuids[0];
    case nn::fs::BisPartitionId::CalibrationFile:
        return &BisGuids[1];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
        return &BisGuids[2];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
        return &BisGuids[3];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
        return &BisGuids[4];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
        return &BisGuids[5];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
        return &BisGuids[6];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
        return &BisGuids[7];
    case nn::fs::BisPartitionId::SafeMode:
    case nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode:
        return &BisGuids[8];
    case nn::fs::BisPartitionId::User:
        return &BisGuids[9];
    case nn::fs::BisPartitionId::System:
    case nn::fs::BisPartitionId::SystemProperEncryption:
    case nn::fs::BisPartitionId::SystemProperPartition:
        return &BisGuids[10];
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool IsValidId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch( id )
    {
    case nn::fs::BisPartitionId::CalibrationBinary:
    case nn::fs::BisPartitionId::CalibrationFile:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
    case nn::fs::BisPartitionId::SafeMode:
    case nn::fs::BisPartitionId::User:
    case nn::fs::BisPartitionId::System:
    case nn::fs::BisPartitionId::SystemProperEncryption:
    case nn::fs::BisPartitionId::SystemProperPartition:
    case nn::fs::BisPartitionId::BootPartition1Root:
    case nn::fs::BisPartitionId::BootPartition2Root:
    case nn::fs::BisPartitionId::UserDataRoot:
    case nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode:
        return true;
    default:
        return false;
    }
}

MmcPartition GetMmcPartitionId(fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
    case fs::BisPartitionId::BootPartition1Root:
        return MmcPartition::BootPartition1;
    case fs::BisPartitionId::BootPartition2Root:
        return MmcPartition::BootPartition2;
    case fs::BisPartitionId::UserDataRoot:
        return MmcPartition::UserData;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool IsMmcRootPartition(fs::BisPartitionId id) NN_NOEXCEPT
{
    return  id == fs::BisPartitionId::BootPartition1Root ||
            id == fs::BisPartitionId::BootPartition2Root ||
            id == fs::BisPartitionId::UserDataRoot;
}

bool IsEncryptedPartition(fs::BisPartitionId id) NN_NOEXCEPT
{
    return  id == fs::BisPartitionId::CalibrationBinary ||
            id == fs::BisPartitionId::CalibrationFile   ||
            id == fs::BisPartitionId::SafeMode ||
            id == fs::BisPartitionId::User     ||
            id == fs::BisPartitionId::System   ||
            id == fs::BisPartitionId::SystemProperEncryption ||
            id == fs::BisPartitionId::SystemProperPartition ||
            id == fs::BisPartitionId::SignedSystemPartitionOnSafeMode;
}


struct CryptoResource
{
    BuiltInStorageCreator::Key key[2];
    unsigned char iv[AesXtsStorage::IvSize];
};
NN_STATIC_ASSERT(std::is_pod<CryptoResource>::value);

CryptoResource GetCryptoResource(nn::fs::BisPartitionId id, BuiltInStorageCreator::GetEncryptionKeyFunction pFunc, int keyGeneration) NN_NOEXCEPT
{
    CryptoResource resource;
    memset(&resource, 0x00, sizeof(resource));

    // pFunc != nullptr は呼び出し側で保証します。
    NN_ABORT_UNLESS_NOT_NULL(pFunc);

    switch(id)
    {
    case nn::fs::BisPartitionId::CalibrationBinary:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::Calibration, keyGeneration);
        return resource;

    case nn::fs::BisPartitionId::CalibrationFile:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::Calibration, keyGeneration);
        return resource;

    case nn::fs::BisPartitionId::SafeMode:
    case nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::SafeMode, keyGeneration);
        return resource;

    case nn::fs::BisPartitionId::User:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::UserSystemProperEncryption, keyGeneration);
        return resource;

    case nn::fs::BisPartitionId::System:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::UserSystem, keyGeneration);
        return resource;

    case nn::fs::BisPartitionId::SystemProperEncryption:
    case nn::fs::BisPartitionId::SystemProperPartition:
        pFunc(resource.key, 2, BisPartitionEncryptionKeyId::UserSystemProperEncryption, keyGeneration);
        return resource;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

}


class ProxyStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(ProxyStorage);

public:

    explicit ProxyStorage(IStorage* pBaseStorage)
        : m_pBaseStorage(pBaseStorage),
          m_Valid(true)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("Read : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("Write : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("Flush : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->Flush();
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("SetSize : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->SetSize(size);
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("GetSize : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->GetSize(outValue);
    }

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !m_Valid )
        {
            NN_SDK_LOG("OperateRange : Filesystem is Invalidation.");
            return nn::fs::ResultBisProxyInvalidated();
        }
        return m_pBaseStorage->OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

    void Invalidate() NN_NOEXCEPT
    {
        m_Valid = false;
    }

private:
    IStorage* const m_pBaseStorage;
    bool m_Valid;
};

Result AddProxyStorageLayer(std::unique_ptr<IStorage>* outValue, std::unique_ptr<IStorage>&& pBaseStorage) NN_NOEXCEPT
{
    std::unique_ptr<IStorage> pProxyStorage(new ProxyStorage(pBaseStorage.get()));
    if( pProxyStorage == nullptr )
    {
        return fs::ResultAllocationMemoryFailedInBuiltInStorageCreatorA();
    }

    pBaseStorage.release();

    *outValue = std::move(pProxyStorage);
    NN_RESULT_SUCCESS;
}

} // namespace

Result BuiltInStorageCreator::AddEncryptionLayer(std::unique_ptr<IStorage>* outValue, std::unique_ptr<IStorage>&& pBaseStorage, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    if( IsEncryptedPartition(id) )
    {
        // m_Config.getKeyFunc が設定されていないならプログラミングミスなので ABORT する
        NN_ABORT_UNLESS(m_Config.getKeyFunc != nullptr);

        const int DefaultKeyGeneration = -1;
        auto resource = GetCryptoResource(id, m_Config.getKeyFunc, DefaultKeyGeneration);
        std::unique_ptr<IStorage> pEncryptionStorage(new AesXtsStorage(pBaseStorage.get(), resource.key[0].value, resource.key[1].value, AesXtsStorage::KeySize, resource.iv, AesXtsStorage::IvSize, EncryptionBlockSize));
        NN_RESULT_THROW_UNLESS(pEncryptionStorage != nullptr, ResultAllocationMemoryFailedInBuiltInStorageCreatorB());

        std::unique_ptr<IStorage> pAlignmentStorage(new AlignmentMatchingStorageInBulkRead<1>(pEncryptionStorage.get(), MmcAccessBlockSize));
        NN_RESULT_THROW_UNLESS(pAlignmentStorage != nullptr, ResultAllocationMemoryFailedInBuiltInStorageCreatorC());

        // TODO: 下位レイヤのリソース管理
        pBaseStorage.release();
        pEncryptionStorage.release();

        *outValue = std::move(pAlignmentStorage);
        NN_RESULT_SUCCESS;
    }
    else
    {
        *outValue = std::move(pBaseStorage);
        NN_RESULT_SUCCESS;
    }
}


IStorage** BuiltInStorageCreator::GetCachedPartitionStorage(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch(id)
    {
    case nn::fs::BisPartitionId::CalibrationBinary:
        return &m_CachedBisPartitions[0];
    case nn::fs::BisPartitionId::CalibrationFile:
        return &m_CachedBisPartitions[1];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part1:
        return &m_CachedBisPartitions[2];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part2:
        return &m_CachedBisPartitions[3];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part3:
        return &m_CachedBisPartitions[4];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part4:
        return &m_CachedBisPartitions[5];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part5:
        return &m_CachedBisPartitions[6];
    case nn::fs::BisPartitionId::BootConfigAndPackage2Part6:
        return &m_CachedBisPartitions[7];
    case nn::fs::BisPartitionId::SafeMode:
        return &m_CachedBisPartitions[8];
    case nn::fs::BisPartitionId::User:
        return &m_CachedBisPartitions[9];
    case nn::fs::BisPartitionId::System:
        return &m_CachedBisPartitions[10];
    case nn::fs::BisPartitionId::SystemProperEncryption:
    case nn::fs::BisPartitionId::SystemProperPartition:
        return &m_CachedBisPartitions[11];
    case nn::fs::BisPartitionId::BootPartition1Root:
        return &m_CachedBisPartitions[12];
    case nn::fs::BisPartitionId::BootPartition2Root:
        return &m_CachedBisPartitions[13];
    case nn::fs::BisPartitionId::UserDataRoot:
        return &m_CachedBisPartitions[14];
    case nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode:
        return &m_CachedBisPartitions[15];

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void BuiltInStorageCreator::InvalidateCachePartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsMmcRootPartition(id));

    IStorage** ppCachedStorage = GetCachedPartitionStorage(id);
    ProxyStorage* pProxyStorage = static_cast<ProxyStorage*>(*ppCachedStorage);

    if( pProxyStorage != nullptr )
    {
        pProxyStorage->Invalidate();
        *ppCachedStorage = nullptr;
    }
}

Result BuiltInStorageCreator::InvalidateCache() NN_NOEXCEPT
{
    InvalidateCachePartition(nn::fs::BisPartitionId::CalibrationBinary);
    InvalidateCachePartition(nn::fs::BisPartitionId::CalibrationFile);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part1);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part2);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part3);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part4);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part5);
    InvalidateCachePartition(nn::fs::BisPartitionId::BootConfigAndPackage2Part6);
    InvalidateCachePartition(nn::fs::BisPartitionId::SafeMode);
    InvalidateCachePartition(nn::fs::BisPartitionId::User);
    InvalidateCachePartition(nn::fs::BisPartitionId::System);
    InvalidateCachePartition(nn::fs::BisPartitionId::SystemProperEncryption);
    InvalidateCachePartition(nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode);
    NN_RESULT_SUCCESS;
}

struct SignedSystemPartitionHeader
{
    char signature[256];
    union {
        char contentHeaderRaw[256];
        struct {
            uint64_t version;
            uint64_t size;
            char     iv[16];
            char     hash[32];
            uint64_t keyGeneration;
            uint64_t appendHashCount;
            uint64_t appendSize;
            char     appendHash[32];

            // TODO: append[n]
        } contentHeader;
    };
};
NN_STATIC_ASSERT(sizeof(SignedSystemPartitionHeader) == 512);

Result BuiltInStorageCreator::LoadAndVerifySignedSystemPartition(std::unique_ptr<IStorage>* outValue, std::unique_ptr<IStorage>&& baseStorage) NN_NOEXCEPT
{
    const int HeaderSize = sizeof(SignedSystemPartitionHeader);

    std::shared_ptr<IStorage> signedSystemPartitionRawStorage;
    char* signedSystemPartitionRawBuffer;
    NN_RESULT_DO(m_Config.pMemoryStorageCreator->Create(&signedSystemPartitionRawStorage, &signedSystemPartitionRawBuffer, IMemoryStorageCreator::MemoryStorageId::SignedSystemPartitionRaw));

    int64_t memoryStorageSize;
    NN_RESULT_DO(signedSystemPartitionRawStorage->GetSize(&memoryStorageSize));

    SignedSystemPartitionHeader header;
    NN_RESULT_DO(baseStorage->Read(0, &header, sizeof(header)));

    const char PublicExponent[] = { 0x01, 0x00, 0x01 };
    NN_SDK_ASSERT(m_Config.pSignedSystemPartitionSignKeyPublicModulus != nullptr);

    NN_RESULT_THROW_UNLESS(
        crypto::VerifyRsa2048PssSha256(
            header.signature, sizeof(header.signature),
            m_Config.pSignedSystemPartitionSignKeyPublicModulus, m_Config.SignedSystemPartitionSignKeyPublicModulusSize,
            PublicExponent, sizeof(PublicExponent),
            &header.contentHeader, sizeof(header.contentHeaderRaw)
        ),
        fs::ResultSignedSystemPartitionSignatureVerificationFailed()
    );

    NN_RESULT_THROW_UNLESS(static_cast<int64_t>(header.contentHeader.size) <= memoryStorageSize, nn::fs::ResultSignedSystemPartitionInvalidSize());
    NN_RESULT_THROW_UNLESS(util::is_aligned(header.contentHeader.size, 512), nn::fs::ResultSignedSystemPartitionInvalidSize());

    size_t readSize = static_cast<size_t>(util::align_up(HeaderSize + header.contentHeader.size, 512));
    NN_RESULT_DO(baseStorage->Read(0, signedSystemPartitionRawBuffer, readSize));

    if (m_Config.enablePackage2HashVerification)
    {
        // enablePackage2HashVerification が有効な fs (initialize, 修理)では p2 ハッシュが埋め込まれた SignedSystemPartition のみ許す
        NN_RESULT_THROW_UNLESS(header.contentHeader.appendHashCount == 1, nn::fs::ResultSignedSystemPartitionInvalidAppendHashCount());

        NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(header.contentHeader.appendHash, m_Config.package2Hash, sizeof(header.contentHeader.appendHash)), fs::ResultSignedSystemPartitionPackage2HashVerificationFailed());
    }

    char hash[crypto::Sha256Generator::HashSize];
    crypto::GenerateSha256Hash(hash, sizeof(hash), signedSystemPartitionRawBuffer + HeaderSize, static_cast<size_t>(header.contentHeader.size));
    NN_RESULT_THROW_UNLESS(crypto::IsSameBytes(hash, header.contentHeader.hash, sizeof(hash)), fs::ResultSignedSystemPartitionHashVerificationFailed());

    std::unique_ptr<IStorage> memoryStorage(new SubStorage(std::move(signedSystemPartitionRawStorage), HeaderSize, static_cast<size_t>(header.contentHeader.size)));
    NN_RESULT_THROW_UNLESS(memoryStorage != nullptr, fs::ResultAllocationMemoryFailedNew());

    NN_ABORT_UNLESS(m_Config.getKeyFunc != nullptr);
    auto resource = GetCryptoResource(BisPartitionId::System, m_Config.getKeyFunc, static_cast<int>(header.contentHeader.keyGeneration));

    std::unique_ptr<IStorage> pEncryptionStorage(new AesCtrStorage(memoryStorage.get(), resource.key[0].value, AesCtrStorage::KeySize, header.contentHeader.iv, AesCtrStorage::IvSize));
    NN_RESULT_THROW_UNLESS(pEncryptionStorage != nullptr, ResultAllocationMemoryFailedInBuiltInStorageCreatorB());

    std::unique_ptr<IStorage> pAlignmentStorage(new AlignmentMatchingStorageInBulkRead<1>(pEncryptionStorage.get(), MmcAccessBlockSize));
    NN_RESULT_THROW_UNLESS(pAlignmentStorage != nullptr, ResultAllocationMemoryFailedInBuiltInStorageCreatorC());

    // TODO: 下位レイヤのリソース管理
    memoryStorage.release();
    pEncryptionStorage.release();

    *outValue = std::move(pAlignmentStorage);
    NN_RESULT_SUCCESS;
}


/**
    @brief      id に対応するストレージを一度だけ生成し、以降は生成済みのものを返す

    @details    破棄はしない。そのため、暫定で下位レイヤーのリソース管理はしない
*/
Result BuiltInStorageCreator::OpenAndCacheBuiltInStoragePartitionInternal(IStorage** outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsValidId(id), ResultInvalidArgument());

    IStorage** ppCachedStorage = GetCachedPartitionStorage(id);
    if(*ppCachedStorage != nullptr)
    {
        // 既存であればそれを返す
        *outValue = *ppCachedStorage;
        NN_RESULT_SUCCESS;
    }
    else
    {
        // 既存でなければ一度だけ生成する

        if (IsMmcRootPartition(id))
        {
            IStorage* pMmcRootStorage;
            NN_RESULT_DO(OpenMmcStorage(&pMmcRootStorage, GetMmcPartitionId(id)));

            *ppCachedStorage = pMmcRootStorage;
            *outValue = pMmcRootStorage;
            NN_RESULT_SUCCESS;
        }
        else
        {
            // ユーザデータ領域の GPT パーティション

            IStorage* pMmcUserDataRootStorage;
            NN_RESULT_DO(OpenAndCacheBuiltInStoragePartitionInternal(&pMmcUserDataRootStorage, nn::fs::BisPartitionId::UserDataRoot));

            const Guid* pTargetGuid = GetBisPartitionGuid(id);
            std::unique_ptr<IStorage> pGptPartitionStorage;
            NN_RESULT_DO(OpenGptPartition(&pGptPartitionStorage, pMmcUserDataRootStorage, pTargetGuid));

            std::unique_ptr<IStorage> pEncryptionStorage;

            if (IsSignedSystemPartition(id))
            {
                // SignedSystemPartition の場合はメモリ上に読み込んで検証
                std::unique_ptr<IStorage> pSignedSystemPartitionMemoryStorage;
                NN_RESULT_DO(LoadAndVerifySignedSystemPartition(&pSignedSystemPartitionMemoryStorage, std::move(pGptPartitionStorage)));
                pEncryptionStorage = std::move(pSignedSystemPartitionMemoryStorage);
            }
            else
            {
                NN_RESULT_DO(AddEncryptionLayer(&pEncryptionStorage, std::move(pGptPartitionStorage), id));
            }

            std::unique_ptr<IStorage> pProxyStorage;
            NN_RESULT_DO(AddProxyStorageLayer(&pProxyStorage, std::move(pEncryptionStorage)));

            *ppCachedStorage = pProxyStorage.release();
            *outValue = *ppCachedStorage;
            NN_RESULT_SUCCESS;

        }
    }
}


Result BuiltInStorageCreator::OpenAndCacheBuiltInStoragePartition(IStorage** outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> scopedLock(m_CacheMutex);
    return OpenAndCacheBuiltInStoragePartitionInternal(outValue, RedirectPartitionId(id));
}

BuiltInStorageCreator::BuiltInStorageCreator(const Configuration& config) NN_NOEXCEPT
    : m_Config(config),
      m_CacheMutex(false)
{
    NN_STATIC_ASSERT(BuiltInStorageCreator::BisStorageInstanceCount == BisGptPartitionCount + 5);
    memset(m_CachedBisPartitions, 0x00, sizeof(m_CachedBisPartitions));
}

BuiltInStorageCreator::~BuiltInStorageCreator() NN_NOEXCEPT
{
}


Result BuiltInStorageCreator::Create(std::shared_ptr<fs::IStorage>* outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)

    IStorage* pStorage = nullptr;
    NN_FSP_REQUIRES(IsValidId(id), nn::fs::ResultInvalidArgument());
    NN_RESULT_DO(OpenAndCacheBuiltInStoragePartition(&pStorage, id));

    int64_t size;
    NN_RESULT_DO(pStorage->GetSize(&size));

    std::shared_ptr<IStorage> pNewStorage = fssystem::AllocateShared<SubStorage>(pStorage, 0, size);
    if( pNewStorage == nullptr )
    {
        return fs::ResultAllocationMemoryFailedInBuiltInStorageCreatorD();
    }

    *outValue = std::move(pNewStorage);
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(outValue);
    NN_UNUSED(id);
    return nn::fs::ResultPartitionNotFound();
#endif
}

// 通常 fs
nn::fs::BisPartitionId BuiltInStorageCreator::RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case nn::fs::BisPartitionId::SystemProperPartition:
            // TORIAEZU: System パーティションの二重参照を回避
            return nn::fs::BisPartitionId::Invalid;
        default:
            return id;
    }
}

bool BuiltInStorageCreator::IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    return false;
}


// fssafe.memfs
nn::fs::BisPartitionId BuiltInStorageCreatorSspOnSd::RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    return id;
}

bool BuiltInStorageCreatorSspOnSd::IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    return false; // bisCreator の管理下の ssp は無し
}


// fssafe
nn::fs::BisPartitionId BuiltInStorageCreatorRedirectSafe::RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case nn::fs::BisPartitionId::System:
            return nn::fs::BisPartitionId::SafeMode;
        default:
            return id;
    }
}

bool BuiltInStorageCreatorRedirectSafe::IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    return false;
}


// fssafe.ssp
nn::fs::BisPartitionId BuiltInStorageCreatorSspOnSafe::RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    switch (id)
    {
        case nn::fs::BisPartitionId::System:
            return nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode;
        default:
            return id;
    }
}

bool BuiltInStorageCreatorSspOnSafe::IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    return (id == nn::fs::BisPartitionId::SignedSystemPartitionOnSafeMode);
}


// fs.memfs
nn::fs::BisPartitionId BuiltInStorageCreatorSspOnSystem::RedirectPartitionId(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    return id;
}

bool BuiltInStorageCreatorSspOnSystem::IsSignedSystemPartition(nn::fs::BisPartitionId id) NN_NOEXCEPT
{
    return (id == nn::fs::BisPartitionId::System);
}

}}}
