﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/os/os_TickTypes.h>
#include <nn/fs/fs_PriorityPrivate.h>

#if !defined(NN_FS_SCOPED_TRACE_ENABLED)
#if defined(NN_BUILD_CONFIG_HARDWARE_SUPPORTS_NX)
// ターゲット側でログを有効にするかどうかを定義します
#define NN_FS_SCOPED_TRACE_ENABLED 0 // NOLINT(preprocessor/const)
#else
// NX 実機以外では使用しません。 0 のまま変更しないでください
#define NN_FS_SCOPED_TRACE_ENABLED 0 // NOLINT(preprocessor/const)
#endif
#endif

#if NN_FS_SCOPED_TRACE_ENABLED

#define NN_FS_SCOPED_TRACE_FLUSH                nn::fssrv::detail::FlushLog()

#define NN_FS_SCOPED_TRACE(...) \
    nn::fssrv::detail::ScopedTrace fsAccessLogScopedTrace(nn::fssrv::detail::ScopedTrace::GetClassName(this), \
    NN_CURRENT_FUNCTION_NAME, \
    [=](char* buffer, size_t bufferLength) NN_NOEXCEPT { \
        return nn::fssrv::detail::ScopedTrace::Formatter(buffer, bufferLength).FormatLog(__VA_ARGS__); \
    })
#define NN_FS_SCOPED_TRACE_APPEND_LOG(...) \
    fsAccessLogScopedTrace.AppendFormatFunction([=](char* buffer, size_t bufferLength) NN_NOEXCEPT { \
        return nn::fssrv::detail::ScopedTrace::Formatter(buffer, bufferLength).FormatLog(__VA_ARGS__); \
    })

#define NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(classname)  \
    class classname; \
    template<>const char* ::nn::fssrv::detail::ScopedTrace::GetClassName<classname>(classname*) NN_NOEXCEPT { return #classname; }

#define NN_FS_SCOPED_TRACE_CAPTURE_RESULT       return fsAccessLogScopedTrace << [&]() NN_NOEXCEPT -> nn::Result
#define NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT   ();

#define NN_FS_SCOPED_TRACE_SUPPRESS()           fsAccessLogScopedTrace.Suppress()

namespace nn { namespace fssrv { namespace detail {

    void FlushLog() NN_NOEXCEPT;

    class ScopedTrace
    {
    public:
        // __VA_ARGS__ が空だった場合のカンマ問題に対応するための一時オブジェクト
        class Formatter
        {
        public:
            Formatter(char* buffer, size_t length) NN_NOEXCEPT;

        public:
            int FormatLog(const char* format, ...) NN_NOEXCEPT;
            int FormatLog() NN_NOEXCEPT;
        private:
            char* m_Buffer;
            size_t m_Length;
        };

    private:
        typedef std::function<int(char*, size_t)> LogFormatFunction;
    public:
        ScopedTrace(const char* className, const char* functionName, LogFormatFunction formatFunction) NN_NOEXCEPT;
        ~ScopedTrace() NN_NOEXCEPT;

    public:
        nn::Result operator << (nn::Result&& result) NN_NOEXCEPT;

        void AppendFormatFunction(LogFormatFunction formatFunction) NN_NOEXCEPT;
        void Suppress() NN_NOEXCEPT;

    public:
        template<typename T>
        static const char* GetClassName(T*) NN_NOEXCEPT;

    private:
        const char* m_ClassName;
        const char* m_FunctionName;
        LogFormatFunction m_LogFormatFunction;
        nn::os::Tick m_StartTick;
        nn::Result m_Result;
        bool m_HasResult;
        bool m_IsSuppressed;
        nn::fs::PriorityRaw m_Priority;
    };


}}}

#else // NN_FS_SCOPED_TRACE_ENABLED

#define NN_FS_SCOPED_TRACE_FLUSH

#define NN_FS_SCOPED_TRACE(...) (void)([&]() NN_NOEXCEPT { [](...) NN_NOEXCEPT {}( __VA_ARGS__); })
#define NN_FS_SCOPED_TRACE_APPEND_LOG(...) (void)([&]() NN_NOEXCEPT { [](...) NN_NOEXCEPT {}( __VA_ARGS__); })
#define NN_FS_SCOPED_TRACE_CAPTURE_RESULT
#define NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
#define NN_FS_SCOPED_TRACE_SUPPRESS()

#define NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(classname)

#endif // NN_FS_SCOPED_TRACE_ENABLED
