﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_PriorityUtility.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fssystem/fs_ServiceContext.h>

#include "fssrv_RingBuffer.h"
#include "fssrv_Trace.h"

#if NN_FS_SCOPED_TRACE_ENABLED

namespace nn { namespace fs {
    PriorityRaw GetPriorityRawOnCurrentThreadImpl() NN_NOEXCEPT;
}}

namespace nn { namespace fssrv {
    NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(FileSystemProxyImpl)
    NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(SaveDataInfoFilterReader)

    namespace detail {
        NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(FileSystemInterfaceAdapter)
        NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(FileInterfaceAdapter)
        NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(DirectoryInterfaceAdapter)
        NN_FS_SCOPED_TRACE_CLASSNAME_SPECIALIZATION(StorageInterfaceAdapter)
    }
}}

namespace nn { namespace fssrv { namespace detail {

namespace {

    class LogRingBuffer : public RingBuffer<char>
    {
    public:
        size_t ReadLine(char* outBuffer, size_t outBufferLength) NN_NOEXCEPT
        {
            auto scopedLock = GetScopedLock();
            size_t countRead = PeekImpl(outBuffer, outBufferLength);
            for( size_t index = 0; index < countRead; ++index )
            {
                if( outBuffer[index] == '\n' )
                {
                    countRead = index + 1;
                    break;
                }
            }
            return IgnoreImpl(countRead);
        }

        size_t ReadLineLongestMatch(char* outBuffer, size_t outBufferLength) NN_NOEXCEPT
        {
            auto scopedLock = GetScopedLock();
            size_t countRead = PeekImpl(outBuffer, outBufferLength);
            for( size_t index = countRead; index > 0; --index )
            {
                if( outBuffer[index - 1] == '\n' )
                {
                    countRead = index;
                    break;
                }
            }
            return IgnoreImpl(countRead);
        }
    };

    const size_t TraceLogBufferSize = 1024;
    const size_t FsLogBufferSize = 2 * 1024 * 1024;
    NN_ALIGNAS(32) char g_FsLogBuffer[FsLogBufferSize];
    LogRingBuffer g_FsLogRingBuffer;
    nn::os::Mutex g_LogMutex(false);
    bool g_IsInitialized = false;
    const nn::TimeSpan FirstWait(nn::TimeSpan::FromSeconds(16)); // 起動時にログを多く出すと止まるので長め
    const nn::TimeSpan IntervalWait(nn::TimeSpan::FromMilliSeconds(100));
    const size_t ThreadStackSize = 16 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
    nn::os::ThreadType g_Thread;
    nn::os::EventType g_EventFlush;

    bool g_IsEnabledTrace = true;
    const char* const MountName = "$fssrvTrace";
    const char* const LogFilePath = "$fssrvTrace:/fssrv_first_log_%d.txt";

    NN_STATIC_ASSERT(TraceLogBufferSize < FsLogBufferSize);

    void FlushLogImpl() NN_NOEXCEPT
    {
        while( g_FsLogRingBuffer.GetReadableCount() > 0 )
        {
            char buffer[TraceLogBufferSize];
            const size_t countBuffer = g_FsLogRingBuffer.ReadLineLongestMatch(buffer, sizeof(buffer));
            NN_UNUSED(countBuffer);
            NN_SDK_PUT(buffer, countBuffer);
        }
    }

    void FlushLogToSdCardImpl() NN_NOEXCEPT
    {
        bool isWritten = false;
        // fs 関数呼び出しにより、ログ出力が呼び出されないように無効化
        g_IsEnabledTrace = false;
        if( nn::fs::MountSdCard(MountName).IsSuccess() )
        {
            // 書き込みファイルの作成を試みる
            nn::Result result;
            char fileName[260];
            for( int i = 0; ; ++i )
            {
                nn::util::SNPrintf(fileName, sizeof(fileName), LogFilePath, i);
                result = nn::fs::CreateFile(fileName, 0);
                if( result.IsSuccess() )
                {
                    break;
                }
                if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
                {
                    break;
                }
            }
            if( result.IsSuccess() )
            {
                nn::fs::FileHandle handle;
                if( nn::fs::OpenFile(&handle, fileName, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend).IsSuccess() )
                {
                    int64_t offset = 0;
                    while( g_FsLogRingBuffer.GetReadableCount() > 0 )
                    {
                        char buffer[TraceLogBufferSize];
                        const size_t countBuffer = g_FsLogRingBuffer.Read(buffer, sizeof(buffer));
                        (void)nn::fs::WriteFile(handle, offset, buffer, countBuffer, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                        offset += countBuffer;
                    }
                    (void)nn::fs::FlushFile(handle);
                    nn::fs::CloseFile(handle);

                    isWritten = true;
                }
            }
            nn::fs::Unmount(MountName);
        }
        g_IsEnabledTrace = true;

        if( !isWritten )
        {
            // SD への書き込みできなかったら Log に出す
            FlushLogImpl();
        }
    }

    void ThreadFunction(void* arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);

        nn::fssystem::ServiceContext context;
        nn::fssystem::RegisterServiceContext(&context);

        // 初回のログは SD への書き込みをする
        nn::os::TimedWaitEvent(&g_EventFlush, FirstWait);
        FlushLogToSdCardImpl();

        for(;;)
        {
            nn::os::TimedWaitEvent(&g_EventFlush, IntervalWait);
            FlushLogImpl();
        }
    }

    void InitializeRingBuffer() NN_NOEXCEPT
    {
        if( !g_IsInitialized )
        {
            g_FsLogRingBuffer.SetBuffer(g_FsLogBuffer, FsLogBufferSize);

            nn::os::InitializeEvent(&g_EventFlush, false, nn::os::EventClearMode_AutoClear);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, ThreadFunction, NULL, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&g_Thread);

            g_IsInitialized = true;
        }
    }

    void Log(const char* format, ...) NN_NOEXCEPT
    {
        if( !g_IsEnabledTrace )
        {
            return;
        }

        char buffer[TraceLogBufferSize];
        std::lock_guard<nn::os::Mutex> scopedLock(g_LogMutex);

        InitializeRingBuffer();

        va_list arg;
        va_start(arg, format);
        size_t length = static_cast<size_t>(nn::util::VSNPrintf(buffer, sizeof(buffer), format, arg));
        va_end(arg);

        if( length >= sizeof(buffer) )
        {
            // ReadLine で読み取るのでバッファ溢れた場合は最後の文字を '\0' から '\n' に書き換えておく
            buffer[sizeof(buffer) - 1] = '\n';
            length = sizeof(buffer);
        }

        while( g_FsLogRingBuffer.GetWritableCount() < length )
        {
            nn::os::SignalEvent(&g_EventFlush);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        g_FsLogRingBuffer.Write(buffer, length);
    }
}

    void FlushLog() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_LogMutex);

        InitializeRingBuffer();

        while(g_FsLogRingBuffer.GetReadableCount() > 0)
        {
            nn::os::SignalEvent(&g_EventFlush);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }

    ScopedTrace::ScopedTrace(const char* className, const char* functionName, LogFormatFunction formatFunction) NN_NOEXCEPT
        : m_ClassName(className)
        , m_FunctionName(functionName)
        , m_LogFormatFunction(formatFunction)
        , m_HasResult(false)
        , m_IsSuppressed(false)
        , m_Priority(nn::fs::PriorityRaw_Normal)
    {
        nn::fs::PriorityRaw priority;
        if( nn::fs::detail::ConvertTlsIoPriorityToFsPriority(&priority, nn::fs::detail::GetTlsIoPriority()).IsSuccess() )
        {
            m_Priority = priority;
        }
        m_StartTick = nn::os::GetSystemTick();
    }

    ScopedTrace::~ScopedTrace() NN_NOEXCEPT
    {
        if( m_IsSuppressed )
        {
            return;
        }

        auto endTick = nn::os::GetSystemTick();
        char buffer[TraceLogBufferSize - 20];   // tick の分を除いておく

        int length = 0;
        if( m_HasResult )
        {
            length += nn::util::SNPrintf(buffer, sizeof(buffer), "%08x ", m_Result.GetInnerValueForDebug());
        }
        else
        {
            length += nn::util::SNPrintf(buffer, sizeof(buffer), "%8s ", "");
        }

        if( length < sizeof(buffer) )
        {
            length += nn::util::SNPrintf(buffer + length, sizeof(buffer) - length, "%s ", nn::fs::detail::IdString().ToString(m_Priority));
        }

        if( length < sizeof(buffer) )
        {
            if( m_ClassName != nullptr )
            {
                length += nn::util::SNPrintf(buffer + length, sizeof(buffer) - length, "%s::%-8s ", m_ClassName, m_FunctionName);
            }
            else
            {
                length += nn::util::SNPrintf(buffer + length, sizeof(buffer) - length, "%-8s ", m_FunctionName);
            }
        }
        if( length < sizeof(buffer) )
        {
            length += m_LogFormatFunction(buffer + length, sizeof(buffer) - length);
        }

        buffer[sizeof(buffer) - 1] = '\0';

        Log("[fs] %8lld %8lld %s\n",
            nn::os::ConvertToTimeSpan(m_StartTick).GetMicroSeconds(),
            nn::os::ConvertToTimeSpan(endTick - m_StartTick).GetMicroSeconds(),
            buffer
        );
    }

    nn::Result ScopedTrace::operator << (nn::Result&& result) NN_NOEXCEPT
    {
        m_Result = result;
        m_HasResult = true;
        return result;
    }

    void ScopedTrace::AppendFormatFunction(LogFormatFunction formatFunction) NN_NOEXCEPT
    {
        LogFormatFunction current = m_LogFormatFunction;
        m_LogFormatFunction = [=](char* buffer, size_t bufferLength) NN_NOEXCEPT
        {
            int length = current(buffer, bufferLength);
            if( length < 0 )
            {
                length = 0;
            }
            if( bufferLength > static_cast<size_t>(length) )
            {
                length += formatFunction(buffer + length, bufferLength - length);
            }
            return length;
        };
    }

    void ScopedTrace::Suppress() NN_NOEXCEPT
    {
        m_IsSuppressed = true;
    }

    ScopedTrace::Formatter::Formatter(char* buffer, size_t length) NN_NOEXCEPT
        : m_Buffer(buffer)
        , m_Length(length)
    {
    }

    int ScopedTrace::Formatter::FormatLog(const char* format, ...) NN_NOEXCEPT
    {
        va_list arg;
        va_start(arg, format);
        const int length = nn::util::VSNPrintf(m_Buffer, m_Length, format, arg);
        va_end(arg);
        return length;
    }

    int ScopedTrace::Formatter::FormatLog() NN_NOEXCEPT
    {
        return 0;
    }

}}}

#endif // NN_FS_SCOPED_TRACE_ENABLED
