﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_SaveDataExporter.h"
#include "fssrv_SaveDataTransferManager.h"

namespace nn { namespace fssrv { namespace detail {

    SaveDataExporter::SaveDataExporter(
        std::unique_ptr<fs::SaveDataExtraData> pSaveDataExtraData,
        const fs::SaveDataInfo& info,
        std::shared_ptr<fs::fsa::IFile>&& file,
        int64_t fileSize,
        std::unique_ptr<crypto::Aes128GcmEncryptor>&& initialDataEncryptor,
        std::unique_ptr<crypto::Aes128GcmEncryptor>&& encryptor,
        uint32_t version) NN_NOEXCEPT
        : m_pSaveDataExtraData(std::move(pSaveDataExtraData))
        , m_SaveDataInfo(info)
        , m_Pulled(0)
        , m_Length(fileSize + decltype(m_pEncryptor)::element_type::MacSize)
        , m_pFile(std::move(file))
        , m_FileOffset(0)
        , m_pInitialDataEncryptor(std::move(initialDataEncryptor))
        , m_pEncryptor(std::move(encryptor))
        , m_Version(version)
    {
    }

    Result SaveDataExporter::GetSaveDataInfo(nn::sf::Out<nn::fs::SaveDataInfo> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_SaveDataInfo);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataExporter::GetRestSize(nn::sf::Out<std::uint64_t> outValue) NN_NOEXCEPT
    {
        outValue.Set(m_Length - m_Pulled);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataExporter::Pull(nn::sf::Out<std::uint64_t> outValue, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        static const auto MacSize = decltype(m_pEncryptor)::element_type::MacSize;

        auto pDst = buffer.GetPointerUnsafe();
        NN_FSP_REQUIRES(pDst != nullptr, fs::ResultNullptrArgument());

        auto restBufferSize = std::min(static_cast<uint64_t>(buffer.GetSize()), m_Length - m_Pulled);


        size_t pulledSize = 0;

        // 実体
        if (m_Length - m_Pulled > MacSize)
        {
            auto restFileSize = m_Length - m_Pulled - MacSize;
            auto tryReadSize = static_cast<size_t>(std::min(static_cast<uint64_t>(restBufferSize), restFileSize));
            size_t readSize;
            NN_RESULT_DO(m_pFile->Read(&readSize, m_FileOffset, pDst, tryReadSize, nn::fs::ReadOption()));
            m_FileOffset += readSize;

            m_pEncryptor->Update(pDst, readSize, pDst, readSize);

            m_Pulled       += readSize;
            restBufferSize -= readSize;
            pDst           += readSize;
            pulledSize     += readSize;
        }

        NN_SDK_ASSERT(m_Length - m_Pulled >= 0);

        // mac
        if (m_Length - m_Pulled <= MacSize)
        {
            char mac[MacSize];
            m_pEncryptor->GetMac(&mac, sizeof(mac));

            auto macOffset = m_Pulled - (m_Length - MacSize);

            auto readMacSize = static_cast<size_t>(std::min(restBufferSize, m_Length - m_Pulled));
            memcpy(pDst, mac + macOffset, readMacSize);
            m_Pulled   += readMacSize;
            pulledSize += readMacSize;
        }

        outValue.Set(pulledSize);
        NN_RESULT_SUCCESS;
    }

    Result SaveDataExporter::PullInitialData(const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(buffer.GetSize() >= sizeof(fs::InitialData), fs::ResultInvalidSize());
        auto pDst = buffer.GetPointerUnsafe();
        NN_FSP_REQUIRES(pDst != nullptr, fs::ResultNullptrArgument());

        fs::InitialData initialData;
        memset(&initialData, 0x00, sizeof(initialData));
        memcpy(&initialData.encryptedArea.extraData, m_pSaveDataExtraData.get(), sizeof(initialData.encryptedArea.extraData));
        initialData.encryptedArea.version = m_Version;
        m_pInitialDataEncryptor->Update(&initialData.encryptedArea, sizeof(initialData.encryptedArea), &initialData.encryptedArea, sizeof(initialData.encryptedArea));
        m_pInitialDataEncryptor->GetMac(&initialData.mac, sizeof(initialData.mac));

        std::memcpy(pDst, &initialData, sizeof(fs::InitialData));
        NN_RESULT_SUCCESS;
    }

}}}
