﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_LocationResolverSet.h"

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_DataId.h>
#include <nn/ncm/ncm_Result.h>

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
#define LOCATION_RESOLVER_ENABLED
#endif

namespace nn { namespace fssrv {

#ifdef LOCATION_RESOLVER_ENABLED
    namespace {
        nn::os::Mutex g_LocationResolverSetInitializeMutex(false);
        bool g_Initialized = false;
    }
#endif

    void InitializeLocationResolverSet() NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        std::lock_guard<os::Mutex> scopedLock(g_LocationResolverSetInitializeMutex);
        if (!g_Initialized)
        {
            // この処理は lr サービスが利用できるようになるまで待たされる。
            nn::lr::Initialize();
            g_Initialized = true;
        }
#endif
    }


namespace detail {

    LocationResolverSet::LocationResolverSet() NN_NOEXCEPT
      : m_LocationResolverMutex(false)
    {
        for (int i = 0; i < NumOfResolvers; i++)
        {
            m_LocationResolverSet[i] = nullptr;
        }
    }

    LocationResolverSet::~LocationResolverSet() NN_NOEXCEPT
    {
    }

    Result LocationResolverSet::GetLocationResolver(nn::lr::LocationResolver** outValue, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        InitializeLocationResolverSet();

        int index = -1;
        for (int i = 0; i < NumOfResolvers; i++)
        {
            if (storageId == StorageList[i])
            {
                index = i;
                break;
            }
        }

        if (index == -1)
        {
            return nn::lr::ResultApplicationNotFound();
        }

        std::lock_guard<os::Mutex> scopedLock(m_LocationResolverMutex);
        if (!m_LocationResolverSet[index])
        {
            m_LocationResolverSet[index].emplace();
            auto result = nn::lr::OpenLocationResolver(&m_LocationResolverSet[index].value(), StorageList[index]);
            if (result.IsFailure())
            {
                m_LocationResolverSet[index] = nn::util::nullopt;
                return nn::lr::ResultApplicationNotFound();
            }
        }
        *outValue = &m_LocationResolverSet[index].value();
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(outValue);
        NN_UNUSED(storageId);
        return nn::lr::ResultApplicationNotFound();
#endif
    }

    Result LocationResolverSet::GetRegisteredLocationResolver(
        nn::lr::RegisteredLocationResolver* pOutValue
    ) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        InitializeLocationResolverSet();

        return nn::lr::OpenRegisteredLocationResolver(pOutValue);
#else
        NN_UNUSED(pOutValue);
        return nn::lr::ResultApplicationNotFound();
#endif
    }

    Result LocationResolverSet::GetAddOnContentLocationResolver(
        nn::lr::AddOnContentLocationResolver** ppOutValue
    ) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        InitializeLocationResolverSet();
        std::lock_guard<os::Mutex> scopedLock(m_LocationResolverMutex);

        if (m_AddOnContentLocationResolver == nn::util::nullopt)
        {
            nn::util::optional<nn::lr::AddOnContentLocationResolver> aocLr;
            aocLr.emplace();
            NN_RESULT_DO(nn::lr::OpenAddOnContentLocationResolver(&aocLr.value()));
            m_AddOnContentLocationResolver = std::move(aocLr);
        }

        *ppOutValue = &m_AddOnContentLocationResolver.value();
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(ppOutValue);
        return nn::lr::ResultApplicationNotFound();
#endif
}

    Result LocationResolverSet::ResolveApplicationControlPath(nn::lr::Path* outValue, nn::ncm::ApplicationId id, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::LocationResolver* pLocationResolver = nullptr;
        NN_RESULT_DO(GetLocationResolver(&pLocationResolver, storageId));

        return pLocationResolver->ResolveApplicationControlPath(outValue, id);
#else
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_UNUSED(storageId);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveApplicationHtmlDocumentPath(nn::lr::Path* outValue, nn::ncm::ApplicationId id, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::LocationResolver* pLocationResolver = nullptr;
        NN_RESULT_DO(GetLocationResolver(&pLocationResolver, storageId));

        return pLocationResolver->ResolveApplicationHtmlDocumentPath(outValue, id);
#else
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_UNUSED(storageId);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveProgramPath(nn::lr::Path* outValue, Bit64 id, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::LocationResolver* pLocationResolver = nullptr;
        NN_RESULT_DO(GetLocationResolver(&pLocationResolver, storageId));

        nn::ncm::ProgramId programId = { id };
        return pLocationResolver->ResolveProgramPath(outValue, programId);
#else
        NN_UNUSED(outValue);
        NN_UNUSED(id);
        NN_UNUSED(storageId);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveAddOnContentPath(nn::lr::Path* outValue, nn::ncm::DataId dataId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::AddOnContentLocationResolver* pLocationResolver;
        NN_RESULT_DO(GetAddOnContentLocationResolver(&pLocationResolver));
        return pLocationResolver->ResolveAddOnContentPath(outValue, dataId);
#else
        NN_UNUSED(outValue);
        NN_UNUSED(dataId);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveDataPath(nn::lr::Path* outValue, nn::ncm::DataId dataId, nn::ncm::StorageId storageId) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        NN_RESULT_THROW_UNLESS(storageId != nn::ncm::StorageId::None, nn::fs::ResultInvalidArgument());
        nn::lr::LocationResolver* pLocationResolver = nullptr;
        NN_RESULT_DO(GetLocationResolver(&pLocationResolver, storageId));
        return pLocationResolver->ResolveDataPath(outValue, dataId);
#else
        NN_UNUSED(outValue);
        NN_UNUSED(dataId);
        NN_UNUSED(storageId);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveRegisteredProgramPath(nn::lr::Path* pOutValue, Bit64 id) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::RegisteredLocationResolver locationResolver;
        NN_RESULT_DO(GetRegisteredLocationResolver(&locationResolver));

        nn::ncm::ProgramId programId = { id };
        return locationResolver.ResolveProgramPath(pOutValue, programId);

#else
        NN_UNUSED(pOutValue);
        NN_UNUSED(id);
        return nn::fs::ResultNotImplemented();
#endif
    }

    Result LocationResolverSet::ResolveRegisteredHtmlDocumentPath(nn::lr::Path* pOutValue, Bit64 id) NN_NOEXCEPT
    {
#ifdef LOCATION_RESOLVER_ENABLED
        nn::lr::RegisteredLocationResolver locationResolver;
        NN_RESULT_DO(GetRegisteredLocationResolver(&locationResolver));

        nn::ncm::ProgramId programId = { id };
        return locationResolver.ResolveHtmlDocumentPath(pOutValue, programId);
#else
        NN_UNUSED(pOutValue);
        NN_UNUSED(id);
        return nn::fs::ResultNotImplemented();
#endif
    }

}}}
