﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/gc/gc.h>
#include <nn/gc/writer/gc_Writer.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_GameCard.h>

#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Define.h>

// ダミー gc
namespace nn { namespace gc {

namespace {

nn::fs::IStorage* g_DummyStorage = nullptr;
detail::CardHeader g_CardHeader = {0};
bool g_Attached = false;
bool g_Activated = false;
bool g_IsSecureMode = false;
bool g_Initialized = false;
bool g_WriteMode = false;


nn::Result CheckActivated()
{
    if( g_Activated )
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        return nn::fs::ResultGameCardAccessFailed();
    }
}

nn::Result CheckAttached()
{
    if( g_Attached )
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        return nn::fs::ResultGameCardAccessFailed();
    }
}

}

void SetStorage(nn::fs::IStorage* pStorage)
{
    g_DummyStorage = pStorage;
    if( pStorage != nullptr )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_DummyStorage->Read(0x100, &g_CardHeader, sizeof(g_CardHeader)));
    }
}

void Detach()
{
    g_Attached = false;
    g_Activated = false;
}

void Attach()
{
    g_Attached = true;
}

namespace writer {
void ChangeMode(const AsicMode asicMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    NN_UNUSED(asicMode);
}

nn::Result ActivateForWriter() NN_NOEXCEPT
{
    g_WriteMode = true;
    g_Activated = true;
    NN_SDK_REQUIRES(g_Initialized);
    NN_RESULT_SUCCESS;
}

nn::Result EraseAndWriteParameter(MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    NN_UNUSED(memorySize);
    NN_UNUSED(secureAreaStartPageAddress);
    NN_RESULT_SUCCESS;
}

nn::Result Write(void* pInDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
{
    NN_UNUSED(pageCount);
    NN_SDK_REQUIRES(g_Initialized);
    NN_SDK_REQUIRES(g_WriteMode);

    NN_RESULT_DO(CheckActivated());

    if( dataBufferSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    else if( pInDataBuffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }

    return g_DummyStorage->Write(pageAddress * GcPageSize, pInDataBuffer, dataBufferSize);

}

void SetVerifyEnalbleFlag(bool isVerifyEnable) NN_NOEXCEPT
{
    NN_UNUSED(isVerifyEnable);
}

void SetUserAsicFirmwareBuffer(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_UNUSED(buffer);
    NN_UNUSED(bufferLength);
}
nn::Result GetRmaInformation(RmaInformation* pOutValue) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_RESULT_SUCCESS;
}

nn::Result GetCardAvailableRawSize(int64_t* pOutValue) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_RESULT_SUCCESS;
}

nn::Result WriteDevCardParam(detail::DevCardParameter& devCardParameter) NN_NOEXCEPT
{
    NN_UNUSED(devCardParameter);
    NN_RESULT_SUCCESS;
}

nn::Result ReadDevCardParam(detail::DevCardParameter* pOutValue) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_RESULT_SUCCESS;
}

nn::Result ForceErase() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

} // namespace writer


void PresetInternalKeys(const void* encryptedKeyBuffer, const size_t encryptedKeyBufferSize,
                     const void* socCertBuffer, const size_t socCertBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(encryptedKeyBuffer);
    NN_UNUSED(encryptedKeyBufferSize);
    NN_UNUSED(socCertBuffer);
    NN_UNUSED(socCertBufferSize);
    return;
}

void Initialize(void* pWorkBuffer,const size_t workBufferSize, const nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_UNUSED(pWorkBuffer);
    NN_UNUSED(workBufferSize);
    NN_UNUSED(workBufferDeviceVirtualAddress);
    g_Initialized = true;
}

void Finalize() NN_NOEXCEPT
{
    g_Initialized = false;
}

void PowerOffGameCard() NN_NOEXCEPT
{
}

void RegisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_UNUSED(bufferAddress);
    NN_UNUSED(bufferSize);
    NN_UNUSED(bufferDeviceVirtualAddress);
}

void UnregisterDeviceVirtualAddress(const uintptr_t bufferAddress, const size_t bufferSize, const nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    NN_UNUSED(bufferAddress);
    NN_UNUSED(bufferSize);
    NN_UNUSED(bufferDeviceVirtualAddress);
}

nn::Result GetInitializationResult() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

nn::Result Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    g_Activated = true;
    g_WriteMode = false;
    NN_RESULT_SUCCESS;
}

void Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    g_Activated = false;
    g_IsSecureMode = false;
}

nn::Result SetCardToSecureMode() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    g_IsSecureMode = true;
    NN_RESULT_SUCCESS;
}

nn::Result Read(void* pOutDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
{
    NN_UNUSED(pageCount);
    NN_SDK_REQUIRES(g_Initialized);

    NN_RESULT_DO(CheckActivated());

    if( dataBufferSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    else if( pOutDataBuffer == nullptr )
    {
        return nn::fs::ResultNullptrArgument();
    }
    else if( g_IsSecureMode )
    {
        if( (pageAddress + pageCount) * GcPageSize >= g_CardHeader.limArea * GcPageSize )
        {
            return g_DummyStorage->Read(pageAddress * GcPageSize, pOutDataBuffer, dataBufferSize);
        }
        else
        {
            memset(pOutDataBuffer, 0xCD, dataBufferSize);
            NN_RESULT_SUCCESS;
        }
    }
    else
    {
        if( (pageAddress + pageCount) * GcPageSize < g_CardHeader.limArea * GcPageSize )
        {
            return g_DummyStorage->Read(pageAddress * GcPageSize, pOutDataBuffer, dataBufferSize);
        }
        else
        {
            memset(pOutDataBuffer, 0xFF, dataBufferSize); // all 0xff
            NN_RESULT_SUCCESS;
        }
    }
}


void PutToSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
}


void Awaken() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
}


bool IsCardInserted() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    return g_Attached;
}


bool IsCardActivationValid() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    return g_Activated;
}


nn::Result GetCardStatus(GameCardStatus* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_Initialized);
    NN_RESULT_DO(CheckAttached());
    NN_RESULT_DO(CheckActivated());

    if( g_DummyStorage )
    {
        GameCardStatus status;

        memcpy(status.packageId, g_CardHeader.packageId, GcPackageIdSize);
        status.partitionFsHeaderAddress = g_CardHeader.partitionFsHeaderAddress;
        status.partitionFsHeaderSize = g_CardHeader.partitionFsHeaderSize;
        memcpy(status.partitionFsHeaderHash, g_CardHeader.partitionFsHeaderHash, GcPartitionFsHeaderHashSize);

        status.cardSize        = 32 * detail::AvailableSizeBase; // TODO: 32 以外のサイズ設定
        status.cupVersion      = 0;                              // TODO: 取得
        status.normalAreaSize  = g_CardHeader.limArea * GcPageSize;
        status.secureAreaSize  = status.cardSize - status.normalAreaSize;
        status.flags           = g_CardHeader.flags;

        memcpy(pOutValue, &status, sizeof(status));
    }
    else
    {
        pOutValue->cardSize       = 1ULL * 1024 * 1024 * 1024;
        pOutValue->normalAreaSize = 256ULL * 1024 * 1024;
        pOutValue->secureAreaSize = 512ULL * 1024 * 1024;
    }

    // TODO: 残りのフィールド

    NN_RESULT_SUCCESS;
}

nn::Result GetCardDeviceCertificate(void* outBuffer,const size_t outBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(outBuffer);
    NN_UNUSED(outBufferSize);
    NN_RESULT_SUCCESS;
}

nn::Result GetCardImageHash(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(outBuffer);
    NN_UNUSED(outBufferSize);
    NN_RESULT_SUCCESS;
}

nn::Result GetGameCardIdSet(GameCardIdSet* pOutValue) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_RESULT_SUCCESS;
}

void RegisterDetectionEventCallback(nn::gc::detail::GcCallbackFunctionPointer const pDetectionEventCallback, void* const pParameter) NN_NOEXCEPT
{
    NN_UNUSED(pDetectionEventCallback);
    NN_UNUSED(pParameter);
}

void UnregisterDetectionEventCallback() NN_NOEXCEPT
{
}

Result GetCardHeader(void* outBuffer, const size_t outBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(outBuffer);
    NN_UNUSED(outBufferSize);
    NN_RESULT_SUCCESS;
}

nn::Result GetErrorInfo(nn::fs::GameCardErrorReportInfo* pOutGameCardErrorReportInfo) NN_NOEXCEPT
{
    NN_UNUSED(pOutGameCardErrorReportInfo);
    NN_RESULT_SUCCESS;
}

nn::Result GetCardDeviceId(void* pOutValue, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(size);
    NN_RESULT_SUCCESS;
}

}}
