﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <mutex>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/sf/fssrv_IFileSystem.h>
#include <nn/fssrv/fssrv_FileSystemProxyImpl.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/sf/sf_ContextControl.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_AllocationPolicies.h>
#include <nn/util/util_Optional.h>

#include "fssrv_FileSystemInterfaceAdapter.h"
#include "fssrv_AllocatorForServiceFrameWork.h"
#include "fssrv_PathNormalizer.h"

#include "fssrv_Trace.h"

namespace nn { namespace fssrv { namespace detail {

namespace {
    typedef nn::sf::ObjectFactory<AllocatorForServiceFrameWork::Policy> FileSystemFactory;
}

    class FileSystemInterfaceAdapter;

    FileInterfaceAdapter::FileInterfaceAdapter(std::unique_ptr<nn::fs::fsa::IFile>&& fileInterface, FileSystemInterfaceAdapter* parentFileSystem, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT
        : m_parentFileSystem(parentFileSystem, true),
          m_fileInterface(std::move(fileInterface)),
          m_OpenCountSemaphore(std::move(openCountSemaphore))
    {
        NN_FS_SCOPED_TRACE("%p", m_fileInterface.get());
    }
    FileInterfaceAdapter::~FileInterfaceAdapter() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_fileInterface.get());
    }

    Result FileInterfaceAdapter::Read(nn::sf::Out<std::int64_t> outValue, std::int64_t offset, const nn::sf::OutBuffer& buffer, std::int64_t size, nn::fssrv::sf::ReadOption option) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld %lld", m_fileInterface.get(), offset, size);

        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(offset >= 0, nn::fs::ResultInvalidOffset());
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());

            auto pBuffer = buffer.GetPointerUnsafe();
            size_t readSize = 0;
            nn::fs::ReadOption readOption;
            readOption.reserved = option.reserved;

            // TODO: StorageInterfaceAdapter のロジックと共通化
            if (m_parentFileSystem->IsDeepRetryEnabled())
            {
                const int RetryCountMax = 2;
                Result result;

                for (int retryCount = 0; retryCount < RetryCountMax + 1; retryCount++)
                {
                    if (retryCount != 0)
                    {
                        NN_DETAIL_FS_WARN_ON_DATA_CORRUPTED("DataCorrupted (retry count: %d), 0x%08x at %s() - %d %lld\n", retryCount - 1, result.GetInnerValueForDebug(), __FUNCTION__, __LINE__);
                    }

                    auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();

                    result = m_fileInterface->Read(&readSize, offset, pBuffer, static_cast<size_t>(size), readOption);
                    if (result.IsSuccess())
                    {
                        // TODO: 5 NUP 時点では StorageInterfaceAdapter でのリトライ効果様子見のためカウントアップしない
                        // m_parentFileSystem->IncrementRomFsRecoveredByInvalidateCacheCount();
                        outValue.Set(readSize);
                        NN_RESULT_SUCCESS;
                    }

                    auto isDataCorrupted = fs::ResultDataCorrupted::Includes(result);
                    auto isAccessFailed = fs::ResultGameCardAccessFailed::Includes(result);

                    // corruption もしくは access failure 検知時には retry する
                    if (!isDataCorrupted && !isAccessFailed)
                    {
                        // retry 不能なエラー扱いなので即時返す
                        return result;
                    }
                    // access failed 検知時にはアクセスエラー検知の対象であればキャッシュを無効化した上で処理を後回しにする
                    // アクセスエラー検知の対象でなければエラーを即時返す
                    else if (isAccessFailed)
                    {
                        if (m_parentFileSystem->IsAccessFailureDetectionObserved())
                        {
                            scopedReaderLock = util::nullopt;
                            InvalidateCacheOnStorage();
                            return nn::sf::DeferProcess();
                        }
                        return result;
                    }
                    // corruption 検知時には retry する
                    else
                    {
                        scopedReaderLock = util::nullopt;

                        InvalidateCacheOnStorage();

                        if (retryCount >= RetryCountMax)
                        {
                            // TODO: 5 NUP 時点では StorageInterfaceAdapter でのリトライ効果様子見のためカウントアップしない
                            // m_parentFileSystem->IncrementRomFsUnrecoverableDataCorruptionByRemountCount();

                            // retry してもストレージアクセスが一向にできないときアクセスエラー検知の対象であれば処理を後回しにする。
                            // アクセスエラー検知の対象でなければエラーを即時返す。
                            if (m_parentFileSystem->IsAccessFailureDetectionObserved())
                            {
                                return nn::sf::DeferProcess();
                            }
                            return result;
                        }

                        continue;
                    }
                }

                // ここには到達しない
                NN_ABORT("Must be unreachable.");
            }
            else
            {
                NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_fileInterface->Read(&readSize, offset, pBuffer, static_cast<size_t>(size), readOption));
                outValue.Set(readSize);
                NN_RESULT_SUCCESS;
            }
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileInterfaceAdapter::Write(std::int64_t offset, const nn::sf::InBuffer& buffer, std::int64_t size, nn::fssrv::sf::WriteOption option) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld %lld", m_fileInterface.get(), offset, size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(offset >= 0, nn::fs::ResultInvalidOffset());
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());

            fssystem::ScopedThreadPriorityChangerByAccessPriority scopedPriorityChanger(fssystem::ScopedThreadPriorityChangerByAccessPriority::AccessMode::Write);

            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            auto pBuffer = buffer.GetPointerUnsafe();
            nn::fs::WriteOption writeOption;
            writeOption.flags = option.flags;
            return m_fileInterface->Write(offset, pBuffer, static_cast<size_t>(size), writeOption);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileInterfaceAdapter::Flush() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_fileInterface.get());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            return m_fileInterface->Flush();
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileInterfaceAdapter::SetSize(std::int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %lld", m_fileInterface.get(), size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());
            return m_fileInterface->SetSize(size);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileInterfaceAdapter::GetSize(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_fileInterface.get());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            std::int64_t size;
            NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_fileInterface->GetSize(&size));
            outValue.Set(size);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileInterfaceAdapter::OperateRange(
        nn::sf::Out<nn::fs::QueryRangeInfo> outValue,
        std::int32_t operationId,
        int64_t offset,
        int64_t size) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %d %lld %lld", m_fileInterface.get(), operationId, offset, size);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            NN_RESULT_THROW_UNLESS(outValue.GetPointer() != nullptr, nn::fs::ResultNullptrArgument());

            // プロセスの情報が混入しないように、返却する値は必ず Clear してから設定します。
            outValue->Clear();

            // デバッグ用の operationId のみで動作するようにします。
            if( operationId == static_cast<int32_t>(nn::fs::OperationId::QueryRange) )
            {
                auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
                nn::fs::StorageQueryRangeInfo info;
                NN_RESULT_DO(m_fileInterface->OperateRange(
                    &info,
                    sizeof(info),
                    nn::fs::OperationId::QueryRange,
                    offset,
                    size,
                    nullptr,
                    0));

                outValue->Merge(info);
            }

            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    void FileInterfaceAdapter::InvalidateCacheOnStorage() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_parentFileSystem->IsDeepRetryEnabled());
        std::lock_guard<os::ReaderWriterLock> scopedWriterLock(m_parentFileSystem->GetReaderWriterLockForCacheInvalidation());
        m_fileInterface->OperateRange(nullptr, 0, fs::OperationId::Invalidate, 0, std::numeric_limits<int64_t>::max(), nullptr, 0);
    }

    DirectoryInterfaceAdapter::DirectoryInterfaceAdapter(std::unique_ptr<nn::fs::fsa::IDirectory>&& directoryInterface, FileSystemInterfaceAdapter* parentFileSystem, std::unique_lock<fssystem::SemaphoreAdaptor>&& openCountSemaphore) NN_NOEXCEPT
        : m_parentFileSystem(parentFileSystem, true),
          m_directoryInterface(std::move(directoryInterface)),
          m_OpenCountSemaphore(std::move(openCountSemaphore))
    {
        NN_FS_SCOPED_TRACE("%p", m_directoryInterface.get());
    }
    DirectoryInterfaceAdapter::~DirectoryInterfaceAdapter() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_directoryInterface.get());
    }
    Result DirectoryInterfaceAdapter::Read(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_directoryInterface.get());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            int64_t num = 0;
            int64_t count = outEntries.GetSize() / sizeof(nn::fs::DirectoryEntry);
            NN_RESULT_THROW_UNLESS(count >= 0, nn::fs::ResultInvalidSize());
            nn::fs::DirectoryEntry* de = reinterpret_cast<nn::fs::DirectoryEntry*>(outEntries.GetPointerUnsafe());
            NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_directoryInterface->Read(&num, de, count));
            outValue.Set(num);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result DirectoryInterfaceAdapter::GetEntryCount(nn::sf::Out<std::int64_t> outValue) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_directoryInterface.get());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = m_parentFileSystem->AcquireReaderLockForCacheInvalidation();
            int64_t count = 0;
            NN_RESULT_DO(m_directoryInterface->GetEntryCount(&count));
            outValue.Set(count);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    FileSystemInterfaceAdapter::FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, bool isOpenCountLimited) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxyImpl, true)
        , m_fileSystemInterface(std::move(fileSystemInterface))
        , m_IsOpenCountLimited(isOpenCountLimited)
        , m_DeepRetryEnabled(false)
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
    }
    FileSystemInterfaceAdapter::FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, std::unique_lock<fssystem::SemaphoreAdaptor>&& mountCountSemaphore, bool isOpenCountLimited) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxyImpl, true)
        , m_fileSystemInterface(std::move(fileSystemInterface))
        , m_MountCountSemaphore(std::move(mountCountSemaphore))
        , m_IsOpenCountLimited(isOpenCountLimited)
        , m_DeepRetryEnabled(false)
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
    }
    FileSystemInterfaceAdapter::FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, bool isOpenCountLimited, bool enableDeepRetry) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxyImpl, true)
        , m_fileSystemInterface(std::move(fileSystemInterface))
        , m_IsOpenCountLimited(isOpenCountLimited)
        , m_DeepRetryEnabled(enableDeepRetry)
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
    }
    FileSystemInterfaceAdapter::FileSystemInterfaceAdapter(std::shared_ptr<nn::fs::fsa::IFileSystem>&& fileSystemInterface, FileSystemProxyImpl* pFileSystemProxyImpl, std::unique_lock<fssystem::SemaphoreAdaptor>&& mountCountSemaphore, bool isOpenCountLimited, bool enableDeepRetry) NN_NOEXCEPT
        : m_pFileSystemProxyImpl(pFileSystemProxyImpl, true)
        , m_fileSystemInterface(std::move(fileSystemInterface))
        , m_MountCountSemaphore(std::move(mountCountSemaphore))
        , m_IsOpenCountLimited(isOpenCountLimited)
        , m_DeepRetryEnabled(enableDeepRetry)
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
    }
    FileSystemInterfaceAdapter::~FileSystemInterfaceAdapter() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
    }

    Result FileSystemInterfaceAdapter::CreateFile(const nn::fssrv::sf::Path& path, std::int64_t size, uint32_t option) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s %lld %x", m_fileSystemInterface.get(), path.str, size, option);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            NN_RESULT_THROW_UNLESS(size >= 0, nn::fs::ResultInvalidSize());

            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            return m_fileSystemInterface->CreateFile(normalizedPath.GetPath(), size, option);
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::DeleteFile(const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            return m_fileSystemInterface->DeleteFile(normalizedPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::CreateDirectory(const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            if( strncmp(normalizedPath.GetPath(), "/", 2) == 0 )
            {
                return nn::fs::ResultPathAlreadyExists();
            }

            return m_fileSystemInterface->CreateDirectory(normalizedPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::DeleteDirectory(const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            if( strncmp(normalizedPath.GetPath(), "/", 2) == 0 )
            {
                return nn::fs::ResultDirectoryUndeletable();
            }

            return m_fileSystemInterface->DeleteDirectory(normalizedPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::DeleteDirectoryRecursively(const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            if( strncmp(normalizedPath.GetPath(), "/", 2) == 0 )
            {
                return nn::fs::ResultDirectoryUndeletable();
            }

            return m_fileSystemInterface->DeleteDirectoryRecursively(normalizedPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::CleanDirectoryRecursively(const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            return m_fileSystemInterface->CleanDirectoryRecursively(normalizedPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::RenameFile(const nn::fssrv::sf::Path& currentPath, const nn::fssrv::sf::Path& newPath) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s %s", m_fileSystemInterface.get(), currentPath.str, newPath.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedCurrentPath(currentPath.str);
            PathNormalizer normalizedNewPath(newPath.str);
            NN_RESULT_THROW_UNLESS(normalizedCurrentPath.GetPath() != nullptr, normalizedCurrentPath.GetResult());
            NN_RESULT_THROW_UNLESS(normalizedNewPath.GetPath() != nullptr, normalizedNewPath.GetResult());

            return m_fileSystemInterface->RenameFile(normalizedCurrentPath.GetPath(), normalizedNewPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::RenameDirectory(const nn::fssrv::sf::Path& currentPath, const nn::fssrv::sf::Path& newPath) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s %s", m_fileSystemInterface.get(), currentPath.str, newPath.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedCurrentPath(currentPath.str);
            PathNormalizer normalizedNewPath(newPath.str);
            NN_RESULT_THROW_UNLESS(normalizedCurrentPath.GetPath() != nullptr, normalizedCurrentPath.GetResult());
            NN_RESULT_THROW_UNLESS(normalizedNewPath.GetPath() != nullptr, normalizedNewPath.GetResult());

            if( fssystem::PathTool::IsSubpath(normalizedCurrentPath.GetPath(), normalizedNewPath.GetPath()) )
            {
                return nn::fs::ResultDirectoryUnrenamable();
            }

            return m_fileSystemInterface->RenameDirectory(normalizedCurrentPath.GetPath(), normalizedNewPath.GetPath());
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::GetEntryType(nn::sf::Out<std::uint32_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            nn::fs::DirectoryEntryType de;
            NN_RESULT_DO(m_fileSystemInterface->GetEntryType(&de, normalizedPath.GetPath()));
            outValue.Set(static_cast<uint32_t>(de));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::GetFreeSpaceSize(nn::sf::Out<std::int64_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            int64_t size;
            NN_RESULT_DO(m_fileSystemInterface->GetFreeSpaceSize(&size, normalizedPath.GetPath()));
            outValue.Set(size);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::GetTotalSpaceSize(nn::sf::Out<std::int64_t> outValue, const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            int64_t size;
            NN_RESULT_DO(m_fileSystemInterface->GetTotalSpaceSize(&size, normalizedPath.GetPath()));
            outValue.Set(size);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::OpenFile(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFile>> outValue, const nn::fssrv::sf::Path& path, std::uint32_t mode) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s %x", m_fileSystemInterface.get(), path.str, mode);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            std::unique_lock<fssystem::SemaphoreAdaptor> openCountSemaphore;
            if (m_IsOpenCountLimited)
            {
                NN_SDK_ASSERT_NOT_NULL(m_pFileSystemProxyImpl);
                NN_RESULT_DO(m_pFileSystemProxyImpl->TryAcquireSaveDataEntryOpenCountSemaphore(&openCountSemaphore));
            }

            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            std::unique_ptr<nn::fs::fsa::IFile> file;
            NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_fileSystemInterface->OpenFile(&file, normalizedPath.GetPath(), static_cast<nn::fs::OpenMode>(mode)));
            std::unique_ptr<nn::fs::fsa::IFile> asyncFile(new fssystem::AsynchronousAccessFile(std::move(file), static_cast<nn::fs::OpenMode>(mode), fssystem::GetRegisteredThreadPool()));
            NN_RESULT_THROW_UNLESS(asyncFile != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemInterfaceAdapterA());
            nn::sf::SharedPointer<nn::fssrv::sf::IFile> pFile = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IFile, FileInterfaceAdapter>(std::move(asyncFile), this, std::move(openCountSemaphore));
            NN_RESULT_THROW_UNLESS(pFile != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemInterfaceAdapterA());
            outValue.Set(std::move(pFile));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }
    Result FileSystemInterfaceAdapter::OpenDirectory(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDirectory>> outValue, const nn::fssrv::sf::Path& path, std::uint32_t mode) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s %x", m_fileSystemInterface.get(), path.str, mode);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            std::unique_lock<fssystem::SemaphoreAdaptor> openCountSemaphore;
            if (m_IsOpenCountLimited)
            {
                NN_SDK_ASSERT_NOT_NULL(m_pFileSystemProxyImpl);
                NN_RESULT_DO(m_pFileSystemProxyImpl->TryAcquireSaveDataEntryOpenCountSemaphore(&openCountSemaphore));
            }

            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            std::unique_ptr<nn::fs::fsa::IDirectory> directory;
            NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(m_fileSystemInterface->OpenDirectory(&directory, normalizedPath.GetPath(), static_cast<nn::fs::OpenDirectoryMode>(mode)));
            nn::sf::SharedPointer<nn::fssrv::sf::IDirectory> pDirectory = FileSystemFactory::CreateSharedEmplaced<nn::fssrv::sf::IDirectory, DirectoryInterfaceAdapter>(std::move(directory), this, std::move(openCountSemaphore));
            NN_RESULT_THROW_UNLESS(pDirectory != nullptr, nn::fs::ResultAllocationMemoryFailedInFileSystemInterfaceAdapterA());
            outValue.Set(std::move(pDirectory));
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemInterfaceAdapter::Commit() NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p", m_fileSystemInterface.get());
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            return m_fileSystemInterface->Commit();
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemInterfaceAdapter::GetFileTimeStampRaw(nn::sf::Out<nn::fs::FileTimeStampRaw> outTimeStamp, const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        NN_FS_SCOPED_TRACE("%p %s", m_fileSystemInterface.get(), path.str);
        NN_FS_SCOPED_TRACE_CAPTURE_RESULT
        {
            auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
            PathNormalizer normalizedPath(path.str);
            NN_RESULT_THROW_UNLESS(normalizedPath.GetPath() != nullptr, normalizedPath.GetResult());

            nn::fs::FileTimeStampRaw ts;
            NN_RESULT_DO(m_fileSystemInterface->GetFileTimeStampRaw(&ts, normalizedPath.GetPath()));
            outTimeStamp.Set(ts);
            NN_RESULT_SUCCESS;
        }
        NN_FS_SCOPED_TRACE_END_CAPTURE_RESULT
    }

    Result FileSystemInterfaceAdapter::QueryEntry(const nn::sf::OutBuffer& outBuffer, const nn::sf::InBuffer& inBuffer, int32_t queryId, const nn::fssrv::sf::Path& path) NN_NOEXCEPT
    {
        auto scopedReaderLock = AcquireReaderLockForCacheInvalidation();
        NN_RESULT_DO(m_fileSystemInterface->QueryEntry(outBuffer.GetPointerUnsafe(), outBuffer.GetSize(), inBuffer.GetPointerUnsafe(), inBuffer.GetSize(), static_cast<nn::fs::fsa::QueryId>(queryId), path.str));
        NN_RESULT_SUCCESS;
    }

    Result FileSystemInterfaceAdapter::GetImpl(nn::sf::Out<nn::fs::detail::IFileSystemPointer> outValue) NN_NOEXCEPT
    {
        *outValue = m_fileSystemInterface;
        NN_RESULT_SUCCESS;
    }

    bool FileSystemInterfaceAdapter::IsDeepRetryEnabled() NN_NOEXCEPT
    {
        return m_DeepRetryEnabled;
    }

    util::optional<SharedLock<os::ReaderWriterLock>> FileSystemInterfaceAdapter::AcquireReaderLockForCacheInvalidation() NN_NOEXCEPT
    {
        util::optional<SharedLock<os::ReaderWriterLock>> lock;
        if (m_DeepRetryEnabled)
        {
            lock.emplace(m_InvalidateCacheLock);
        }

        return lock;
    }

    os::ReaderWriterLock& FileSystemInterfaceAdapter::GetReaderWriterLockForCacheInvalidation() NN_NOEXCEPT
    {
        return m_InvalidateCacheLock;
    }

    bool FileSystemInterfaceAdapter::IsAccessFailureDetectionObserved() NN_NOEXCEPT
    {
        return m_pFileSystemProxyImpl->IsAccessFailureDetectionObserved();
    }

}}}
