﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_DeviceDetectionEventManager.h"
#include "fssrv_DeviceDetectionEventNotifier.h"

namespace nn { namespace fssrv { namespace detail {

    DeviceDetectionEventManager::DeviceDetectionEventManager() NN_NOEXCEPT : m_Mutex(false)
    {
        m_Param.pList = &m_List;
        m_Param.pMutex = &m_Mutex;
    }

    DeviceDetectionEventManager::~DeviceDetectionEventManager() NN_NOEXCEPT
    {
        // TORIAEZU: テストの Exit で ~DeviceDetectionEventNotifier() より先行して呼ばれるケースを回避（通常ここを通るのは fs プロセス終了時）
        // NN_SDK_ASSERT(m_List.empty());
    }

    Result DeviceDetectionEventManager::CreateNotifier(std::unique_ptr<DeviceDetectionEventNotifier>* outValue) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        std::unique_ptr<DeviceDetectionEventNotifier> pEventNotifier(new DeviceDetectionEventNotifier(this));
        NN_RESULT_THROW_UNLESS(pEventNotifier, nn::fs::ResultAllocationMemoryFailedInDeviceDetectionEventManagerA());

        NN_RESULT_DO(nn::os::CreateSystemEvent(pEventNotifier->GetEvent(), nn::os::EventClearMode_ManualClear, true));
        m_List.push_back(*(pEventNotifier.get()));

        *outValue = std::move(pEventNotifier);

        NN_RESULT_SUCCESS;
    }

    void DeviceDetectionEventManager::SimulateEvent() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(*(m_Param.pMutex));

        for (auto& eventNotifier : *(m_Param.pList))
        {
            nn::os::SignalSystemEvent(eventNotifier.GetEvent());
        }
    }

    // この関数は直接実行しないでください。 DeviceDetectionEventNotifier のデストラクタでのみ呼ばれる想定です。
    void DeviceDetectionEventManager::DeleteNotifier(DeviceDetectionEventNotifier* pEventNotifier) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        m_List.erase(m_List.iterator_to(*pEventNotifier));
    }

    void DeviceDetectionEventManager::DeviceDetectionEventCallback(void* pParameter) NN_NOEXCEPT
    {
        Parameter* pParam = reinterpret_cast<Parameter*>(pParameter);

        std::lock_guard<nn::os::Mutex> scopedLock(*(pParam->pMutex));

        for (auto& eventNotifier : *(pParam->pList))
        {
            nn::os::SignalSystemEvent(eventNotifier.GetEvent());
        }
    }

}}}
