﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/time/time_PosixTime.h>

#include "fs_Allocator.h"
#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"

using namespace nn::fs::fsa;
using namespace nn::fs::detail;

namespace nn { namespace fs {

namespace detail {

Result ReadSaveDataFileSystemExtraData(SaveDataExtraData *outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->ReadSaveDataFileSystemExtraData(nn::sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(SaveDataExtraData)), saveDataId));
    NN_RESULT_SUCCESS;
}

Result ReadSaveDataFileSystemExtraData(SaveDataExtraData *outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->ReadSaveDataFileSystemExtraDataBySaveDataSpaceId(nn::sf::OutBuffer(reinterpret_cast<char*>(outValue), sizeof(SaveDataExtraData)), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result WriteSaveDataFileSystemExtraData(SaveDataSpaceId spaceId, SaveDataId saveDataId, const SaveDataExtraData& extraDataValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->WriteSaveDataFileSystemExtraData(saveDataId, static_cast<uint8_t>(spaceId), nn::sf::InBuffer(reinterpret_cast<const char*>(&extraDataValue), sizeof(SaveDataExtraData))));
    NN_RESULT_SUCCESS;
}

Result WriteSaveDataFileSystemExtraData(SaveDataSpaceId spaceId, SaveDataId saveDataId, const SaveDataExtraData& extraDataValue, const SaveDataExtraData& extraDataMask) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->WriteSaveDataFileSystemExtraDataWithMask(
        saveDataId,
        static_cast<uint8_t>(spaceId),
        nn::sf::InBuffer(reinterpret_cast<const char*>(&extraDataValue), sizeof(SaveDataExtraData)),
        nn::sf::InBuffer(reinterpret_cast<const char*>(&extraDataMask), sizeof(SaveDataExtraData))));
    NN_RESULT_SUCCESS;
}

}

namespace {

    class CacheStorageListCache : public detail::Newable
    {
        NN_DISALLOW_COPY(CacheStorageListCache);
        NN_DISALLOW_MOVE(CacheStorageListCache);

    public:
        class CacheEntry : public  detail::Newable
        {
        public:
            CacheEntry() NN_NOEXCEPT
                : m_CacheStorageIndex(0)
            {
            }

            explicit CacheEntry(int cacheStorageIndex) NN_NOEXCEPT
                : m_CacheStorageIndex(cacheStorageIndex)
            {
            }

        public:
            int GetCacheStorageIndex() const NN_NOEXCEPT
            {
                return m_CacheStorageIndex;
            }

        private:
            int m_CacheStorageIndex;
        };

    public:
        CacheStorageListCache() NN_NOEXCEPT
            : m_CacheReadOffset(0)
        {
        }

    public:
        Result PushBack(CacheEntry&& entry) NN_NOEXCEPT
        {
            m_CacheEntries.push_back(entry);
            NN_RESULT_THROW_UNLESS(!m_CacheEntries.get_allocator().IsAllocationFailed(), ResultAllocationMemoryFailedNew());
            NN_RESULT_SUCCESS;
        }

        const CacheEntry* PopFront() NN_NOEXCEPT
        {
            if( m_CacheEntries.size() <= m_CacheReadOffset )
            {
                return nullptr;
            }
            auto& entry = m_CacheEntries[m_CacheReadOffset];
            ++m_CacheReadOffset;
            return &entry;
        }

#if !defined(NN_SDK_BUILD_RELEASE)
        size_t GetCacheEntriesCount() NN_NOEXCEPT
        {
            return m_CacheEntries.size();
        }

        const CacheEntry& GetAt(size_t index) NN_NOEXCEPT
        {
            return m_CacheEntries[index];
        }
#endif // !defined(NN_SDK_BUILD_RELEASE)

    private:
        size_t m_CacheReadOffset;
        std::vector<CacheEntry, Allocator<CacheEntry>> m_CacheEntries;
    };

    CacheStorageListCache* GetCacheStorageListCache(CacheStorageListHandle handle) NN_NOEXCEPT
    {
        return reinterpret_cast<CacheStorageListCache*>(handle.handle);
    }
}

Result DeleteSaveData(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
{
    auto deletion = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_RESULT_DO(fileSystemProxy->DeleteSaveDataFileSystem(saveDataId));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(deletion(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        saveDataId));
    NN_RESULT_SUCCESS;
}

Result DeleteSaveData(nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
{
    auto deletion = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_RESULT_DO(fileSystemProxy->DeleteSaveDataFileSystemBySaveDataSpaceId(static_cast<uint8_t>(spaceId), saveDataId));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(deletion(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_DELETE_SAVEDATA(spaceId, saveDataId)));
    NN_RESULT_SUCCESS;
}

Result DeleteSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, const UserId& userId) NN_NOEXCEPT
{
    auto deletion = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(InvalidProgramId, SaveDataType::System, userId, systemSaveDataId);
        NN_RESULT_DO(fileSystemProxy->DeleteSaveDataFileSystemBySaveDataAttribute(static_cast<uint8_t>(saveDataSpaceId), attribute));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(deletion(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_DELETE_SYSTEMSAVEDATA(saveDataSpaceId, systemSaveDataId, userId)));
    NN_RESULT_SUCCESS;
}

Result RegisterSaveDataAtomicDeletion(SaveDataId* saveDataIdArray, int saveDataIdArrayCount) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

    NN_FS_RESULT_DO(fileSystemProxy->RegisterSaveDataFileSystemAtomicDeletion(nn::sf::InBuffer(reinterpret_cast<char*>(saveDataIdArray), sizeof(SaveDataId) * saveDataIdArrayCount)));
    NN_RESULT_SUCCESS;
}

Result SaveDataIterator::ReadSaveDataInfo(int64_t* outCount, SaveDataInfo* outInfoBuffer, int64_t infoBufferCount) NN_NOEXCEPT
{
    auto read = [=]() NN_NOEXCEPT -> Result
    {
        NN_FS_ABORT_UNLESS_WITH_RESULT(infoBufferCount >= 0, ResultInvalidArgument());
        NN_RESULT_DO(m_pReader->Read(nn::sf::Out<int64_t>(outCount), nn::sf::OutBuffer(reinterpret_cast<char*>(outInfoBuffer), sizeof(SaveDataInfo) * static_cast<size_t>(infoBufferCount))));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(read(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SIZE, infoBufferCount));
    NN_RESULT_SUCCESS;
}

SaveDataIterator::SaveDataIterator(nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader>&& pReader) NN_NOEXCEPT
    : m_pReader(pReader)
{
}

Result OpenSaveDataIterator(std::unique_ptr<SaveDataIterator>* outValue, SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
{
    auto open = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> reader;
        NN_RESULT_DO(fileSystemProxy->OpenSaveDataInfoReaderBySaveDataSpaceId(&reader, static_cast<uint8_t>(saveDataSpaceId)));

        std::unique_ptr<SaveDataIterator> iterator(new SaveDataIterator(std::move(reader)));
        NN_RESULT_THROW_UNLESS(iterator != nullptr, ResultAllocationMemoryFailedInSaveDataManagementA());

        *outValue = std::move(iterator);

        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(open(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId)));
    NN_RESULT_SUCCESS;
}

Result OpenSaveDataIterator(std::unique_ptr<SaveDataIterator>* outValue, SaveDataSpaceId saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT
{
    auto open = [=]() NN_NOEXCEPT->Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> reader;
        NN_RESULT_DO(fileSystemProxy->OpenSaveDataInfoReaderWithFilter(&reader, static_cast<uint8_t>(saveDataSpaceId), filter));

        std::unique_ptr<SaveDataIterator> iterator(new SaveDataIterator(std::move(reader)));
        NN_RESULT_THROW_UNLESS(iterator != nullptr, ResultAllocationMemoryFailedInSaveDataManagementA());

        *outValue = std::move(iterator);

        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(open(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId)));
    NN_RESULT_SUCCESS;
}

Result FindSaveDataWithFilter(SaveDataInfo* pOutValue, SaveDataSpaceId saveDataSpaceId, const SaveDataFilter& filter) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutValue != nullptr);

    auto find = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        int64_t count = 0;
        SaveDataInfo info;
        NN_RESULT_DO(fileSystemProxy->FindSaveDataWithFilter(nn::sf::Out<int64_t>(&count), nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(SaveDataInfo)), static_cast<uint8_t>(saveDataSpaceId), filter));
        if (count == 0)
        {
            return nn::fs::ResultTargetNotFound();
        }

        *pOutValue = info;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(find(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId)));
    NN_RESULT_SUCCESS;
}

Result CreateSaveData(nn::ncm::ApplicationId applicationId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Account, userId, InvalidSystemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = SaveDataSpaceId::User;
        creationInfo.isPseudoSaveFs = false;

        // TORIAEZU: 決め打ちでサムネイルデータ作成
        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::Thumbnail;
        metaInfo.size = detail::ThumbnailFileSize;

        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESAVEDATA(applicationId, userId, saveDataOwnerId, size, journalSize, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateSaveData(nn::ncm::ApplicationId applicationId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, const nn::fs::HashSalt& salt, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Account, userId, InvalidSystemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = SaveDataSpaceId::User;
        creationInfo.isPseudoSaveFs = false;

        // TORIAEZU: 決め打ちでサムネイルデータ作成
        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::Thumbnail;
        metaInfo.size = detail::ThumbnailFileSize;

        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystemWithHashSalt(attribute, creationInfo, metaInfo, salt));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESAVEDATA(applicationId, userId, saveDataOwnerId, size, journalSize, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateBcatSaveData(nn::ncm::ApplicationId applicationId, int64_t size) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        const uint32_t BcatSaveDataFlags = 0;               // TODO: bcat セーブデータに応じた flags を指定

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Bcat, InvalidUserId, InvalidSystemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = BcatSaveDataJournalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = BcatOwnerId;
        creationInfo.flags = BcatSaveDataFlags;
        creationInfo.spaceId = SaveDataSpaceId::User;
        creationInfo.isPseudoSaveFs = false;

        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::None;
        metaInfo.size = 0;
        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATEBCATSAVEDATA(applicationId, size)));
    NN_RESULT_SUCCESS;
}

Result CreateDeviceSaveData(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Device, InvalidUserId, InvalidSystemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = SaveDataSpaceId::User;
        creationInfo.isPseudoSaveFs = false;

        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::None;
        metaInfo.size = 0;
        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATEDEVICESAVEDATA(applicationId, saveDataOwnerId, size, journalSize, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateTemporaryStorage(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Temporary, InvalidUserId, InvalidSystemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = 0;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = SaveDataSpaceId::Temporary;
        creationInfo.isPseudoSaveFs = false;

        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::None;
        metaInfo.size = 0;

        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATETEMPORARYSTORAGE(applicationId, saveDataOwnerId, size, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, SaveDataSpaceId spaceId, uint64_t saveDataOwnerId, uint16_t index, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(applicationId, SaveDataType::Cache, InvalidUserId, InvalidSystemSaveDataId, index);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = spaceId;
        creationInfo.isPseudoSaveFs = false;

        SaveDataMetaInfo metaInfo;
        metaInfo.type = SaveDataMetaType::None;
        metaInfo.size = 0;

        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystem(attribute, creationInfo, metaInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SYSTEM_CREATECACHESTORAGE(applicationId, spaceId, saveDataOwnerId, size, journalSize, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, SaveDataSpaceId spaceId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateCacheStorage(applicationId, spaceId, saveDataOwnerId, 0, size, journalSize, flags);
}

Result CreateCacheStorage(nn::ncm::ApplicationId applicationId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateCacheStorage(applicationId, SaveDataSpaceId::User, saveDataOwnerId, size, journalSize, flags);
}

Result CreateSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(InvalidProgramId, SaveDataType::System, userId, systemSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = saveDataOwnerId;
        creationInfo.flags = flags;
        creationInfo.spaceId = saveDataSpaceId;
        creationInfo.isPseudoSaveFs = false;

        NN_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystemBySystemSaveDataId(attribute, creationInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESYSTEMSAVEDATA(saveDataSpaceId, systemSaveDataId, userId, saveDataOwnerId, size, journalSize, flags)));
    NN_RESULT_SUCCESS;
}

Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateSystemSaveData(SaveDataSpaceId::System, systemSaveDataId, userId, saveDataOwnerId, size, journalSize, flags);
}

Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, UserId userId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateSystemSaveData(systemSaveDataId, userId,        0,               size, journalSize, flags);
}

Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateSystemSaveData(systemSaveDataId, InvalidUserId, saveDataOwnerId, size, journalSize, flags);
}

Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateSystemSaveData(systemSaveDataId, InvalidUserId, 0,               size, journalSize, flags);
}

Result CreateSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
{
    return CreateSystemSaveData(saveDataSpaceId, systemSaveDataId, InvalidUserId, saveDataOwnerId, size, journalSize, flags);
}

Result CreateSystemBcatSaveData(SystemBcatSaveDataId systemBcatSaveDataId, int64_t size, uint32_t flags) NN_NOEXCEPT
{
    const int64_t journalSize = 2LL * 1024 * 1024;

    auto creation = [=]() NN_NOEXCEPT -> Result
    {
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        auto attribute = SaveDataAttribute::Make(InvalidProgramId, SaveDataType::SystemBcat, InvalidUserId, systemBcatSaveDataId);
        SaveDataCreationInfo creationInfo;
        creationInfo.size = size;
        creationInfo.journalSize = journalSize;
        creationInfo.blockSize = DefaultSaveDataBlockSize;
        creationInfo.ownerId = 0;
        creationInfo.flags = flags;
        creationInfo.spaceId = SaveDataSpaceId::System;
        creationInfo.isPseudoSaveFs = false;

        NN_FS_RESULT_DO(fileSystemProxy->CreateSaveDataFileSystemBySystemSaveDataId(attribute, creationInfo));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(creation(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CREATESYSTEMBCATSAVEDATA(systemBcatSaveDataId, size, flags)));
    NN_RESULT_SUCCESS;
}

Result ExtendSaveData(
    nn::fs::SaveDataSpaceId spaceId,
    nn::fs::SaveDataId saveDataId,
    int64_t saveDataAvailableSize,
    int64_t saveDataJournalSize) NN_NOEXCEPT
{
    auto extend = [=]() NN_NOEXCEPT -> Result
    {
        const auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        NN_RESULT_DO(fileSystemProxy->ExtendSaveDataFileSystem(
            static_cast<uint8_t>(spaceId),
            saveDataId,
            saveDataAvailableSize,
            saveDataJournalSize));
        NN_RESULT_SUCCESS;
    };

    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(extend(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE,
        ::nn::fs::detail::IdString().ToString(spaceId), saveDataId, saveDataAvailableSize, saveDataJournalSize));
    NN_RESULT_SUCCESS;
}

Result QuerySaveDataTotalSize(int64_t* outValue, int64_t saveDataSize, int64_t saveDataJournalSize)
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        int64_t totalSize;
        NN_RESULT_DO(fileSystemProxy->QuerySaveDataTotalSize(nn::sf::Out<int64_t>(&totalSize), saveDataSize, saveDataJournalSize));

        *outValue = totalSize;
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAJOURNALSIZE, saveDataSize, saveDataJournalSize));
    NN_RESULT_SUCCESS;
}


Result GetSaveDataOwnerId(Bit64* outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(ReadSaveDataFileSystemExtraData(&extraData, saveDataId));
        *outValue = extraData.ownerId;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASIZE, saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataOwnerId(Bit64* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        *outValue = extraData.ownerId;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataFlags(uint32_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(ReadSaveDataFileSystemExtraData(&extraData, saveDataId));
        *outValue = extraData.flags;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID, saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataFlags(uint32_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        *outValue = extraData.flags;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result SetSaveDataFlags(SaveDataId saveDataId, SaveDataSpaceId saveDataSpaceId, uint32_t flags) NN_NOEXCEPT
{
    auto setter = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        extraData.flags = flags;
        NN_RESULT_DO(WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(setter(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAFLAGS,
        saveDataId, ::nn::fs::detail::IdString().ToString(saveDataSpaceId), flags));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataTimeStamp(nn::time::PosixTime* outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(ReadSaveDataFileSystemExtraData(&extraData, saveDataId));
        outValue->value = extraData.timeStamp;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID, saveDataId));
    NN_RESULT_SUCCESS;
}

Result SetSaveDataTimeStamp(SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId, const nn::time::PosixTime& saveDataTimeStamp) NN_NOEXCEPT
{
    auto setter = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraDataMask;
        {
            memset(&extraDataMask, 0, sizeof(extraDataMask));
            memset(&extraDataMask.timeStamp, 0xFF, sizeof(extraDataMask.timeStamp));
        }
        SaveDataExtraData extraData = {};
        extraData.timeStamp = saveDataTimeStamp.value;
        NN_RESULT_DO(WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData, extraDataMask));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(setter(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATATIMESTAMP,
        saveDataId, ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataTimeStamp.value));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataTimeStamp(nn::time::PosixTime* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        outValue->value = extraData.timeStamp;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataAvailableSize(int64_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(ReadSaveDataFileSystemExtraData(&extraData, saveDataId));
        *outValue = extraData.availableSize;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID, saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataAvailableSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        *outValue = extraData.availableSize;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataJournalSize(int64_t* outValue, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(ReadSaveDataFileSystemExtraData(&extraData, saveDataId));
        *outValue = extraData.journalSize;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID, saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataJournalSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraData;
        NN_RESULT_DO(detail::ReadSaveDataFileSystemExtraData(&extraData, saveDataSpaceId, saveDataId));
        *outValue = extraData.journalSize;
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
        ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result GetSaveDataCommitId(SaveDataCommitId* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        int64_t hash;
        NN_RESULT_DO(fileSystemProxy->GetSaveDataCommitId(nn::sf::Out<int64_t>(&hash), static_cast<uint8_t>(saveDataSpaceId), saveDataId));
        *outValue = hash;
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_SYSTEM(
            query(),
            nullptr,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
            ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result SetSaveDataCommitId(SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId, const SaveDataCommitId& saveDataCommitId) NN_NOEXCEPT
{
    auto setter = [=]() NN_NOEXCEPT -> Result
    {
        SaveDataExtraData extraDataMask;
        {
            memset(&extraDataMask, 0, sizeof(extraDataMask));
            memset(&extraDataMask.commitId, 0xFF, sizeof(extraDataMask.commitId));
        }
        SaveDataExtraData extraData = {};
        extraData.commitId = saveDataCommitId;
        NN_RESULT_DO(WriteSaveDataFileSystemExtraData(saveDataSpaceId, saveDataId, extraData, extraDataMask));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(setter(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATACOMMITID,
        saveDataId, ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataCommitId));
    NN_RESULT_SUCCESS;
}

Result QuerySaveDataInternalStorageTotalSize(int64_t* outValue, SaveDataSpaceId saveDataSpaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();
        int64_t totalSize;
        NN_RESULT_DO(fileSystemProxy->QuerySaveDataInternalStorageTotalSize(nn::sf::Out<int64_t>(&totalSize), static_cast<uint8_t>(saveDataSpaceId), saveDataId));

        *outValue = totalSize;
        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG_SYSTEM(
            query(),
            nullptr,
            NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATASPACEID NN_DETAIL_FS_ACCESS_LOG_FORMAT_SAVEDATAID,
            ::nn::fs::detail::IdString().ToString(saveDataSpaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

Result VerifySaveData(bool* pOutIsValid, SaveDataId saveDataId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    return VerifySaveData(pOutIsValid, SaveDataSpaceId::System, saveDataId, workBuffer, workBufferSize);
}

Result VerifySaveData(bool* pOutIsValid, SaveDataSpaceId spaceId, SaveDataId saveDataId, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();

    auto result = fileSystemProxy->VerifySaveDataFileSystemBySaveDataSpaceId(static_cast<uint8_t>(spaceId), saveDataId, nn::sf::OutBuffer(static_cast<char*>(workBuffer), workBufferSize));
    if( ResultDataCorrupted::Includes(result) )
    {
        *pOutIsValid = false;
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_FS_RESULT_DO(result);
    }

    *pOutIsValid = true;
    NN_RESULT_SUCCESS;
}

Result CorruptSaveDataForDebug(SaveDataId saveDataId) NN_NOEXCEPT
{
    return CorruptSaveDataForDebug(SaveDataSpaceId::System,saveDataId);
}

Result CorruptSaveDataForDebug(SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT
{
    auto fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->CorruptSaveDataFileSystemBySaveDataSpaceId(static_cast<uint8_t>(spaceId), saveDataId));
    NN_RESULT_SUCCESS;
}

void DisableAutoSaveDataCreation() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(fileSystemProxy->DisableAutoSaveDataCreation());
}

Result DeleteCacheStorage(int index) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultInvalidArgument());
    uint16_t internalIndex = static_cast<uint16_t>(index);
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG(
            fileSystemProxy->DeleteCacheStorage(internalIndex)
            , nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX, index
        )
    );
    NN_RESULT_SUCCESS;
}

Result GetCacheStorageSize(int64_t* outCacheStorageSize, int64_t* outCacheStorageJournalSize, int index) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultInvalidArgument());
    uint16_t internalIndex = static_cast<uint16_t>(index);
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG(
            fileSystemProxy->GetCacheStorageSize(nn::sf::Out<int64_t>(outCacheStorageSize), nn::sf::Out<int64_t>(outCacheStorageJournalSize), internalIndex)
            , nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_INDEX, index
        )
    );
    NN_RESULT_SUCCESS;
}

Result OpenCacheStorageList(CacheStorageListHandle* pOutValue) NN_NOEXCEPT
{
    // NOTE:
    //   Open 時に全ての CacheStorage を列挙して CacheStorageListCache にキャッシュする
    //   Read は単にキャッシュから CacheStorageInfo に変換する
    //   このため、CacheStorageInfo にメンバーを足したときは
    //   キャッシュ処理と変換処理を変更する必要がある

    std::unique_ptr<CacheStorageListCache> pListCache;

    auto openImpl = [=, &pListCache]() NN_NOEXCEPT -> Result
    {
        if( !pListCache )
        {
            pListCache.reset(new CacheStorageListCache);
            NN_RESULT_THROW_UNLESS(pListCache, ResultAllocationMemoryFailedNew());
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();

        {
            nn::sf::SharedPointer<nn::fssrv::sf::ISaveDataInfoReader> reader;
            NN_RESULT_DO(fileSystemProxy->OpenSaveDataInfoReaderOnlyCacheStorage(&reader));

            for( ;; )
            {
                SaveDataInfo info;
                int64_t outCount = 0;
                NN_RESULT_DO(
                    reader->Read(
                        nn::sf::Out<int64_t>(&outCount),
                        nn::sf::OutBuffer(reinterpret_cast<char*>(&info), sizeof(SaveDataInfo))
                    )
                );
                if( outCount == 0 )
                {
                    break;
                }

                CacheStorageListCache::CacheEntry cacheEntry(info.index);
                NN_RESULT_DO(pListCache->PushBack(std::move(cacheEntry)));
            }
        }

        NN_RESULT_SUCCESS;
    };

    void* pHandle = nullptr;
    auto open = [&openImpl, &pHandle, &pListCache]() NN_NOEXCEPT -> Result
    {
        // EnsureSaveData と CacheStorageList へのアクセスが競合するため、
        // ResultInvalidHandle に対しては再試行する
        NN_RESULT_DO(DoContinouslyUntilSaveDataListFetched(openImpl));
        NN_SDK_REQUIRES(pListCache);
        pHandle = pListCache.release();
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG(
            open(),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CACHESTORAGELISTHANDLE, pHandle
        )
    );

    pOutValue->handle = pHandle;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)
Result ReadCacheStorageList(int* outValue, CacheStorageInfo *infoBuffer, CacheStorageListHandle handle, int infoBufferCount) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(infoBufferCount >= 0, nn::fs::ResultInvalidArgument());

    // NOTE:
    //   OpenCacheStorageList の先頭にある NOTE を確認してください

    int cacheStorageNum = 0;
    auto readCacheStorageList = [=, &cacheStorageNum]() NN_NOEXCEPT -> Result
    {
        auto pListCache = GetCacheStorageListCache(handle);

        for( ; cacheStorageNum < infoBufferCount; ++cacheStorageNum )
        {
            const auto* pEntry = pListCache->PopFront();
            if( pEntry == nullptr )
            {
                break;
            }

            std::memset(&infoBuffer[cacheStorageNum], 0, sizeof(infoBuffer[cacheStorageNum]));
            infoBuffer[cacheStorageNum].index = pEntry->GetCacheStorageIndex();
        }

        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(
        NN_DETAIL_FS_ACCESS_LOG(
            readCacheStorageList()
            , nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_READCACHESTORAGELIST(handle, infoBufferCount)
        )
    );
    *outValue = cacheStorageNum;
    NN_RESULT_SUCCESS;
}

void CloseCacheStorageList(CacheStorageListHandle handle) NN_NOEXCEPT
{
    delete GetCacheStorageListCache(handle);
    NN_DETAIL_FS_ACCESS_LOG(nn::ResultSuccess(), nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_CACHESTORAGELISTHANDLE, handle);
}

Result UpdateSaveDataMacForDebug(uint8_t saveDataSpaceId, uint64_t saveDataId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->UpdateSaveDataMacForDebug(saveDataSpaceId, saveDataId));
    NN_RESULT_SUCCESS;
}

}}

