﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "fs_FileSystemProxyServiceObject.h"

namespace nn { namespace fs {

Result GetRightsId(RightsId* pOutValue, nn::ncm::ProgramId programId, nn::ncm::StorageId storageId) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->GetRightsId(nn::sf::Out<RightsId>(pOutValue), programId, storageId));

    NN_RESULT_SUCCESS;
}

Result GetRightsId(RightsId* pOutValue, const char* path) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());
    nn::fssrv::sf::FspPath sfPath;
    NN_FS_RESULT_DO(FspPathPrintf(&sfPath, "%s", path));

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->GetRightsIdByPath(nn::sf::Out<RightsId>(pOutValue), sfPath));

    NN_RESULT_SUCCESS;
}

Result GetRightsId(RightsId* pOutRightsId, uint8_t* pOutKeyGeneration, const char* path) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutRightsId != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(pOutKeyGeneration != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());
    nn::fssrv::sf::FspPath sfPath;
    NN_FS_RESULT_DO(FspPathPrintf(&sfPath, "%s", path));

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->GetRightsIdAndKeyGenerationByPath(nn::sf::Out<RightsId>(pOutRightsId), nn::sf::Out<uint8_t>(pOutKeyGeneration), sfPath));

    NN_RESULT_SUCCESS;
}


Result RegisterExternalKey(const RightsId& rightsId, const nn::spl::AccessKey& accessKey) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->RegisterExternalKey(rightsId, accessKey));

    NN_RESULT_SUCCESS;
}

Result UnregisterAllExternalKey() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    NN_FS_RESULT_DO(fileSystemProxy->UnregisterAllExternalKey());

    NN_RESULT_SUCCESS;
}

}}
