﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/fssrv/sf/fssrv_IDeviceOperator.h>

#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>

#include "fs_FileSystemProxyServiceObject.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

Result GetMmcSpeedMode(MmcSpeedMode* pOutValue) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    int64_t speedMode = 0;
    NN_FS_RESULT_DO(deviceOperator->GetMmcSpeedMode(nn::sf::Out<int64_t>(&speedMode)));
    *pOutValue = static_cast<MmcSpeedMode>(speedMode);
    NN_RESULT_SUCCESS;
}

Result GetMmcCid(void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outBuffer != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetMmcCid(nn::sf::OutBuffer(reinterpret_cast<char*>(outBuffer), size), static_cast<int64_t>(size)));
    NN_RESULT_SUCCESS;
}

Result EraseMmc(MmcPartition id) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->EraseMmc(static_cast<int64_t>(id)));
    NN_RESULT_SUCCESS;
}

Result GetMmcPartitionSize(int64_t* outValue, MmcPartition id) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetMmcPartitionSize(nn::sf::Out<int64_t>(outValue), static_cast<int64_t>(id)));
    NN_RESULT_SUCCESS;
}

Result GetMmcPatrolCount(uint32_t* outValue) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outValue != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetMmcPatrolCount(nn::sf::Out<uint32_t>(outValue)));
    NN_RESULT_SUCCESS;
}

Result GetAndClearMmcErrorInfo(StorageErrorInfo* pOutStorageErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(pOutStorageErrorInfo != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(pOutLogSize != nullptr, ResultNullptrArgument());
    NN_FS_RESULT_THROW_UNLESS(pOutLogBuffer != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    int64_t logSize;
    NN_FS_RESULT_DO(deviceOperator->GetAndClearMmcErrorInfo(nn::sf::Out<StorageErrorInfo>(pOutStorageErrorInfo),
        nn::sf::Out<int64_t>(&logSize), nn::sf::OutBuffer(pOutLogBuffer, logBufferSize), static_cast<int64_t>(logBufferSize)));
    *pOutLogSize = static_cast<size_t>(logSize);
    NN_RESULT_SUCCESS;
}

Result GetMmcExtendedCsd(void* outBuffer, size_t size) NN_NOEXCEPT
{
    NN_FS_RESULT_THROW_UNLESS(outBuffer != nullptr, ResultNullptrArgument());

    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->GetMmcExtendedCsd(nn::sf::OutBuffer(reinterpret_cast<char*>(outBuffer), size), static_cast<int64_t>(size)));
    NN_RESULT_SUCCESS;
}

Result SuspendMmcPatrol() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->SuspendMmcPatrol());
    NN_RESULT_SUCCESS;
}

Result ResumeMmcPatrol() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator> deviceOperator;
    NN_FS_RESULT_DO(fileSystemProxy->OpenDeviceOperator(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IDeviceOperator>>(&deviceOperator)));
    NN_FS_RESULT_DO(deviceOperator->ResumeMmcPatrol());
    NN_RESULT_SUCCESS;
}

}}
